function [d_rn_u,rssi_u,traj_u,nts_u]=average_rssi_measurements_rm(node_s,robot_s,d_cum_max,varargin)
% Averages the rssi measurements over robot movement. The rssi is averaged over distance traversed
% by the robot. In short, the outputs are calculated as follows:
%	- Divide the trajectory of the robot in segments of length d_cum_max
%	- Calculate the mean RSS for each of those intervals
%
% SYNTAX: [d_rn_u,rssi_u,traj_u,nts_u]=average_rssi_measurements_rm(node_s,robot_s,d_cum_max);
%
% INPUTS:
% - node_s: structure with the target node
% - robot_s: structure with the position of the robot
% - d_cum_max: the distance that the robot has to traverse before the average is calculated.
%
% RETURNS
%
% - d_rn_u: distance robot-node for each measurement
% - rssi_u: RSSI averaged
% - traj_u: trajectory of the robot
% - nts_u: timestamp for the (averaged) measurements
%
% Example
% 	[d_rn_u,rssi_u,traj_u,nts_u]=average_rssi_measurements_rm(node_s,robot_s,0.25);
%

if iscell(varargin) && numel(varargin)==1
  opts=validate_input_args(node_s,robot_s,d_cum_max,varargin{1}{:});
else
  opts=validate_input_args(node_s,robot_s,d_cum_max,varargin{:});
endif

%--------------------------%


%Robot node id
nr_id=robot_s.nr_id;

%Get the measurements corresponding to the node from the robot
m_i=find(node_s.tx_id==nr_id);
nts=node_s.nts(m_i);
rssi_r=node_s.rssi(m_i);

if strcmp(opts.AvMode,'P')
  rssi_r=10.^(rssi_r/10);
endif

% Calculate the trajectory of the robot (its position at each measurement)
[traj]=robot_pos_at_time(nts,robot_s);
trajx=traj(:,1);
trajy=traj(:,2);
traj=[trajx trajy];

%Distance robot-node
d_rn=sqrt( sum((repmat(node_s.pos,length(traj),1)-traj).^2,2) );

if d_cum_max==0
  d_rn_u=d_rn;
  rssi_u=rssi_r;
  traj_u=traj;
  nts_u=nts;
  return;
endif

cp_i=divide_robot_traj(traj,d_cum_max);

for ind=1:length(cp_i)-1
  idx=[cp_i(ind):(cp_i(ind+1)-1)];
  %idx=[cp_i(ind):(cp_i(ind+1))];
  rssi_u(ind,1)=mean(rssi_r(idx)); %rssi used
  nts_u(ind,1)=mean(nts(idx)); %nts used
  trajx_u(ind,1)=mean(trajx(idx));
  trajy_u(ind,1)=mean(trajy(idx));
  %d_rn_u(ind,1)=mean(d_rn((cp_i(ind):cp_i(ind+1))));
  d_rn_u(ind,1)=sqrt( (trajx_u(ind,1)-node_s.pos(1))^2 + (trajy_u(ind,1)-node_s.pos(2))^2 );
endfor

traj_u=[trajx_u trajy_u];
if strcmp(opts.AvMode,'P') %back to dB
  rssi_u=10*log10(rssi_u);
endif


function opts=validate_input_args(node_s,robot_s,d_cum_max,varargin);

  p = inputParser;
  p.FunctionName = "average_rssi_measurements_rm";
  %p.KeepUnmatched = true;

  %REQUIRED parameters in order (must be before optionals)
  validate_node_s = @(x)isstruct(x);
  p.addRequired("node_s", validate_node_s);
  validate_robot_s = @(x)isstruct(x);
  p.addRequired ("robot_s", validate_robot_s);
  p.addRequired ("d_cum_max", @(x)isscalar(x));

  %%% OPTIONAL parameters

  %%% PARAM VALUE pairs
  val_average_distance = @(x) isscalar(x);
  p.addParamValue("AvMode",'dB', @(x)ischar(x)); %Default averaging mode is in dB

  %Finally parse
  p.parse (node_s,robot_s,d_cum_max,varargin{:});
  opts = p.Results;


