%%
% code to reproduce figures from Taylor et al (2019) "Early deviation from 
% normal structural connectivity: a novel intrinsic severity score for mild TBI"
% 
clear all
close all
clc


load Data.mat
% Data.mat contains the following data for 65 scans 
%   - age for each subject
%   - funcPerf functional performance from neurocognitive testing
%   - grpCtl locations of the controls
%   - grpAcuteMild locations of the patients
%   - subjID subject ID (not needed)
%   - tractFAmean mean FA of each tract for each subject
%   - trkName name of each tract in the same order as tractFAmean


%% regress age 
tractFAmeanR=zeros(size(tractFAmean)); % preallocate space for residuals
for ii=1:22 % for each tract...
    mdl = fitlm(age(grpCtl),tractFAmean(grpCtl,ii),'linear','RobustOpts','on');
    tractFAmeanR(:,ii)=tractFAmean(:,ii)-((age*mdl.Coefficients.Estimate(2))+mdl.Coefficients.Estimate(1));    
end


%% compute multivariate mahalanobis distance
nboot=10000;
M=zeros(65,nboot);% preacllocate space for maha dist
nbControls=numel(grpCtl);
nbInBoot=25;
rng(0,'twister'); % for reproducibility
tic
for ii=1:65 %can parfor this if you have a multicore computer to speed up
    ii
    if ismember(ii,grpCtl) %if current subj is control
        tmp=grpCtl;tmp(tmp==ii)=[];% remove current subject from group
        for j=1:nboot
            tmpboot=tmp(randperm(nbControls-1,nbInBoot));% -1 to avoid current control
            M(ii,j)=mahalkernel(tractFAmeanR(ii,:),tractFAmeanR(tmpboot,:));
        end
    else % current subject is not a control
        for j=1:nboot
            tmpboot=grpCtl(randperm(nbControls,nbInBoot));
            M(ii,j)=mahalkernel(tractFAmeanR(ii,:),tractFAmeanR(tmpboot,:));
        end
    end
    
end
toc
clear tmp
M_med=median(M,2);

%% compute multivariate M distance AUC and correlations with funcPerf
[X,Y,~,AUC_m]=perfcurve([zeros(size(grpCtl));ones(size(grpAcuteMild))],M_med([grpCtl;grpAcuteMild]),1);%AUC for figure 2e,f
[rho_m,p_m]=corr(M_med(grpAcuteMild),funcPerf(grpAcuteMild),'type','spearman','tail','right');%rho, p for figure 3b

%% compute univariate |z| distance
rois=[1:22];
nboot=1000;
z_roi=zeros(65,nboot,22);

nbInBoot=25;
rng(0,'twister')
tic
for ii=1:65%for each subject
    ii
    for r=1:22%for each tract
        if ismember(ii,grpCtl) %if current subj is control
            tmp=grpCtl;tmp(tmp==ii)=[];
            for j=1:nboot
                tmpboot=tmp(randperm(30,nbInBoot));
                z_roi(ii,j,r)=mahalkernel(tractFAmeanR(ii,r),tractFAmeanR(tmpboot,r));
            end
        else % current subject is not a control
            for j=1:nboot
                tmpboot=grpCtl(randperm(31,nbInBoot));
                z_roi(ii,j,r)=mahalkernel(tractFAmeanR(ii,r),tractFAmeanR(tmpboot,r));
            end
        end
    end
end
toc
clear tmp
z_med=squeeze(median(z_roi,2));


%% compute univariate |z| distance AUC and correlations with funcPerf
AUC_z=zeros(22,1);
rho_z=zeros(22,1);
p_z=zeros(22,1);
for ii=1:22
    [~,~,~,AUC_z(ii)]=perfcurve([zeros(size(grpCtl));ones(size(grpAcuteMild))],z_med([grpCtl;grpAcuteMild],ii),1);
    [rho_z(ii),p_z(ii)]=corr(z_med(grpAcuteMild,ii),funcPerf(grpAcuteMild),'type','spearman','tail','right');
end


%% figure 2d
figure
barh(AUC_z([1:2:22 2:2:22]));
hold on
plot([AUC_m AUC_m],[0 23],'k','linewidth',3)
set(gca,'ytick',1:22,'yticklabel',trkName([1:2:22 2:2:22]),'fontsize',12)
xlabel('AUC')
xlim([0.4 0.85]);
grid on
ylim([0 23])

%% figure 3a
figure
barh(rho_z([1:2:22 2:2:22]));
hold on
plot([rho_m rho_m],[0 23],'k','linewidth',3)
set(gca,'ytick',1:22,'yticklabel',trkName([1:2:22 2:2:22]),'fontsize',12)
xlabel('Spearman''s \rho')
xlim([-0.5 .5]);
grid on
ylim([0 23])

%% Fig.3b : scatter M against funcPerf

figure
scatter(M_med(grpAcuteMild),funcPerf(grpAcuteMild),ones(size(grpAcuteMild))*100,'filled');hold on
scatter(M_med(grpCtl),funcPerf(grpCtl),ones(size(grpCtl))*100,'filled','markerfacealpha',0.3);hold off
[brob,stats] = robustfit(M_med(grpAcuteMild),funcPerf(grpAcuteMild));
hold on;plot(M_med(grpAcuteMild),brob(1)+brob(2)*M_med(grpAcuteMild),'LineWidth',3,'Color',[0 0.45 0.74])
xlabel('M')
ylabel('funcPerf')
grid on
title(strcat(' p=',num2str(p_m),' \rho=',num2str(rho_m)));

