/***************************************************************************/
/**                                                                       **/
/**                    l  a  k  e  t  e  m  p  .  c                       **/
/**                                                                       **/
/**     lake temperature scheme, modeled after soiltemp.c                 **/
/**                                                                       **/
/**     soiltemp() is invoked only for the stands of each cell,           **/
/**     i.e. excluding the lake areas. This function fills that gap.      **/
/**     Parameterization taken from Piccolroaz et al.,                    **/
/**     Hydrol. Earth Syst. Sci., 17, 3323-3338, 2013                     **/
/**     in a simplified form:                                             **/
/**                                                                       **/
/**     dT/dt=rho_4/delta(T) (T_air-T),                                   **/
/**                                                                       **/
/**     where delta(T) is the thickness of the surface well-mixed layer   **/
/**     A simple Euler scheme is used with adaptive time steps            **/
/**                                                                       **/
/**     written by Werner von Bloh, Stefan Petri                          **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: $Date:: 2016-04-22 11:39:04 +0200 (Fri, 22 Apr 201#$ **/
/**     By         : $Author:: bloh                            $          **/
/**                                                                       **/
/***************************************************************************/

#ifdef COUPLING_WITH_FMS

#include "lpj.h"

#define T_I 4.0  /* (deg C) */
#define rho_4 6.18e-3
#define rho_6 3.08
#define rho_7 14.41
#define rho_8 0.31

/* Function for calculating the dimensionless thickness delta of the 
 * surface well-mixed layer
 */

static Real delta(Real laketemp)
{
  Real res;
  if(laketemp>=T_I)
  {
    res=exp((T_I-laketemp)/rho_6);
    if(res<1e-5)
      res=1e-5;
  }
  else
  {
    res=exp((laketemp-T_I)/rho_7)+exp(-laketemp/rho_8);
    if(res>1)
      res=1;
  }
  return res;
} /* of 'delta' */

void laketemp(Cell *cell,const Dailyclimate *climate)
{
  int i,k;
  Real deltat,t,t_old;
  /* calculate new lake temp for time step 1 day */
  deltat=rho_4/delta(cell->laketemp)*(climate->temp-cell->laketemp);
  t=cell->laketemp+deltat;
  k=1;
  do
  {
    t_old=t; /* save lake temperature from previous time steps */
    k*=2; /* double the time steps */
    t=cell->laketemp;
    for(i=0;i<k;i++)
    {
      deltat=rho_4/delta(t)*(climate->temp-t)/k;
      t+=deltat;
    }
  }
  while(fabs(t-t_old)>1e-2); /* repeat until error<0.01 */
  cell->laketemp=t;
} /* of 'laketemp' */

#endif /* COUPLING_WITH_FMS */
