/***************************************************************************/
/**                                                                       **/
/**               f  s  c  a  n  s  o  i  l  p  a  r  .  c                **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran/C++ version     **/
/**                                                                       **/
/**     written by Werner von Bloh, Sibyll Schaphoff                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: $Date:: 2017-06-12 19:12:00 +0200 (Mon, 12 Jun 201#$ **/
/**     By         : $Author:: bloh                            $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"

#define UNDEF (-1)

#define fscanreal2(isout,file,var,name)\
  if(fscanreal(file,var,name,isout))\
  {\
     return 0;\
  }
#define fscanint2(isout,file,var,name) \
  if(fscanint(file,var,name,isout))\
  {\
     return 0;\
  }

Real soildepth[NSOILLAYER];
Real layerbound[NSOILLAYER];
Real midlayer[NSOILLAYER];
Real logmidlayer[NSOILLAYER];   /*log10(midlayer[l]/midlayer[NSOILLAYER-2]), for vertical soc profile*/
Real fbd_fac[NFUELCLASS];

int fscansoilpar(FILE *file,         /* file  pointer */
                 Soilpar **soilpar,  /* Pointer to Soilpar array */
                 Bool isout         /* generate error output */
                )                    /* returns number of elements in array */
{
  int nsoil,n,id,l;
  String s;
  Soilpar *soil;
  for(l=0;l<NSOILLAYER;l++)
    if(fscanreal(file,soildepth+l,"soildepth",isout))
    {
      return 0;
    }
  for(l=0;l<NSOILLAYER;l++)
  {
    if(fscanreal(file,layerbound+l,"layerbound",isout))
    {
      return 0;
    }
    midlayer[l]=l>0?(layerbound[l-1]+soildepth[l]/2.):soildepth[l]/2.;
  }
  foreachsoillayer(l){
    logmidlayer[l]=log10(midlayer[l]/midlayer[NSOILLAYER-2]);
  }
  for(l=0;l<NFUELCLASS;l++)
    if(fscanreal(file,fbd_fac+l,"fbd_fac",isout))
    {
      return 0;
    }
  if(fscanint(file,&nsoil,"nsoil",isout))
  {
    return 0;
  }
  if(nsoil<1 || nsoil>USHRT_MAX)
  {
    if(isout)
      fprintf(stderr,"ERROR170: Invalid value for number of soil types=%d\n",
              nsoil);
    return 0;
  }
  *soilpar=newvec(Soilpar,nsoil);
  check(*soilpar);
  for(n=0;n<nsoil;n++)
    (*soilpar)[n].type=UNDEF;
  for(n=0;n<nsoil;n++)
  {
    if(fscanint(file,&id,"soiltype",isout))
    {
      return 0;
    }
    if(id<0 || id>=nsoil)
    {
      if(isout)
        fprintf(stderr,"ERROR115: Invalid range of soil type=%d in fscansoilpar(), valid range is [0,%d].\n",id,nsoil-1);
      break;
    }
    soil=(*soilpar)+id;
    if(soil->type!=UNDEF)
    {
      if(isout)
        fprintf(stderr,"ERROR177: Soil type=%d has been already defined in fscansoilpar().\n",id);
      break;
    }
    if(fscanstring(file,s))
    {
      if(isout)
        readstringerr("name");
      break;
    }
    soil->name=strdup(s);
    check(soil->name);
    soil->type=id;
    fscanreal2(isout,file,&soil->Ks,"Ks");
    fscanreal2(isout,file,&soil->Sf,"Sf");
    fscanreal2(isout,file,&soil->sand,"sand");
    fscanreal2(isout,file,&soil->silt,"silt");
    fscanreal2(isout,file,&soil->clay,"clay");
/*    soil->beta_soil=-2.655/log10(soil->wfc/soil->wsat);
    soil->whcs_all=0.0;
    for(l=0;l<LASTLAYER;l++)
    {
      soil->whc[l]=soil->wfc-soil->wpwp;
      soil->whcs[l]=soil->whc[l]*soildepth[l];
      soil->wpwps[l]=soil->wpwp*soildepth[l];
      soil->wsats[l]=soil->wsat*soildepth[l];
      soil->bulkdens[l]=(1-soil->wsat)*MINERALDENS;
      if(soil->type==ROCK)
        soil->k_dry[l]=8.8;
      else
        soil->k_dry[l]=(0.135*soil->bulkdens[l]+64.7)/
               (MINERALDENS-0.947*soil->bulkdens[l]);
    }
    assume last layer is bedrock in 6-layer version 
    soil->whc[BOTTOMLAYER]=0.002; 0.006 wsats - 0.002 whc - 0.001 wpwps = 0.003 for free water */
/*    soil->whcs[BOTTOMLAYER]=soil->whc[BOTTOMLAYER]*soildepth[BOTTOMLAYER];
    soil->wpwps[BOTTOMLAYER]=0.001*soildepth[BOTTOMLAYER];
    soil->wsats[BOTTOMLAYER]=0.006*soildepth[BOTTOMLAYER];
    soil->bulkdens[BOTTOMLAYER]=(1-soil->wsats[BOTTOMLAYER]/soildepth[BOTTOMLAYER])*MINERALDENS;
    soil->k_dry[BOTTOMLAYER]=0.039*pow(soil->wsats[BOTTOMLAYER]/soildepth[BOTTOMLAYER],-2.2); */
    fscanint2(isout,file,&soil->hsg,"hsg");
    if(soil->hsg<1 || soil->hsg>NHSG)
    {
      if(isout)
        fprintf(stderr,"ERROR199: Invalid value=%d for hsg in soil '%s'.\n",
                soil->hsg,soil->name);
      break;
    }
    fscanreal2(isout,file,&soil->tdiff_0,"tdiff_0");
    fscanreal2(isout,file,&soil->tdiff_15,"tdiff_15");
    fscanreal2(isout,file,&soil->tdiff_100,"tdiff_100");
    fscanreal2(isout,file,&soil->tcond_pwp,"cond_pwp");
    fscanreal2(isout,file,&soil->tcond_100,"cond_100");
    fscanreal2(isout,file,&soil->tcond_100_ice,"cond_100_ice");
    fscanreal2(isout,file,&soil->denit_rate,"denit_rate");
    fscanreal2(isout,file,&soil->a_denit,"a_denit");
    fscanreal2(isout,file,&soil->b_denit,"b_denit");
    fscanreal2(isout,file,&soil->anion_excl,"anion_excl");
    fscanreal2(isout,file,&soil->nitvol_factor_temp,"nitvol_temp_factor");
    fscanreal2(isout,file,&soil->vol_cation_exchange,"vol_cation_exchange");
    fscanreal2(isout,file,&soil->denit_water_threshold,"denit_water_threshold");
    fscanreal2(isout,file,&soil->a_nit,"a_nit");
    fscanreal2(isout,file,&soil->b_nit,"b_nit");
    fscanreal2(isout,file,&soil->c_nit,"c_nit");
    fscanreal2(isout,file,&soil->d_nit,"d_nit");
    soil->z_nit=soil->d_nit*(soil->b_nit-soil->a_nit)/(soil->a_nit-soil->c_nit);
    soil->n_nit=soil->a_nit-soil->b_nit;
    soil->m_nit=soil->a_nit-soil->c_nit;
    fscanreal2(isout,file,&soil->C_corr,"C_corr");
    fscanreal2(isout,file,&soil->bd,"bd");
    fscanreal2(isout,file,&soil->cn_ratio,"C:N ratio");
  } /* of 'for(n=0;...)' */

   
  if(isout && n<nsoil)
    fprintf(stderr,"WARNING010: Number of soil types truncated to %d.\n",n);
  return n;
} /* of 'fscansoilpar' */
