/***************************************************************************/
/**                                                                       **/
/**                     e  q  u  i  l  s  o  i  l  . c                    **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran/C++ version     **/
/**                                                                       **/
/**     SOIL DECOMPOSITION EQUILIBRIUM CALCULATION                        **/
/**                                                                       **/
/**     Analytical solution of differential flux equations for fast and   **/
/**     slow soil carbon pools.  Implemented after (soil_equil_year)      **/
/**     simulation years, when annual litter inputs should be close to    **/
/**     equilibrium.  Assumes average climate (temperature and soil       **/
/**     moisture) from all years up to soil_equil_year.                   **/
/**                                                                       **/
/**       Rate of change of soil pool size = litter input - decomposition **/
/**         (5) dc/dt = litter_decom - kc                                 **/
/**       At equilibrium,                                                 **/
/**         (6) dc/dt = 0                                                 **/
/**       From (5) & (6),                                                 **/
/**         (7) c = litter_decom / k                                      **/
/**                                                                       **/
/**     written by Sibyll Schaphoff                                       **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: $Date:: 2018-02-26 10:18:51 +0100 (Mon, 26 Feb 201#$ **/
/**     By         : $Author:: herzfeld                        $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"

void equilsoil(Soil *soil,int ntotpft,const Pftpar pftpar[])
{
  int l,p,f;
  Real sumlitter,pftlitter,wood=0,socfraction;
  Pool *pool,*sum_cpool;
  Pool *equilc[LASTLAYER],equilc_tot[LASTLAYER];
  Poolpar *equil_shift[LASTLAYER];
  Poolpar *sum_shift,*k_mean,*equilc_pft,*c0,*sum,*n0;
  pool=newvec(Pool,ntotpft);
  check(pool);
  sum_cpool=newvec(Pool,ntotpft);
  check(sum_cpool);
  sum_shift=newvec(Poolpar,ntotpft);
  check(sum_shift);
  equilc_pft=newvec(Poolpar,ntotpft);
  check(equilc_pft);
  k_mean=newvec(Poolpar,ntotpft);
  check(k_mean);
  c0=newvec(Poolpar,ntotpft);
  check(c0);
  n0=newvec(Poolpar,ntotpft);
  check(n0);
  sum=newvec(Poolpar,ntotpft);
  check(sum);

  for(l=0;l<LASTLAYER;l++)
  {
    equilc[l]=newvec(Pool,ntotpft);
    check(equilc[l]);
    equil_shift[l]=newvec(Poolpar,ntotpft);
    check(equil_shift[l]);
  }

  sumlitter=littercarbon(&soil->litter);
  soil->decomp_litter_mean.carbon/=(soil_equil_year-veg_equil_year);
  soil->decomp_litter_mean.nitrogen/=(soil_equil_year-veg_equil_year);
  forrootsoillayer(l)
  {
     soil->k_mean[l].fast/=(soil_equil_year-veg_equil_year);
     soil->k_mean[l].slow/=(soil_equil_year-veg_equil_year);
  }
  for(p=0;p<ntotpft;p++)
	pool[p].slow.carbon=pool[p].fast.carbon=sum_shift[p].fast=sum_shift[p].slow=
	equilc_pft[p].slow=equilc_pft[p].fast=k_mean[p].fast=k_mean[p].slow=sum[p].fast=sum[p].slow=0.0;
  forrootsoillayer(l)
    for(p=0;p<ntotpft;p++)
    {
       socfraction=pow(10,pftpar[p].soc_k*logmidlayer[l])
                 - (l>0 ? pow(10,pftpar[p].soc_k*logmidlayer[l-1]): 0);
       k_mean[p].fast+=soil->k_mean[l].fast*socfraction;
       k_mean[p].slow+=soil->k_mean[l].slow*socfraction;
    }
  for(p=0;p<ntotpft;p++)
  {
    c0[p].fast=k_mean[p].fast>epsilon ? (1-param.atmfrac)*param.fastfrac*(soil->decomp_litter_mean.carbon)/k_mean[p].fast : 0;
    c0[p].slow=k_mean[p].slow>epsilon ? (1-param.atmfrac)*(1.0-param.fastfrac)*(soil->decomp_litter_mean.carbon)/k_mean[p].slow : 0;
    n0[p].fast=k_mean[p].fast>epsilon ? (1-param.atmfrac)*param.fastfrac*(soil->decomp_litter_mean.nitrogen)/k_mean[p].fast : 0;
    n0[p].slow=k_mean[p].slow>epsilon ? (1-param.atmfrac)*(1.0-param.fastfrac)*(soil->decomp_litter_mean.nitrogen)/k_mean[p].slow : 0;
  }

  if(soil->decomp_litter_mean.carbon>epsilon) 
  {
    forrootsoillayer(l)
    {
      soil->pool[l].slow.carbon=soil->pool[l].fast.carbon=0;
      soil->pool[l].slow.nitrogen=soil->pool[l].fast.nitrogen=0;
      equilc_tot[l].fast.carbon=soil->k_mean[l].fast>epsilon ? (1-param.atmfrac)*param.fastfrac*soil->decomp_litter_mean.carbon/soil->k_mean[l].fast : 0;
      equilc_tot[l].slow.carbon=soil->k_mean[l].slow>epsilon ? (1-param.atmfrac)*(1.0-param.fastfrac)*soil->decomp_litter_mean.carbon/soil->k_mean[l].slow : 0;
      /*  equilc_tot[l].fast=soil->k_mean[l].fast>0 ? soilfrac*fastfrac*(soil->decomp_litter_mean)/
                     ((1.0-exp(-soil->k_mean[l].fast))) : 0;
      equilc_tot[l].slow.carbon=soil->k_mean[l].slow>0 ? soilfrac*(1.0-fastfrac)*(soil->decomp_litter_mean)/
                   ((1.0-exp(-soil->k_mean[l].slow))) : 0;*/


      for(p=0;p<ntotpft;p++)
      {
        socfraction=pow(10,pftpar[p].soc_k*logmidlayer[l])
                - (l>0 ? pow(10,pftpar[p].soc_k*logmidlayer[l-1]): 0);
        equil_shift[l][p].slow=soil->k_mean[l].slow*socfraction;
        equil_shift[l][p].fast=soil->k_mean[l].fast*socfraction;
        sum_shift[p].fast+=equil_shift[l][p].fast;
        sum_shift[p].slow+=equil_shift[l][p].slow;
/*          equilc[l][p].fast=equilc_tot[l].fast*socfraction;
          equilc[l][p].slow=equilc_tot[l].slow*socfraction;
          equilc_pft[p].fast+=equilc[l][p].fast;
          equilc_pft[p].slow+=equilc[l][p].slow;*/
      }
    }
    forrootsoillayer(l)
    {
/*	  for(p=0;p<ntotpft;p++){
	    soil->c_shift_fast[l][p]=equilc[l][p].fast/equilc_pft[p].fast;
	    soil->c_shift_slow[l][p]=equilc[l][p].slow/equilc_pft[p].slow;
	    soil->c_shift_fast[l][p]=equil_shift[l][p].fast/sum_shift[p].fast;
            soil->c_shift_slow[l][p]=equil_shift[l][p].slow/sum_shift[p].slow;

	  }*/
      for(p=0;p<soil->litter.n;p++)
      {
        wood=0;
        for(f=0;f<NFUELCLASS;f++)
          wood+=soil->litter.ag[p].trait.wood[f].carbon+soil->litter.agsub[p].trait.wood[f].carbon;
        pftlitter=soil->litter.bg[p].carbon+soil->litter.ag[p].trait.leaf.carbon+soil->litter.agsub[p].trait.leaf.carbon+wood;
        socfraction=pow(10,soil->litter.ag[p].pft->soc_k*logmidlayer[l])
                    - (l>0 ? pow(10,soil->litter.ag[p].pft->soc_k*logmidlayer[l-1]): 0);
/*	    soil->pool[l].slow.carbon+=equilc_tot[l].slow*soil->c_shift_slow[l][soil->litter.ag[p].pft->id]*pftlitter/(sumlitter>epsilon?sumlitter:1);
            soil->pool[l].fast.carbon+=equilc_tot[l].fast*soil->c_shift_fast[l][soil->litter.ag[p].pft->id]*pftlitter/(sumlitter>epsilon?sumlitter:1);
 */         
            if(sumlitter>0)
              soil->pool[l].slow.carbon+=c0[soil->litter.ag[p].pft->id].slow*socfraction*pftlitter/sumlitter;
            if(sumlitter>0)
              soil->pool[l].fast.carbon+=c0[soil->litter.ag[p].pft->id].fast*socfraction*pftlitter/sumlitter;
/*            soil->pool[l].slow.nitrogen+=n0[soil->litter.ag[p].pft->id].slow*socfraction*pftlitter/sumlitter;
            soil->pool[l].fast.nitrogen+=n0[soil->litter.ag[p].pft->id].fast*socfraction*pftlitter/sumlitter;*/
      }
      soil->pool[l].slow.nitrogen=soil->pool[l].slow.carbon/soil->par->cn_ratio;
      soil->pool[l].fast.nitrogen=soil->pool[l].fast.carbon/soil->par->cn_ratio;
      soil->NH4[l]=0;//soil->pool[l].slow.nitrogen/10000;
      soil->NO3[l]=0;//soil->pool[l].slow.nitrogen/10000;
      //if (soil->decomp_litter_mean.nitrogen<0.2)
      if (soil->decomp_litter_mean.nitrogen<epsilon)
      {
        soil->NH4[l]=0;
        soil->NO3[l]=0;
        soil->pool[l].slow.nitrogen=0;
        soil->pool[l].fast.nitrogen=0;
      }
    }
    forrootsoillayer(l)
    {
      for(p=0;p<ntotpft;p++)
      {
        socfraction=pow(10,pftpar[p].soc_k*logmidlayer[l])
                  - (l>0 ? pow(10,pftpar[p].soc_k*logmidlayer[l-1]): 0);
        soil->c_shift_fast[l][p]=(c0[p].fast*socfraction-c0[p].fast*socfraction*(1-soil->k_mean[l].fast))/((1-param.atmfrac)*param.fastfrac*(soil->decomp_litter_mean.carbon));
        soil->c_shift_slow[l][p]=(c0[p].slow*socfraction-c0[p].slow*socfraction*(1-soil->k_mean[l].slow))/((1-param.atmfrac)*(1.0-param.fastfrac)*(soil->decomp_litter_mean.carbon));
 /*   fprintf(stdout,"equilsoil socfraction=%.5f soil_c_shift[%d][%d].fast=%.5f soil_c_shift[%d][%d].slow=%.5f\n",socfraction,l,p,soil->c_shift_fast[l][p],l,p,soil->c_shift_slow[l][p]);*/
        sum[p].fast+=soil->c_shift_fast[l][p];
        sum[p].slow+=soil->c_shift_slow[l][p];
      }

      soil->k_mean[l].slow=soil->k_mean[l].fast=0.0;
    }
    for(p=0;p<ntotpft;p++)
    {
      if(sum[p].fast<1.0 && sum[p].fast>0.0)
        for (l=0;l<LASTLAYER;l++) soil->c_shift_fast[l][p]=soil->c_shift_fast[l][p]/sum[p].fast;
      else if (sum[p].fast<epsilon)
      {
        soil->c_shift_fast[0][p]=1.0;
        for (l=1;l<LASTLAYER;l++) soil->c_shift_fast[l][p]=0;
      }
      if(sum[p].slow<1.0 && sum[p].slow>0.0)
        for (l=0;l<LASTLAYER;l++) soil->c_shift_slow[l][p]=soil->c_shift_slow[l][p]/sum[p].slow;
      else if (sum[p].slow<epsilon)
      {
        soil->c_shift_slow[0][p]=1.0;
        for (l=1;l<LASTLAYER;l++) soil->c_shift_slow[l][p]=0;
      }
    }
    soil->count=0;
    soil->decomp_litter_mean.carbon=soil->decomp_litter_mean.nitrogen=0.0;
  } else { /* if there is no carbon, remove mineral N from initialization */
    forrootsoillayer(l){
      soil->NH4[l]=soil->NO3[l]=0.0;
      soil->pool[l].slow.nitrogen=0;
      soil->pool[l].fast.nitrogen=0;
      soil->pool[l].slow.carbon=0;
      soil->pool[l].fast.carbon=0;
    }
  }
  /* freeing memory */
  free(pool);
  free(sum_cpool);
  free(sum_shift);
  free(equilc_pft);
  free(k_mean);
  free(c0);
  free(sum);

  for(l=0;l<LASTLAYER;l++)
  {
    free(equilc[l]);
    free(equil_shift[l]);
  }

} /* of 'equilsoil' */
