/***************************************************************************/
/**                                                                       **/
/**         d  e  n  i  t  r  i  f  i  c  a  t  i  o  n  .  c             **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran/C++ version     **/
/**                                                                       **/
/**     written by Sibyll Schaphoff, Guilaume Vilain                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: $Date:: 2018-11-19 11:44:30 +0100 (Mon, 19 Nov 201#$ **/
/**     By         : $Author:: cmueller                        $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"
#include "agriculture.h"

void denitrification(Stand *stand,
                     Output *output)
{
  /* determines NO2 and N2 from nitrate NO3 */
  Real N_denit=0; /* amount of nitrate lost to denitrification */
  Real N2O_denit, denit_t;
  Real FT,FW,TCDF;
  Real Corg;
  Irrigation *data;
  int l,p,i;
  Pft *pft;
  Bool growing = FALSE;
  Stand *stand2;
#ifdef DEBUG
  printf("NBEFORE");
  for(l=0;l<NSOILLAYER-1;l++)
    printf("%g ",soil.NO3[l]);
  printf("\n");
#endif
  /* checking if CFT for daily outputs is still growing 
  on an agricultural stand so that setaside data can be used otherwise */
  foreachstand(stand2, i, stand->cell->standlist)
  {
    if (stand2->type->landusetype == AGRICULTURE)
    {
      data = stand2->data;
      foreachpft(pft, p, &stand2->pftlist)
      {
        if (pft->par->id == output->daily.cft && data->irrigation == output->daily.irrigation)
          growing = TRUE;
      }
    }
  }
  forrootsoillayer(l)
  {
    Corg = stand->soil.pool[l].fast.carbon+ stand->soil.pool[l].slow.carbon;
    if(Corg<0)
      Corg=0;
    if(stand->soil.temp[l]>epsilon)
      FT = 0.0326+0.00351*pow(stand->soil.temp[l],1.652)-pow((stand->soil.temp[l]/41.748),7.19);
      /* Equation C5 from Smith et al 2014 but only for positive temp */
    else if (stand->soil.temp[l] > 45.9) /* otherwise FT is negative */
      FT=0.0;
    else
      FT=0.0326;
#ifdef DEBUG
    printf("w=(%g + %g + %g  + %g + %g )/ %g\n",soil.wpwps[l],soil.w[l]*soil.whcs[l],soil.ice_depth[l],
         soil.w_fw[l],soil.ice_fw[l],soil.wsats[l]);
#endif
    denit_t = (stand->soil.wpwps[l]+ stand->soil.w[l]* stand->soil.whcs[l]+ stand->soil.ice_depth[l]+
      stand->soil.w_fw[l]+ stand->soil.ice_fw[l])/ stand->soil.wsats[l]; /* denitrification threshold dependent on water filled pore space */

    /* Version without threshold*/
    N_denit = 0.0;
    N2O_denit = 0.0;
    if(stand->soil.temp[l]<=45.9){
      FW = 6.664096e-10*exp(21.12912*denit_t); /* newly fitted parameters on curve with threshold */
      TCDF = 1-exp(-CDN*FT*Corg);
      N_denit = FW*TCDF*stand->soil.NO3[l];
    }
#ifdef SAFE
    if((FW*TCDF)>1.0 && N_denit>(stand->soil.NO3[l]-epsilon*10)){
      fprintf(stdout,"too large denitrification in layer %d: N_denit %g FW %g TCDF %g NO3 %g FT %g Corg %g\n",l,N_denit,FW,TCDF, stand->soil.NO3[l],FT,Corg);
      fflush(stdout);

      N_denit= stand->soil.NO3[l];
    }
#endif
    if(N_denit>stand->soil.NO3[l])
      N_denit= stand->soil.NO3[l];
    stand->soil.NO3[l]-=N_denit;
#ifdef SAFE
    if(stand->soil.NO3[l]<-epsilon)
      fail(NEGATIVE_SOIL_NO3_ERR,TRUE,"Negative soil NO3=%g in layer %d", stand->soil.NO3[l],l);
#endif
    /* Calculation of N2O from denitrification after Bessou 2010 */
    N2O_denit = 0.11 * N_denit;
    N_denit -= N2O_denit;
    if (stand->type->landusetype == AGRICULTURE) {
      data = stand->data;
      foreachpft(pft, p, &stand->pftlist) {
        if (pft->par->id == output->daily.cft && data->irrigation == output->daily.irrigation) {
          output->daily.n2_denit += N_denit;
          output->daily.n2o_denit += N2O_denit;
        }
      }
    }
    else if (stand->type->landusetype == SETASIDE_RF && output->daily.irrigation == FALSE && growing == FALSE) {
      output->daily.n2_denit += N_denit;
      output->daily.n2o_denit += N2O_denit;
    }
    else if (stand->type->landusetype == SETASIDE_IR && output->daily.irrigation == TRUE && growing == FALSE) {
      output->daily.n2_denit += N_denit;
      output->daily.n2o_denit += N2O_denit;
    }
    output->mn2o_denit+=N2O_denit*stand->frac;
    output->mn2_emissions+=N_denit*stand->frac;
  }
#ifdef DEBUG
  printf("NAFTER");
  for(l=0;l<NSOILLAYER-1;l++)
    printf("%g ",soil.NO3[l]);
  printf("\n");
#endif
} /* of 'denitrification' */
