/***************************************************************************/
/**                                                                       **/
/**             w  a  t  e  r  _  s  t  r  e  s  s  e  d  .  c            **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran/C++ version     **/
/**                                                                       **/
/**     written by Werner von Bloh, Sibyll Schaphoff                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: $Date:: 2018-11-19 12:12:20 +0100 (Mon, 19 Nov 201#$ **/
/**     By         : $Author:: cmueller                        $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"
#include "agriculture.h"

#define EPSILON 0.001  /* min precision of solution in bisection method */

typedef struct
{
  Real fac,co2,temp,apar,daylength,tstress,vmax;
  int path;
} Data;

static Real fcn(Real lambda,Data *data)
{
  Real agd,rd,vmax;

/*
 *              Call photosynthesis to determine alternative total
 *              daytime photosynthesis estimate (adt2) implied by
 *              Eqns 2 & 19, Haxeltine & Prentice 1996, and current
 *              guess for lambda (xmid)
 */
  vmax=data->vmax;
  return data->fac*(1-lambda)-photosynthesis(&agd,&rd,&vmax,data->path,lambda,
                                             data->tstress,data->co2,
                                             data->temp,data->apar,
                                             data->daylength);
/*
 *              Calculate total daytime photosynthesis implied by
 *              canopy conductance from water balance routine and
 *              current guess for lambda (xmid).  Units are mm/m2/day
 *              (mm come from gpd value, mm/day)
 *              Eqn 18, Haxeltine & Prentice 1996
 */

} /* of 'fcn' */

Real water_stressed(Pft *pft, /* pointer to PFT variabels */
                    Real aet_layer[LASTLAYER],
                    Real gp_stand,
                    Real gp_stand_leafon, /* pot. canopy conduct. at full leaf cover */
                    Real gp_pft, /* potential canopy conductance */
                    Real *gc_pft,
                    Real *rd,
                    Real *wet,
                    Real pet,  /* potential evapotranspiration (mm) */
                    Real co2,  /* Atmospheric CO2 partial pressure (ppmv) */
                    Real temp, /* Temperature (deg C) */
                    Real par,  /* photosynthetic active radiation (J/m2/day) */
                    Real daylength, /* Daylength (h) */
                    Real *wdf,           /* water deficit fraction (0..100) */
                    int npft,
                    int ncft,
                    const Config *config
                   ) /* returns gross primary productivity (gC/m2) */
{
  int l,i,iter;
  Real supply,supply_pft,demand,demand_pft,wr,lambda,gpd,agd,gc,aet,aet_cor;
  Data data;
  Real roots,vmax;
  Real rootdist_n[LASTLAYER];
  Real aet_tmp[LASTLAYER];
  Real layer,root_u,root_nu;
  Real freeze_depth,thaw_depth;
  Real adtmm;
  Real gc_old,agd_old,gc_new,demand_old;
  Bool isless=FALSE;
  Irrigation *irrig;

  wr=gpd=agd=*rd=layer=root_u=root_nu=aet_cor=0.0;
  forrootsoillayer(l){
    rootdist_n[l]=pft->par->rootdist[l];
    aet_tmp[l]=0.0;
  }
  if(config->permafrost)
  {
    /*adjust root layer*/
    if(layerbound[BOTTOMLAYER]>pft->stand->soil.mean_maxthaw &&
       pft->stand->soil.mean_maxthaw>epsilon)
    {
      forrootsoillayer(l)
      {
        layer+=soildepth[l];
        root_u+=pft->par->rootdist[l];
        freeze_depth=layer-pft->stand->soil.mean_maxthaw;
        if (freeze_depth>0)
        {
          thaw_depth=soildepth[l]-freeze_depth;
          rootdist_n[l]=thaw_depth/soildepth[l]*pft->par->rootdist[l];
          root_nu=pft->par->rootdist[l]-rootdist_n[l];
          root_u-= root_nu;
          l++;
          break;
        }
      }
      for(i=l;i<BOTTOMLAYER;i++)
      {
        root_nu+=rootdist_n[i];
        rootdist_n[i]=0;
      }
      for(i=l-1;i>=0;--i)
        rootdist_n[i]=rootdist_n[i]/root_u*root_nu+rootdist_n[i];
    }
  }
  wr=roots=0;
  for(l=0;l<LASTLAYER;l++)
  {
    wr+=rootdist_n[l]*pft->stand->soil.w[l];
    roots+=rootdist_n[l];
    /*printf("wr=%lf rootdist=%lf w=%lf\n",wr,rootdist_n[l],pft->stand->soil.w[l]);*/
  }

  if(*wet>0.99)
    *wet=0.99;

  if(pft->stand->type->landusetype==AGRICULTURE)
  {
    supply=pft->par->emax*wr*(1-exp(-1.0*pft->par->sla*((Pftcrop *)pft->data)->ind.root.carbon));
    //if (pft->phen>0)
    //{
    //  gp_stand=gp_stand/pft->phen*fpar(pft);
    //  gp_pft=gp_pft/pft->phen*fpar(pft);
   // }
  }
  else
  {
    supply=pft->par->emax*wr*pft->phen;
  }

  supply_pft=supply*pft->fpc;
  demand=(gp_stand>0) ? (1.0-*wet)*pet*param.ALPHAM/(1+(param.GM*param.ALPHAM)/gp_stand) : 0;
  demand_pft=(gp_pft>0) ? (1.0-*wet)*pet*param.ALPHAM/(1+(param.GM*param.ALPHAM)/gp_pft) : 0;
  *wdf=wdf(pft,demand,supply);

  if(pet>0 && gp_stand_leafon>0 && pft->fpc>0)
  {
    /*pft->wscal=(pft->par->emax*wr*pft->fpc)/(pet*ALPHAM/(1+GM/(gp_stand_leafon+pft->par->gmin*pft->fpc)));*/
    pft->wscal=(pft->par->emax*wr)/(pet*param.ALPHAM/(1+(param.GM*param.ALPHAM)/gp_stand_leafon));
    if(pft->wscal>1)
      pft->wscal=1;
  }
  else
    pft->wscal=1;

  pft->wscal_mean+=pft->wscal;

  if(supply_pft>=demand_pft)
    *gc_pft=gp_pft;
  else if(pet>0)
  {
    *gc_pft=(param.GM*param.ALPHAM)*supply_pft/((1.0-*wet)*pet*param.ALPHAM-supply_pft);
    if(*gc_pft<0)
      *gc_pft=0;
  }
  else
    *gc_pft=0;

  aet=(wr>0) ? min(supply,demand)/wr*pft->fpc : 0;
  for (l=0;l<LASTLAYER;l++)
  {
    aet_tmp[l]+=aet*rootdist_n[l]*pft->stand->soil.w[l];
    if (aet_tmp[l]>pft->stand->soil.w[l]*pft->stand->soil.whcs[l])
    {
      aet_cor+=pft->stand->soil.w[l]*pft->stand->soil.whcs[l]-aet_layer[l];
      //printf("aet_layer[%d]=%g,aet=%g,aet_cor=%g\n",l,aet_layer[l],aet,aet_cor);
      isless=TRUE;
    }
    else
      aet_cor+=aet_tmp[l];
  }
  if (isless==TRUE && aet_cor<aet)
  {
    supply=aet_cor*wr/pft->fpc;
    aet=aet_cor;
  }
  if(supply>=demand)
    gc=gp_stand;
  else if(pet>0)
  {
    gc=(param.GM*param.ALPHAM)*supply/((1.0-*wet)*pet*param.ALPHAM-supply);
    if(gc<0)
      gc=0;
  }
  else
    gc=0;
  gc_old=gc;
  /*gpd=hour2sec(daylength)*(gc-pft->par->gmin*pft->fpc*pft->phen);*/

  if(pft->stand->type->landusetype==AGRICULTURE)
    gpd=hour2sec(daylength)*(gc-pft->par->gmin*fpar(pft))*pft->fpc;
  else
    gpd=hour2sec(daylength)*(gc-pft->par->gmin*pft->phen)*pft->fpc;

  data.tstress=temp_stress(pft->par,temp,daylength);
  if(gpd>1e-5 && isphoto(data.tstress))
  {
    data.fac=gpd/1.6*ppm2bar(co2);
    data.path=pft->par->path;
    data.temp=temp;
    data.co2=ppm2Pa(co2);
    data.apar=par*alphaa(pft,config->laimax_interpolate)*fpar(pft);
    data.daylength=daylength;
    data.vmax=pft->vmax;
    lambda=bisect((Bisectfcn)fcn,0.02,LAMBDA_OPT+0.05,&data,0,EPSILON,30,&iter);
/*     lambda=zbrent((Bisectfcn)fcn,0.02,LAMBDA_OPT+0.05,EPSILON,&data); */
    vmax=pft->vmax;
    adtmm=photosynthesis(&agd,rd,&vmax,data.path,lambda,data.tstress,data.co2,
                   temp,data.apar,daylength);
      gc_new=(1.6*adtmm/(ppm2bar(co2)*(1.0-lambda)*hour2sec(daylength)))+
                    pft->par->gmin*fpar(pft);
    //if(fabs(gc*pft->fpc-gc_new)>0.1)
    //  printf("gc!=gc_new: %g<%g, lambda=%g,adtmm=%g,fpc=%g,iter=%d\n",gc*pft->fpc,gc_new,lambda,adtmm,pft->fpc,iter);
    agd_old=agd;
    pft->vmax=vmax;
    if(config->with_nitrogen)
    {
      //printf("vm=%g agd=%g\n",pft->vmax,agd);
      nitrogen_stress(&pft->stand->soil,pft,temp,daylength,npft,config->nbiomass,ncft) ;
      //printf("vm=%g,rd=%g\n",pft->vmax,*rd);

      //data.vmax=pft->vmax;
      //lambda=bisect((Bisectfcn)fcn,0.02,LAMBDA_OPT+0.05,&data,0,EPSILON,100);
      adtmm=photosynthesis(&agd,rd,&pft->vmax,data.path,lambda,data.tstress,data.co2,
                     temp,data.apar,daylength);
      gc=(1.6*adtmm/(ppm2bar(co2)*(1.0-lambda)*hour2sec(daylength)))+
                    pft->par->gmin*fpar(pft);
      demand=(gc>0) ? (1-*wet)*pet*param.ALPHAM/(1+(param.GM*param.ALPHAM)/gc) :0;
      //if(gc>gc_new)
      //  printf("gc_old=%g > gc=%g,gc_new=%g, agd_old=%g, agd=%g,supply=%g,demand=%g\n",gc_old,gc,gc_new,agd_old,agd,supply_pft,demand);
      if(gc_new-gc>0.01 &&  demand-supply_pft>0.1)
      {
         demand_old=demand;
         gc_old=gc;
         gc=(param.GM*param.ALPHAM)*supply_pft/((1.0-*wet)*pet*param.ALPHAM-supply_pft);
         if(gc<0)
           gc=0;
         if(pft->stand->type->landusetype==AGRICULTURE)
           gpd=hour2sec(daylength)*(gc-pft->par->gmin*fpar(pft));
         else
           gpd=hour2sec(daylength)*(gc-pft->par->gmin*pft->phen*pft->fpc);
        data.fac=gpd/1.6*ppm2bar(co2);
        data.vmax=pft->vmax;
        lambda=bisect((Bisectfcn)fcn,0.02,lambda,&data,0,EPSILON,20,&iter);
        adtmm=photosynthesis(&agd,rd,&pft->vmax,data.path,lambda,data.tstress,data.co2,
                             temp,data.apar,daylength);
        gc=(1.6*adtmm/(ppm2bar(co2)*(1.0-lambda)*hour2sec(daylength)))+
                      pft->par->gmin*fpar(pft);
        demand=(gc>0) ? (1-*wet)*pet*param.ALPHAM/(1+(param.GM*param.ALPHAM)/gc) :0;
      //if(iter!=0 && iter!=100 && fabs(gc_old-gc)>1e-3)
       // printf("gc=%g > gc_new=%g,agd_old=%g, agd=%g,supply=%g,demand=%g,demand_old=%g,gc_old=%g,lambda=%g,iter=%d\n",gc,gc_new,agd_old,agd,supply_pft,demand,demand_old,gc_old,lambda,iter);
      }
      aet=(wr>0) ? demand*fpar(pft)/wr :0 ;

      //printf("rd=%g\n",*rd);
      pft->nlimit+=pft->vmax/vmax;
      if(pft->stand->type->landusetype==AGRICULTURE){
        irrig=pft->stand->data;
        if(&pft->stand->cell->output.daily!=NULL &&
          pft->par->id==pft->stand->cell->output.daily.cft &&
          irrig->irrigation==pft->stand->cell->output.daily.irrigation){
            pft->stand->cell->output.daily.nlimit=pft->vmax/vmax;
        }
      }
    }
    /* in rare occasions, agd(=GPP) can be negative, but shouldn't */
    agd=max(0,agd);
    *rd=*rd;    /* DON'T DELETE THIS LINE */
  }
  else
    agd=0;
  for (l=0;l<LASTLAYER;l++)
  {
    aet_layer[l]+=aet*rootdist_n[l]*pft->stand->soil.w[l];
    if (aet_layer[l]>pft->stand->soil.w[l]*pft->stand->soil.whcs[l])
      aet_layer[l]=pft->stand->soil.w[l]*pft->stand->soil.whcs[l];
  }
  return agd;
} /* of 'water_stressed' */
