/***************************************************************************/
/**                                                                       **/
/**                u  p  d  a  t  e  _  d  a  i  l  y  .  c               **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran/C++ version     **/
/**                                                                       **/
/**     Function of daily update of individual grid cell                  **/
/**                                                                       **/
/**     written by Werner von Bloh, Sibyll Schaphoff                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     See file AUTHORS for list of authors contributing to this code    **/
/**                                                                       **/
/**     Last change: $Date:: 2019-07-11 15:43:14 +0200 (Thu, 11 Jul 201#$ **/
/**     By         : $Author:: felutz                          $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"
#include "natural.h"
#include "agriculture.h"

#define length 1.0 /* characteristic length (m) */
#ifdef ADJUSTED
#define BIOTURBRATE 0.006308452 /* daily rate for 90% annual bioturbation rate [-]*/
#else
#define BIOTURBRATE 0.001897 /* daily rate for 50% annual bioturbation rate [-]*/
#endif
void update_daily(Cell *cell,            /* cell pointer           */
                  Real co2,              /* atmospheric CO2 (ppmv) */
                  Real popdensity,       /* population density (capita/km2) */
                  Dailyclimate climate,  /* Daily climate values */
                  int day,               /* day (1..365)           */
                  int npft,              /* number of natural PFTs */
                  int ncft,              /* number of crop PFTs   */
                  int year,              /* simulation year */
                  int month,             /* month (0..11) */
                  Bool withdailyoutput,
                  Bool intercrop,
                  const Config *config   /* LPJ config */
                 )
{
  int p,s;
  Real melt=0,pet,par,daylength;
  Real gp_stand,gp_stand_leafon,runoff,snowrunoff;
  Real fpc_total_stand;
  Real gtemp_air;  /* value of air temperature response function */
  Real gtemp_soil[NSOILLAYER]; /* value of soil temperature response function */
  Real temp_bs;    /* temperature beneath snow */
  Real prec_energy; /* energy from temperature difference between rain and soil [J/m2]*/
  Stocks flux_estab={0,0};
  Real evap=0,irrig_water=0;
  Stocks hetres={0,0};
  Real *gp_pft;
  Real bnf;
  Real nh3;
  Stand *stand;
  Pft *pft;
  Real n_immo = 0;
  Irrigation *data;
  int l,index;
  Livefuel livefuel={0,0,0,0,0};
  const Real prec_save=climate.prec;
  Real n2o_nit = 0.0;
  gp_pft=newvec(Real,npft+ncft);
  check(gp_pft);
  switch(config->with_radiation)
  {
    case CLOUDINESS:
      petpar(&daylength,&par,&pet,&climate.swdown,cell->coord.lat,day,climate.temp,climate.sun);
      break;
    case RADIATION:
      petpar2(&daylength,&par,&pet,cell->coord.lat,day,climate.temp,
              climate.lwnet,climate.swdown,FALSE);
      break;
    case RADIATION_LWDOWN:
      petpar2(&daylength,&par,&pet,cell->coord.lat,day,climate.temp,
              climate.lwnet,climate.swdown,TRUE);
      break;
    case RADIATION_SWONLY:
      petpar3(&daylength,&par,&pet,cell->coord.lat,day,climate.temp,
              climate.swdown);
      break;
  }
  cell->output.mpet+=pet*param.ALPHAM;

#ifdef DEBUG3
  printf("par= %.2f  pet= %.5f daylength= %.5f\n",par,pet,daylength);
  printf("temp= %.2f sun= %.2f\n",climate.temp,climate.sun);
  fflush(stdout);
#endif
  updategdd(cell->gdd,config->pftpar,npft,climate.temp);
  cell->balance.aprec+=climate.prec;
  gtemp_air=temp_response(climate.temp);
  daily_climbuf(&cell->climbuf,climate.temp);

  cell->output.mprec+=climate.prec;
  cell->output.msnowf+=climate.temp<tsnow ? climate.prec : 0;
  cell->output.mrain+=climate.temp<tsnow ? 0 : climate.prec;

  if(cell->ml.landfrac!=NULL) /* landuse enabled? */
  {
    /* calling reduced calc_cropdates for computing vern_date20 as needed for vernalization */
    calc_cropdates(config->pftpar + npft, &cell->climbuf, cell->ml.cropdates, cell->coord.lat,
      day, ncft);
    if(config->sdate_option==NO_FIXED_SDATE ||
      (config->sdate_option==FIXED_SDATE && year<=param.sdate_fixyear))
    {
      update_fallowdays(cell->ml.cropdates,cell->coord.lat,day,ncft);
      flux_estab=sowing_season(cell,day,npft,ncft,climate.prec,config,year);
    }
    else
      flux_estab=sowing_prescribe(cell,day,npft,ncft,config,year);
//	if ((config->with_tillage==TILLAGE && cell->ml.sdate_fixed[ncft]-1==day) || (config->with_tillage==TILLAGE && cell->ml.sowing_month[ncft]-1==month))
  }
  cell->discharge.drunoff=0.0;

  if(config->fire==SPITFIRE)
  {
    if (climate.prec >= 3.0 || (climate.tmin - 4.0) <= 0) /* benp, kirsten */
      cell->ignition.nesterov_accum=0;
    else
      cell->ignition.nesterov_accum += nesterovindex(climate.tmin,climate.tmax);
  }

  foreachstand(stand,s,cell->standlist)
  {
    /* bioturbation assuming 0.2% incorporation of above ground litter leaves to ag_sub */
    for (l = 0; l < stand->soil.litter.n; l++) {
      stand->soil.litter.agsub[l].trait.leaf.carbon += stand->soil.litter.ag[l].trait.leaf.carbon*BIOTURBRATE;
      stand->soil.litter.ag[l].trait.leaf.carbon *= 1 - BIOTURBRATE;
      stand->soil.litter.agsub[l].trait.leaf.nitrogen += stand->soil.litter.ag[l].trait.leaf.nitrogen*BIOTURBRATE;
      stand->soil.litter.ag[l].trait.leaf.nitrogen *= 1 - BIOTURBRATE;
    }

    if(stand->type->landusetype==NATURAL && param.black_fallow && (day==152 || day==335))
    {
      if(param.prescribe_residues)
      {
        index=findlitter(&stand->soil.litter,config->pftpar+npft);
        if(index==NOT_FOUND)
          index=addlitter(&stand->soil.litter,config->pftpar+npft)-1;
        stand->soil.litter.ag[index].trait.leaf.carbon+=param.residue_rate*(1-param.residue_fbg)/2;
        stand->soil.litter.ag[index].trait.leaf.nitrogen+=param.residue_rate*(1-param.residue_fbg)/param.residue_cn/2;
        stand->soil.litter.bg[index].carbon+=param.residue_rate*param.residue_fbg/2;
        stand->soil.litter.bg[index].nitrogen+=param.residue_rate*param.residue_fbg/param.residue_cn/2;
        stand->cell->output.flux_estab.carbon+=param.residue_rate/2;
        stand->cell->output.flux_estab.nitrogen+=param.residue_rate/param.residue_cn/2;
        updatelitterproperties(stand);
      }
      if(param.fix_fertilization)
      {
        stand->soil.NO3[0]+=param.fertilizer_rate/2/2;
        stand->soil.NH4[0]+=param.fertilizer_rate/2/2;
        cell->balance.n_influx+=param.fertilizer_rate/2*stand->frac;
      }
      if(param.till_fallow)
      {
        tillage(stand,param.residue_frac);
        pedotransfer(stand,NULL,NULL,stand->frac);
        updatelitterproperties(stand);
      }
    }

    /* data=stand->data;
       if(stand->type->landusetype==AGRICULTURE){
      data=stand->data;
      irrig_water=((climate.temp-stand->soil.temp[TOPLAYER])*data->irrig_amount)*1e-3;
    } */ /* TODO This is a template to account for temperature of irrigation water,
    but since we don't know which temperature to start with (climate.temp), we leave it for further development JJ 05/2015 */
    if(config->fire==SPITFIRE && cell->afire_frac<1)
      dailyfire_stand(stand,&livefuel,popdensity,climate,config->ntypes);
    if(config->permafrost)
    {
      snowrunoff=snow(&stand->soil,&climate.prec,&melt,
                                    climate.temp,&temp_bs,&evap)*stand->frac;
      cell->discharge.drunoff+=snowrunoff;
      cell->output.mevap+=evap*stand->frac; /* evap from snow runoff*/
      /* prec_energy = ((climate.temp-stand->soil.temp[TOPLAYER])*climate.prec*1e-3
                    +melt*1e-3*(T_zero-stand->soil.temp[TOPLAYER])+irrig_water)*c_water; */
      if(climate.prec+melt>epsilon)
      {
        prec_energy=climate.temp*climate.prec*1e-3*c_water + T_zero*melt*1e-3*c_water + stand->soil.litter.agtop_temp*stand->soil.litter.agtop_moist*1e-3*c_water + stand->soil.litter.agtop_temp*litter_ag_sum(&stand->soil.litter)/0.5*1e-3/1300*2.5e6;
        stand->soil.litter.agtop_temp=prec_energy/(climate.prec*1e-3*c_water + melt*1e-3*c_water + stand->soil.litter.agtop_moist*1e-3*c_water + litter_ag_sum(&stand->soil.litter)/0.5*1e-3/1300*2.5e6);
        stand->soil.perc_energy[TOPLAYER]=(stand->soil.litter.agtop_temp-stand->soil.temp[TOPLAYER])*(climate.prec+melt)*1e-3*c_water;
      }
      else
	stand->soil.perc_energy[TOPLAYER]=0;

      /*THIS IS DEDICATED TO MICROBIOLOGICAL HEATING*/
 /*     foreachsoillayer(l) stand->soil.micro_heating[l]=m_heat*stand->soil.decomC[l];
      stand->soil.micro_heating[0]+=m_heat*stand->soil.litter.decomC;*/

      soiltemp(&stand->soil,temp_bs);
      foreachsoillayer(l)
        gtemp_soil[l]=temp_response(stand->soil.temp[l]);
    }
    else
    {
      gtemp_soil[0]=temp_response(soiltemp_lag(&stand->soil,&cell->climbuf));
      for(l=1;l<NSOILLAYER;l++)
        gtemp_soil[l]=gtemp_soil[0];
      stand->soil.litter.agtop_temp=soiltemp_lag(&stand->soil,&cell->climbuf);
      snowrunoff=snow_old(&stand->soil.snowpack,&climate.prec,&melt,climate.temp)*stand->frac;
      cell->discharge.drunoff+=snowrunoff;
    }
    cell->output.msoiltemp1+=stand->soil.temp[0]*ndaymonth1[month]*stand->frac*(1.0/(1-stand->cell->lakefrac));
    cell->output.msoiltemp2+=stand->soil.temp[1]*ndaymonth1[month]*stand->frac*(1.0/(1-stand->cell->lakefrac));
    cell->output.msoiltemp3+=stand->soil.temp[2]*ndaymonth1[month]*stand->frac*(1.0/(1-stand->cell->lakefrac));
    cell->output.msoiltemp4+=stand->soil.temp[3]*ndaymonth1[month]*stand->frac*(1.0/(1-stand->cell->lakefrac));
    cell->output.msoiltemp5+=stand->soil.temp[4]*ndaymonth1[month]*stand->frac*(1.0/(1-stand->cell->lakefrac));
    cell->output.msoiltemp6+=stand->soil.temp[5]*ndaymonth1[month]*stand->frac*(1.0/(1-stand->cell->lakefrac));

#ifdef DEBUG_N
  printf("BEFORE_LITTERSOM: ");
  for(l=0;l<NSOILLAYER-1;l++)
     printf("%g ",stand->soil.NO3[l]);
  printf("\n");
#endif
  

    /* update soil and litter properties to account for all changes since last call of littersom */
    pedotransfer(stand,NULL,NULL,stand->frac);
    updatelitterproperties(stand);

    hetres=littersom(stand,gtemp_soil,&n2o_nit,climate.temp);

    /* update soil and litter properties to account for all changes from littersom */
    pedotransfer(stand,NULL,NULL,stand->frac);
    updatelitterproperties(stand);

#ifdef DEBUG_N
  printf("AFTER_LITTERSOM: ");
  for(l=0;l<NSOILLAYER-1;l++)
     printf("%g ",stand->soil.NO3[l]);
  printf("\n");
#endif
    cell->output.mn2o_nit+=n2o_nit*stand->frac;
    cell->output.mrh+=hetres.carbon*stand->frac;
    if (withdailyoutput)
    {
      switch(stand->type->landusetype)
      {
        case GRASSLAND:
          if (cell->output.daily.cft == C3_PERENNIAL_GRASS)
          {
            cell->output.daily.rh  += hetres.carbon;
            cell->output.daily.swe += stand->soil.snowpack;
            cell->output.daily.n2o_nit += n2o_nit;
          }
          break;
        case AGRICULTURE:
          data = stand->data;
          foreachpft(pft,p,&stand->pftlist)
            if (pft->par->id == cell->output.daily.cft && data->irrigation == cell->output.daily.irrigation)
            { 
              cell->output.daily.rh  = hetres.carbon;
              cell->output.daily.swe = stand->soil.snowpack;
              cell->output.daily.n2o_nit = n2o_nit;
            }
          break;
        case SETASIDE_RF:
          data = stand->data;
          if (data->irrigation == cell->output.daily.irrigation)
          {
            cell->output.daily.rh = hetres.carbon;
            cell->output.daily.swe = stand->soil.snowpack;
            cell->output.daily.n2o_nit = n2o_nit;
          }
          break;
        case SETASIDE_IR:
          data = stand->data;
          if (data->irrigation == cell->output.daily.irrigation)
          {
            cell->output.daily.rh = hetres.carbon;
            cell->output.daily.swe = stand->soil.snowpack;
            cell->output.daily.n2o_nit = n2o_nit;
          }
          break;
        case NATURAL:
          if (cell->output.daily.cft == ALLNATURAL) 
          {
            cell->output.daily.rh  += hetres.carbon;
            cell->output.daily.swe += stand->soil.snowpack;
            cell->output.daily.n2o_nit += n2o_nit;
          }
          break;
      } /* of switch() */
    }

    if(withdailyoutput && stand->type->landusetype==NATURAL)
    {
      for(l=0;l<NTILLLAYER;l++)
      {
        cell->output.daily.whc+=stand->soil.whc[l]*soildepth[l]/layerbound[NTILLLAYER-1];
        cell->output.daily.wsat+=stand->soil.wsat[l]*soildepth[l]/layerbound[NTILLLAYER-1];
        cell->output.daily.ksat+=stand->soil.Ks[l]*soildepth[l]/layerbound[NTILLLAYER-1];
      }
    }


    cell->output.msnowrunoff+=snowrunoff;
    cell->output.mmelt+=melt*stand->frac;

    if(config->fire==FIRE && climate.temp>0)
      stand->fire_sum+=fire_sum(&stand->soil.litter,stand->soil.w[0]);

    if(config->with_nitrogen & !param.no_ndeposition)
    {
      if(stand->soil.par->type==ROCK)
      {
        stand->cell->output.mn_leaching+=climate.nh4deposition*stand->frac;
        stand->cell->output.mn_leaching+=climate.no3deposition*stand->frac;
      }
      else
      {
        /*adding daily N deposition to upper soil layer*/
        stand->soil.NH4[0]+=climate.nh4deposition;
        stand->soil.NO3[0]+=climate.no3deposition;
      }
      cell->balance.n_influx+=(climate.nh4deposition+climate.no3deposition)*stand->frac;
#ifdef DEBUG_N
      printf("BEFORE_STRESS[%s], day %d: ",stand->type->name,day);
      for(l=0;l<NSOILLAYER-1;l++)
        printf("%g ",stand->soil.NO3[l]);
      printf("\n");
#endif
#ifdef DEBUG_N
      printf("AFTER_STRESS: ");
      for(l=0;l<NSOILLAYER-1;l++)
        printf("%g ",stand->soil.NO3[l]);
      printf("\n");
#endif

    } /* of if(config->with_nitrogen) */

    gp_stand=gp_sum(&stand->pftlist,co2,climate.temp,par,daylength,
      &gp_stand_leafon,gp_pft,&fpc_total_stand,config->laimax_interpolate);
    if(config->with_nitrogen)
    {
      bnf=biologicalnfixation(stand);
      stand->soil.NH4[0]+=bnf;
      cell->output.mbnf+=bnf*stand->frac;
    //printf("bnf=%g aetp_mean=%g NH4=%g NO3=%g nh4deposition=%g no3deposition=%g\n",bnf,stand->cell->climbuf.aetp_mean,stand->soil.NH4[0],stand->soil.NO3[0],climate.nh4deposition,climate.no3deposition);
    }
#ifdef DEBUG_N
    printf("BEFORE_DAILY[%s], day %d: ",stand->type->name,day);
    for(l=0;l<NSOILLAYER-1;l++)
      printf("%g ",stand->soil.NO3[l]);
    printf("\n");
#endif
    runoff=daily_stand(stand,co2,climate,day,daylength,gp_pft,
                       gtemp_air,gtemp_soil[0],gp_stand,gp_stand_leafon,pet,par,
                       melt,npft,ncft,year,withdailyoutput,intercrop,config);
#ifdef DEBUG_N
    printf("AFTER_DAILY[%s], day %d: ",stand->type->name,day);
    for(l=0;l<NSOILLAYER-1;l++)
      printf("%g ",stand->soil.NO3[l]);
    printf("\n");
#endif
    if(config->with_nitrogen)
    {
      denitrification(stand,&cell->output);
      nh3=volatilization(stand->soil.NH4[0],climate.windspeed,climate.temp,
                         length,cell->soilph);
      if(nh3>stand->soil.NH4[0])
        nh3=stand->soil.NH4[0]; 
      stand->soil.NH4[0]-=nh3;
      cell->output.mn_volatilization+=nh3*stand->frac;
      cell->balance.n_outflux+=nh3*stand->frac;
    }
    if (withdailyoutput)
    {
      switch (stand->type->landusetype)
      {
      case GRASSLAND:
        if (cell->output.daily.cft == C3_PERENNIAL_GRASS)
        {
          cell->output.daily.rh += hetres.carbon;
          cell->output.daily.nmin += hetres.nitrogen;
          cell->output.daily.swe += stand->soil.snowpack;
          cell->output.daily.nimmo += n_immo;
        }
        break;
      case AGRICULTURE:
        data = stand->data;
        foreachpft(pft, p, &stand->pftlist)
          if (pft->par->id == cell->output.daily.cft && data->irrigation == cell->output.daily.irrigation)
          {
            cell->output.daily.rh = hetres.carbon;
            cell->output.daily.nmin = hetres.nitrogen;
            cell->output.daily.swe = stand->soil.snowpack;
            cell->output.daily.nimmo = n_immo;
          }
        break;
      case SETASIDE_RF:
        data = stand->data;
        if (data->irrigation == cell->output.daily.irrigation)
        {
          cell->output.daily.rh = hetres.carbon;
          cell->output.daily.nmin = hetres.nitrogen;
          cell->output.daily.swe = stand->soil.snowpack;
          cell->output.daily.nimmo = n_immo;
        }
        break;
      case SETASIDE_IR:
        data = stand->data;
        if (data->irrigation == cell->output.daily.irrigation)
        {
          cell->output.daily.rh = hetres.carbon;
          cell->output.daily.nmin = hetres.nitrogen;
          cell->output.daily.swe = stand->soil.snowpack;
          cell->output.daily.nimmo = n_immo;
        }
        break;
      case NATURAL:
        if (cell->output.daily.cft == ALLNATURAL)
        {
          cell->output.daily.rh += hetres.carbon;
          cell->output.daily.nmin += hetres.nitrogen;
          cell->output.daily.swe += stand->soil.snowpack;
          cell->output.daily.nimmo += n_immo;
        }
        break;
     } /* of switch() */
   }
    cell->discharge.drunoff+=runoff*stand->frac;
    climate.prec=prec_save;
  } /* of foreachstand */

#ifdef COUPLING_WITH_FMS
  if (cell->lakefrac > 0) laketemp(cell, &climate);
#endif

  /*cell->mdemand-=cell->discharge.act_irrig_amount;*/
  if(config->river_routing)
  {
    /* reservoir waterbalance */
    if(cell->ml.dam)
    {
      cell->ml.resdata->dmass+=climate.prec*cell->coord.area*cell->ml.reservoirfrac;
      cell->output.mprec_res+=climate.prec*cell->ml.reservoirfrac;
      if(cell->ml.resdata->dmass>cell->ml.resdata->reservoir.capacity)
      {
        cell->discharge.dmass_lake+=cell->ml.resdata->dmass-cell->ml.resdata->reservoir.capacity;
        cell->ml.resdata->dmass=cell->ml.resdata->reservoir.capacity;
      }
      cell->output.mevap_res+=min(cell->ml.resdata->dmass/cell->coord.area,pet*param.ALPHAM*cell->ml.reservoirfrac);
      cell->ml.resdata->dmass=max(cell->ml.resdata->dmass-pet*param.ALPHAM*cell->coord.area*cell->ml.reservoirfrac,0.0);
      cell->output.mres_storage+=cell->ml.resdata->dmass;
    }

    /* lake waterbalance */
    cell->discharge.dmass_lake+=climate.prec*cell->coord.area*cell->lakefrac;
    cell->output.input_lake+=climate.prec*cell->coord.area*cell->lakefrac;

    cell->output.mevap_lake+=min(cell->discharge.dmass_lake/cell->coord.area,pet*param.ALPHAM*cell->lakefrac);
    cell->discharge.dmass_lake=max(cell->discharge.dmass_lake-pet*param.ALPHAM*cell->coord.area*cell->lakefrac,0.0);

  } /* of 'if(river_routing)' */

  cell->output.mswe+=cell->output.daily.swe;

  killstand(cell,config->pftpar,config->with_tillage,npft,intercrop,year);
  cell->output.mrunoff += cell->discharge.drunoff;
  cell->output.daily.runoff += cell->discharge.drunoff;

#ifdef SAFE
  check_stand_fracs(cell->standlist,cell->lakefrac+cell->ml.reservoirfrac);
#endif
  /* Establishment fluxes are area weighted in subroutines */
  cell->output.flux_estab.carbon+=flux_estab.carbon;
  cell->output.flux_estab.nitrogen+=flux_estab.nitrogen;
  cell->output.dcflux-=flux_estab.carbon;
  free(gp_pft);
} /* of 'update_daily' */
