/***************************************************************************/
/**                                                                       **/
/**               u  p  d  a  t  e  _  a  n  n  u  a  l  .  c             **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran/C++ version     **/
/**                                                                       **/
/**     Function performs necessary updates after iteration over one      **/
/**     year                                                              **/
/**                                                                       **/
/**     written by Werner von Bloh, Sibyll Schaphoff                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     See file AUTHORS for list of authors contributing to this code    **/
/**                                                                       **/
/**     Last change: $Date:: 2016-04-25 16:34:30 +0200 (Mon, 25 Apr 201#$ **/
/**     By         : $Author:: bloh                            $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"
#include "grass.h"

void update_annual(Cell *cell,           /* Pointer to cell */
                   int npft,             /* number of natural pfts */
                   int ncft,             /* number of crop pfts */
                   Real popdens,         /* population density (capita/km2) */
                   int year,             /* simulation year (AD) */
                   Bool intercrop,       /* intercropping (TRUE/FALSE) */
                   const Config *config  /* LPJ configuration */
                  )
{
  int s,p;
  Pft *pft;
  Stand *stand;
  Stocks litter_neg;
  if(cell->ml.dam)
    update_reservoir_annual(cell);
  annual_climbuf(&cell->climbuf,cell->output.aevap+cell->output.atransp);
  if(config->sdate_option==NO_FIXED_SDATE ||
    (config->sdate_option==FIXED_SDATE && year<=param.sdate_fixyear)||
    (config->sdate_option==PRESCRIBED_SDATE && year<=param.sdate_fixyear))
    cell->climbuf.atemp_mean20_fix=cell->climbuf.atemp_mean20;

  /* count number of years without harvest
   * and set 20-year mean to zero if no harvest
   * occurred within the last 10 years
   */

    if((year<config->firstyear && config->sdate_option!=PRESCRIBED_SDATE) ||
       config->sdate_option==NO_FIXED_SDATE)
      update_cropdates(cell->ml.cropdates,ncft);

  foreachstand(stand,s,cell->standlist)
  {
    stand->soil.mean_maxthaw=(stand->soil.mean_maxthaw-stand->soil.mean_maxthaw/CLIMBUFSIZE)+stand->soil.maxthaw_depth/CLIMBUFSIZE;
    if(!config->with_nitrogen)
      foreachpft(pft,p,&stand->pftlist)
        pft->vscal=NDAYYEAR;
    if(annual_stand(stand,npft,ncft,popdens,year,intercrop,config))
    {
      /* stand has to be deleted */
      delstand(cell->standlist,s);
      s--; /* adjust stand index */
    }
  } /* of foreachstand */
  foreachstand(stand,s,cell->standlist)
  {
    litter_neg=checklitter(&stand->soil.litter);
    cell->output.negc_fluxes+=litter_neg.carbon*stand->frac;
    cell->output.negn_fluxes+=litter_neg.nitrogen*stand->frac;
    stand->cell->output.soil_storage+=soilwater(&stand->soil)*stand->frac*stand->cell->coord.area;
  }
  cell->output.fpc[0] = 1-cell->ml.cropfrac_rf-cell->ml.cropfrac_ir-cell->lakefrac-cell->ml.reservoirfrac;
#ifdef IMAGE
  cell->output.prod_turnover=product_turnover(cell->ml.image_data);
  cell->output.product_pool_fast=cell->ml.image_data->timber.fast;
  cell->output.product_pool_slow=cell->ml.image_data->timber.slow;
#endif
} /* of 'update_annual' */
