/***************************************************************************/
/**                                                                       **/
/**             f  w  r  i  t  e  c  e  l  l  .  c                        **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran/C++ version     **/
/**                                                                       **/
/**     Function writes cell array and calculates index vector            **/
/**     Index vector is used at reading to seek to specific LPJ cells     **/
/**                                                                       **/
/**     written by Werner von Bloh, Sibyll Schaphoff                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: $Date:: 2016-04-25 16:34:30 +0200 (Mon, 25 Apr 201#$ **/
/**     By         : $Author:: bloh                            $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"

int fwritecell(FILE *file,        /* File pointer of binary file */
               long long index[], /* index vector to be calculated */
               const Cell grid[], /* cell data array */
               int ncell,         /* number of cells */
               int ncft,          /* number of crop PFTs */
               int npft,          /* number of PFTs */
               int sdate_option,  /* sowing date option (0-2) */
               Bool river_routing /* river routing (TRUE/FALSE) */
              )                   /* returns number of cells written */
{
  int cell;
  Byte b;
  for(cell=0;cell<ncell;cell++)
  {
    if(index!=NULL)
      index[cell]=ftell(file); /* store actual position in index vector */
    b=(Byte)grid[cell].skip;
    fwrite(&b,sizeof(b),1,file);
    if(river_routing)
    {
      if(fwrite(&grid[cell].discharge.dmass_lake,sizeof(Real),1,file)!=1)
        break;
      if(fwrite(&grid[cell].discharge.dfout,sizeof(Real),1,file)!=1)
        break;
      if(fwrite(&grid[cell].discharge.dmass_river,sizeof(Real),1,file)!=1)
        break;
      if(fwrite(&grid[cell].discharge.dmass_sum,sizeof(Real),1,file)!=1)
        break;
#ifdef COUPLING_WITH_FMS
      if(fwrite(&grid[cell].laketemp,sizeof(Real),1,file)!=1)
        break;
#endif
      if(fwritequeue(file,grid[cell].discharge.queue))
        break;
      b=(Byte)grid[cell].ml.dam;
      fwrite(&b,sizeof(b),1,file);
      if(grid[cell].ml.dam)
      {
        if(fwriteresdata(file,grid+cell))
          break;
      }
    }
    if(!grid[cell].skip)
    {
      if(fwrite(grid[cell].balance.estab_storage_tree,sizeof(Stocks),2,file)!=2)
    	break;
      if(fwrite(grid[cell].balance.estab_storage_grass,sizeof(Stocks),2,file)!=2)
    	break;
      if(fwrite(&grid[cell].discharge.waterdeficit,sizeof(Real),1,file)!=1)
        break;
      if(fwritestandlist(file,grid[cell].standlist,npft+ncft)!=
         grid[cell].standlist->n)
        break;
      if(fwrite(&grid[cell].ml.cropfrac_rf,sizeof(Real),1,file)!=1)
        break;
      if(fwrite(&grid[cell].ml.cropfrac_ir,sizeof(Real),1,file)!=1)
        break;
      if(fwriteclimbuf(file,&grid[cell].climbuf))
        break;
      if(fwritecropdates(file,grid[cell].ml.cropdates,ncft))
        break;
      if(sdate_option>NO_FIXED_SDATE)
      {
        if(fwrite(grid[cell].ml.sdate_fixed,sizeof(int),2*ncft,file)!=2*ncft)
          break;
      }
      if(fwrite(grid[cell].ml.sowing_month,sizeof(int),2*ncft,file)!=2*ncft)
        break;
      if(fwrite(grid[cell].ml.gs,sizeof(int),2*ncft,file)!=2*ncft)
        break;
      if(grid[cell].ml.landfrac!=NULL)
        fwritelandfrac(file,grid[cell].ml.landfrac,ncft);
      if(grid[cell].ml.fertilizer_nr!=NULL)
        fwritelandfrac(file,grid[cell].ml.fertilizer_nr,ncft);
    }
  } /* of 'for(cell=...)' */
  return cell;
} /* of 'fwritecell' */
