/***************************************************************************/
/**                                                                       **/
/**                f  p  r  i  n  t  f  l  u  x  .  c                     **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran/C++ version     **/
/**                                                                       **/
/**     Function prints global carbon and water fluxes.                   **/
/**                                                                       **/
/**     written by Werner von Bloh, Sibyll Schaphoff                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: $Date:: 2015-11-11 13:03:35 +0100 (Wed, 11 Nov 201#$ **/
/**     By         : $Author:: bloh                            $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"

#define LINES_PER_HEADER 25 

void fprintflux(FILE *file,          /* Output file pointer */
                Flux flux,           /* Carbon and water fluxes */
                Real cflux_total,    /* Total carbon flux  (gC) */
                int year,            /* Simulation year (AD) */
                const Config *config /* LPJ configuration */
               )
{
  int i,tabs;
  if((year<config->firstyear &&
     (year-config->firstyear+config->nspinup) % LINES_PER_HEADER==0) || 
     (year>=config->firstyear && (year-config->firstyear) % LINES_PER_HEADER==0))
  {
    /* print header */
    tabs=(config->fire) ? 4 : 3;
    if(config->withlanduse!=NO_LANDUSE)
      tabs++;
    fputs("\n       ",file);
    frepeatch(file,' ',(tabs*8-16)/2-1);
    fputs("Carbon flux (GtC)",file);
    frepeatch(file,' ',(tabs*8-16)/2);
    if(config->river_routing)
      frepeatch(file,' ',(config->withlanduse==NO_LANDUSE) ? 16 : 20);
    else
      frepeatch(file,' ',(config->withlanduse==NO_LANDUSE) ? 8 : 12);
    fputs("Water (km3)",file);
    if(config->with_nitrogen)
    {
      if(config->river_routing)
        frepeatch(file,' ',(config->withlanduse==NO_LANDUSE) ? 4 : 8);
      fputs("                Nitrogen (TgN)",file);
    }
    fputs("\n       ",file);
    frepeatch(file,'-',tabs*8-1);
    if(config->withlanduse!=NO_LANDUSE)
      fputs(" ----------------------------------",file);
    else
      fputs(" --------------------------",file);
    if(config->river_routing)
      fputs("-----------",file);
    if(config->with_nitrogen)
      fputs(" --------------------------------",file);
    fputc('\n',file);
    if(year<config->firstyear)
      fputs("Spinup ",file);
    else
      fputs("Year   ",file);
    fputs("NEP     estab  ",file);
    if(config->fire)
      fputs(" fire   ",file);
    if(config->withlanduse!=NO_LANDUSE)
      fputs(" harvest",file);
    fputs(" total  ",file);
    fputs(" transp     evap    interc ",file);
    if(config->withlanduse!=NO_LANDUSE)
      fputs(" wd     ",file);
    if(config->river_routing)
      fputs(" discharge ",file);
    if(config->with_nitrogen)
      fputs(" nuptake ndemand  nlosses ninflux",file);
    fputc('\n',file);
    fputs("------",file);
    for(i=0;i<tabs;i++)
      fputs(" -------",file);
    fputs(" ---------- ------- -------",file);
    if(config->withlanduse!=NO_LANDUSE)
      fputs(" -------",file);
    if(config->river_routing)
      fputs(" ----------",file);
    if(config->with_nitrogen)
      fputs(" ------- -------- ------- -------",file);
    fputc('\n',file);
  }
  /* print data */
  fprintf(file,"%6d %7.3f %7.3f",year,flux.nep*1e-15,flux.estab*1e-15);
  if(config->fire)
    fprintf(file," %7.3f",flux.fire*1e-15);
  if(config->withlanduse!=NO_LANDUSE)
    fprintf(file," %7.3f",flux.harvest*1e-15); 
  fprintf(file," %7.3f",cflux_total*1e-15);
  fprintf(file," %10.1f %7.1f %7.1f",
          flux.transp*1e-12,flux.evap*1e-12,flux.interc*1e-12);
  if(config->withlanduse!=NO_LANDUSE)
    fprintf(file," %7.1f",flux.wd*1e-12);
  if(config->river_routing)
    fprintf(file," %10.1f",flux.discharge*1e-12);
  if(config->with_nitrogen)
    fprintf(file," %7.1f %8.1f %7.1f %7.1f",flux.n_uptake*1e-12,flux.n_demand*1e-12,flux.n_outflux*1e-12,flux.n_influx*1e-12);
  fputc('\n',file);
} /* of 'fprintflux' */
