/***************************************************************************/
/**                                                                       **/
/**           t  u  r  n  o  v  e  r  _  g  r  a  s  s  .  c              **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran/C++ version     **/
/**                                                                       **/
/**     written by Werner von Bloh, Sibyll Schaphoff                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: $Date:: 2017-02-01 09:28:07 +0100 (Wed, 01 Feb 201#$ **/
/**     By         : $Author:: bloh                            $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"
#include "grass.h"

/*
 *
 *  Function reproduction + turnover
 *
 *  Turnover of PFT-specific fraction from each living C pool
 *  Leaf and root C transferred to litter, sapwood C to heartwood
 *
 */

Stocks turnover_grass(Litter *litter, /* Litter pool */
                    Pft *pft,       /* Pointer to PFT variables */
                    Bool new_phenology,
                    Real fraction   /* fraction of annual turnover (0..1) */
                   )                /* returns turnover (gC/m2) */
{
  Pftgrass *grass;
  const Pftgrasspar *grasspar;
  Real reprod;
  Grassphys gturn;
  grass=pft->data;
  grasspar=getpftpar(pft,data);
  /* reproduction */
  if(pft->bm_inc.carbon>0)
  {
    reprod=pft->bm_inc.carbon*grasspar->reprod_cost*fraction;
    litter->ag[pft->litter].trait.leaf.carbon+=reprod;
    pft->bm_inc.carbon-= reprod;
  }
  if(pft->bm_inc.nitrogen>0)
  {
    reprod=pft->bm_inc.nitrogen*grasspar->reprod_cost*fraction;
    litter->ag[pft->litter].trait.leaf.nitrogen+=reprod;
    pft->bm_inc.nitrogen-= reprod;
#ifdef NEW_GRASS
    //pft->bm_inc_har.nitrogen -= reprod;
#endif
  }
  /* turnover */
  if (pft->stand->type->landusetype==NATURAL)
  {
    gturn.root.carbon=grass->turn.root.carbon;
    gturn.leaf.carbon=grass->turn.leaf.carbon;
    gturn.root.nitrogen=grass->turn.root.nitrogen;
    gturn.leaf.nitrogen=grass->turn.leaf.nitrogen;
    grass->turn.root.carbon=grass->turn.leaf.carbon=0.0;
    grass->turn.root.nitrogen=grass->turn.leaf.nitrogen=0.0;
    if(new_phenology)
      litter->ag[pft->litter].trait.leaf.nitrogen-=gturn.leaf.nitrogen*pft->nind*(1-pft->par->fn_turnover);
  }
  else
  {
    gturn.root.carbon=grass->ind.root.carbon*grasspar->turnover.root*fraction;
    gturn.leaf.carbon=grass->ind.leaf.carbon*grasspar->turnover.leaf*fraction;
    gturn.root.nitrogen=grass->ind.root.nitrogen*grasspar->turnover.root*fraction;
    gturn.leaf.nitrogen=grass->ind.leaf.nitrogen*grasspar->turnover.leaf*fraction;
    litter->ag[pft->litter].trait.leaf.carbon+=gturn.leaf.carbon*pft->nind;
    litter->ag[pft->litter].trait.leaf.nitrogen+=gturn.leaf.nitrogen*pft->nind*pft->par->fn_turnover;
    update_fbd_grass(litter,pft->par->fuelbulkdensity,gturn.leaf.carbon*pft->nind);
    litter->bg[pft->litter].carbon+=gturn.root.carbon*pft->nind;
    litter->bg[pft->litter].nitrogen+=gturn.root.nitrogen*pft->nind*pft->par->fn_turnover;
  }
  grass->ind.leaf.carbon-= gturn.leaf.carbon;
  grass->ind.root.carbon-= gturn.root.carbon;
  grass->ind.leaf.nitrogen-= gturn.leaf.nitrogen;
  grass->ind.root.nitrogen-= gturn.root.nitrogen;
  pft->bm_inc.nitrogen+= (gturn.root.nitrogen+gturn.leaf.nitrogen)*pft->nind*(1-pft->par->fn_turnover);
  gturn.leaf.carbon+=gturn.root.carbon;  
  gturn.leaf.nitrogen+=gturn.root.nitrogen;  
  /* turnover of excess carbon as root exudates */
  litter->bg[pft->litter].carbon+=grass->excess_carbon*pft->nind*grasspar->turnover.root;
  grass->excess_carbon-=grass->excess_carbon*grasspar->turnover.root;
  return gturn.leaf;
} /* of 'turnover_grass' */
