/***************************************************************************/
/**                                                                       **/
/**                    n  p  p  _  c  r  o  p  .  c                       **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran version         **/
/**                                                                       **/
/**     written by Werner von Bloh, Sibyll Schaphoff                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: 25.01.2006 by S.Jachner                              **/
/**     Last change: $Date:: 2018-05-16 15:15:17 +0200 (Wed, 16 May 201#$ **/
/**     By         : $Author:: herzfeld                        $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"
#include "crop.h"
#include "agriculture.h"

/*
- called in npp_crop()
- calculation of the vegetation growth:
  -> calculation of carbon biomass (total plant growth) 
  -> calculation of carbon mass root (root growth)
  -> calculation of carbon mass leaf (leaf growth)
  -> calculation of carbon mass so (storage organs growth)
  -> calculation of carbon mass pool (additional pool stem + reserve)
*/

Real npp_crop(Pft *pft, /* PFT variables */
              Real gtemp_air, /* value of air temperature response function */
              Real gtemp_soil, /* value of soil temperature response function */
              Real assim,  /* assimilation (gC/m2) */
              Bool *negbm, /* on return: biomass is negative */
              Real wdf,     /* water deficit fraction */
              Bool with_nitrogen,    /* with nitrogen (TRUE/FALSE */
              Daily_outputs *output  /* daily output structure */
             ) /* returns net primary productivity (gC/m2) */
{
  Pftcrop *crop;
  const Pftcroppar *par;
  Real npp;
  Real rosoresp,presp,gresp;
#ifdef NRESP
  Real cn_root,cn_so,cn_pool;
#endif
  Irrigation *data;
  data=pft->stand->data;
  crop=pft->data;
  par=pft->par->data;
#ifdef NRESP
  if(with_nitrogen && crop->ind.root.carbon>epsilon)
    cn_root=crop->ind.root.nitrogen/crop->ind.root.carbon;
  else
    cn_root=par->cn_ratio.root;
  if(with_nitrogen && crop->ind.so.carbon>epsilon)
    cn_so=crop->ind.so.nitrogen/crop->ind.so.carbon;
  else
    cn_so=par->cn_ratio.so;
  if(with_nitrogen && crop->ind.pool.carbon>epsilon)
    cn_pool=crop->ind.pool.nitrogen/crop->ind.pool.carbon;
  else
    cn_pool=par->cn_ratio.pool;

  rosoresp=crop->ind.root.carbon*pft->par->respcoeff*param.k*cn_root*gtemp_soil
           +crop->ind.so.carbon*pft->par->respcoeff*param.k*cn_so*gtemp_air;
  presp=crop->ind.pool.carbon*pft->par->respcoeff*param.k*cn_pool*gtemp_air;
#else
  rosoresp=crop->ind.root.carbon*pft->par->respcoeff*param.k*par->cn_ratio.root*gtemp_soil
           +crop->ind.so.carbon*pft->par->respcoeff*param.k*par->cn_ratio.so*gtemp_air;
  presp = crop->ind.pool.carbon*pft->par->respcoeff*param.k*par->cn_ratio.pool*gtemp_air;
#endif
  /* pools can't be negative any more as LAI growth and SO allocation is limited by NPP now */
  /* presp=(crop->ind.pool<0.0) ? 0.0 : crop->ind.pool*pft->par->respcoeff*param.k*par->cn_ratio.pool*gtemp_air; */
  gresp=(assim-rosoresp-presp)*param.r_growth;
  if(gresp<0.0)
    gresp=0.0;
  npp=assim-rosoresp-presp-gresp;
  if((pft->bm_inc.carbon+npp <=0.0001) ||
      (crop->lai-crop->lai_nppdeficit<=0 && !crop->senescence))
  {
    *negbm=TRUE;
    crop->ind.pool.carbon+=npp;
    pft->bm_inc.carbon+=npp; 
  } 
  else 
    allocation_daily_crop(pft,npp,wdf,with_nitrogen,output);
  if(output!=NULL && output->cft==pft->par->id &&
     output->irrigation==data->irrigation)
  {
    output->rroot=crop->ind.root.carbon*pft->par->respcoeff*param.k*par->cn_ratio.root*gtemp_soil;
    output->rso=crop->ind.so.carbon*pft->par->respcoeff*param.k*par->cn_ratio.so*gtemp_air;
    output->rpool=presp;
    output->gresp=gresp;
    output->npp=npp;
    output->cleaf=crop->ind.leaf.carbon;
    output->croot=crop->ind.root.carbon;
    output->cso=crop->ind.so.carbon;
    output->cpool=crop->ind.pool.carbon;
    output->nleaf=crop->ind.leaf.nitrogen;
    output->npool=crop->ind.pool.nitrogen;
    output->nso=crop->ind.so.nitrogen;
    output->nroot=crop->ind.root.nitrogen;

    output->wdf=wdf;
    output->wscal=pft->wscal;
  }
  return npp;
} /* of 'npp_crop' */

/*
- called in update_daily():
  -> cell->output.mnpp[month]+=npp(pft,pft->phen,gtemp_air,gtemp_soil,(gpp-rd),
				    &negbm);
- this function returns the computed daily npp
  -> calculation of roresp (root respiration)
  -> calculation of soresp (storage organ respiration)
  -> calculation of presp (pool (stem + reserve) respiration)
  -> calculation of gresp (growth respiration)
  -> calculation of daily npp (net primary production)

- calls the function allocation_daily_crop() if npp>0 und carbon mass pool+npp>0
  -> allocation_daily_crop() needs the determined npp of npp_crop() 
- sets negbm to TRUE if the biomass gets negative
  -> cropstand will be deleted in update_daily()
*/
