/***************************************************************************/
/**                                                                       **/
/**             f  s  c  a  n  p  f  t  _  c  r  o  p  .  c               **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran version         **/
/**                                                                       **/
/**     written by Werner von Bloh, Sibyll Schaphoff                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: 03.05.2007   by S.Jachner, M.Gumpenberger            **/
/**     Last change: $Date:: 2017-01-25 11:38:36 +0100 (Wed, 25 Jan 201#$ **/
/**     By         : $Author:: cmueller                        $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"
#include "crop.h"
#include "grass.h"

#define fscanpftreal(isout,file,var,pft,name) \
  if(fscanreal(file,var,name,isout)) \
  { \
    if(isout)\
    fprintf(stderr,"ERROR110: Cannot read CFT '%s' in %s().\n",pft,__FUNCTION__); \
    return TRUE; \
  }
#define fscanpftint(isout,file,var,pft,name) \
  if(fscanint(file,var,name,isout)) \
  { \
    if(isout)\
    fprintf(stderr,"ERROR110: Cannot read CFT '%s' in %s().\n",pft,__FUNCTION__); \
    return TRUE; \
  }
#define fscanpftlimit(isout,file,var,pft,name) \
  if(fscanlimit(file,var,isout)) \
  { \
    if(isout)\
    fprintf(stderr,"ERROR112: Cannot read limit '%s' of CFT '%s' in %s().\n",name,pft,__FUNCTION__); \
    return TRUE; \
  }

#define fscancropdate2(isout,file,var,pft,name)\
  if(fscancropdate(file,var,isout))\
  {\
    if(isout)\
    fprintf(stderr,"ERROR112: Cannot read '%s' of CFT '%s' in %s().\n",name,pft,__FUNCTION__); \
    return TRUE; \
  }

static Bool fscancropdate(FILE *file,Initdate *initdate,Bool isout)
{
  if(fscanint(file,&initdate->sdatenh,"date northern hemisphere",isout))
    return TRUE;
  if(fscanint(file,&initdate->sdatesh,"date southern hemisphere",isout))
    return TRUE;
  return FALSE;
} /* of 'fscancropdate' */

#define fscancropphys2(isout,file,var,pft,name)\
  if(fscancropphys(file,var,isout))\
  {\
    if(isout)\
    fprintf(stderr,"ERROR114: Cannot read cropphys '%s' of CFT '%s' in %s().\n",name,pft,__FUNCTION__); \
    return TRUE; \
  }

#define fscancropratio2(isout,file,var,pft,name)\
  if(fscancropratio(file,var,isout))\
  {\
    if(isout)\
    fprintf(stderr,"ERROR114: Cannot read cropphys '%s' of CFT '%s' in %s().\n",name,pft,__FUNCTION__); \
    return TRUE; \
  }

static Bool fscancropphys(FILE *file,Cropphys *phys,Bool isout)
{
  if(fscanreal(file,&phys->leaf,"leaf",isout))
    return TRUE;
  if(fscanreal(file,&phys->root,"root",isout))
    return TRUE;
  if(fscanreal(file,&phys->so,"so",isout))
    return TRUE;
  if(fscanreal(file,&phys->pool,"pool",isout))
    return TRUE;
  if(phys->leaf<=0 || phys->root<=0 || phys->so<=0 || phys->pool<=0)
    return TRUE;
  return FALSE;
} /* of 'fscancropphys' */

static Bool fscancropratio(FILE *file,Cropratio *ratio,Bool isout)
{
  if(fscanreal(file,&ratio->root,"root",isout))
    return TRUE;
  if(fscanreal(file,&ratio->so,"so",isout))
    return TRUE;
  if(fscanreal(file,&ratio->pool,"pool",isout))
    return TRUE;
  if(ratio->root<=0 || ratio->so<=0 || ratio->pool<=0)
    return TRUE;
  return FALSE;
} /* of 'fscancropratio' */

Bool fscanpft_crop(FILE *file,           /* file pointer */
                   Pftpar *pft,          /* Pointer to Pftpar array */
                   Bool isout            /* error output is generated */
                  )                      /* returns FALSE for success */
{
  Pftcroppar *crop;
  pft->newpft=new_crop;
  pft->npp=NULL;
  pft->leaf_phenology=NULL;
  pft->fwrite=fwrite_crop;
  pft->fprint=fprint_crop;
  pft->fread=fread_crop;
  pft->litter_update=litter_update_crop;
  pft->actual_lai=actual_lai_crop;
  pft->fpar=fpar_crop;
  pft->alphaa_manage=alphaa_crop;
  pft->free=free_crop;
  pft->vegc_sum=vegc_sum_crop;
  pft->vegn_sum=vegn_sum_crop;
  pft->wdf=wdf_crop;
  pft->fprintpar=fprintpar_crop;
  pft->livefuel_consumption=NULL;
  pft->annual=NULL;
  pft->albedo=albedo_grass;
  pft->nuptake=nuptake_crop;
  pft->ndemand=ndemand_crop;
  pft->vmaxlimit=vmaxlimit_crop;
  /* unused PFT parameters for new phenology */
  pft->tmin.sl = pft->tmin.base = pft->tmin.tau = pft->tmax.sl = pft->tmax.base = pft->tmax.tau = 0;
  pft->light.sl = pft->light.base =pft->light.tau = pft->wscal.sl = pft->wscal.base = pft->wscal.tau=0;
  crop=new(Pftcroppar);
  check(crop);
  pft->data=crop;
  fscanpftint(isout,file,&crop->calcmethod_sdate,pft->name,
              "calcmethod_sdate");
  fscancropdate2(isout,file,&crop->initdate,pft->name,"init date");
  fscanpftint(isout,file,&crop->hlimit,pft->name,
              "max length of crop cycle");
  fscanpftint(isout,file,&crop->fallow_days,pft->name,
              "fallow period after harvest");
  fscanpftreal(isout,file,&crop->temp_fall,pft->name,
               "temperature threshold for fall");
  fscanpftreal(isout,file,&crop->temp_spring,pft->name,
               "temperature threshold for spring");
  fscanpftreal(isout,file,&crop->temp_vern,pft->name,
               "temperature threshold for vernalization");
  fscanpftlimit(isout,file,&crop->trg,pft->name,"vernalization temperature");
  fscanpftreal(isout,file,&crop->pvd,pft->name,"number of vernalising days");
  fscanpftreal(isout,file,&crop->psens,pft->name,
               "photoperiod sensitivity");
  fscanpftreal(isout,file,&crop->pb,pft->name,"photoperiod basal");
  fscanpftreal(isout,file,&crop->ps,pft->name,"photoperiod saturated");
  fscanpftlimit(isout,file,&crop->phuw,pft->name,
                "potential heat units winter");
  fscanpftlimit(isout,file,&crop->phus,pft->name,
                "potential heat units summer");
  fscanpftreal(isout,file,&crop->phu_par,pft->name,
               "parameter for determining the variability of phu");
  fscanpftlimit(isout,file,&crop->basetemp,pft->name,"base temperature");
  fscanpftreal(isout,file,&crop->fphuc,pft->name,
               "growing season to the first point");
  fscanpftreal(isout,file,&crop->flaimaxc,pft->name,
               "plant maximal LAI to the first point");
  fscanpftreal(isout,file,&crop->fphuk,pft->name,
               "growing season to the second point");
  fscanpftreal(isout,file,&crop->flaimaxk,pft->name,
               "plant maximal LAI to the second point");
  fscanpftreal(isout,file,&crop->fphusen,pft->name,"growing period");
  fscanpftreal(isout,file,&crop->flaimaxharvest,pft->name,
               "maximal LAI at harvest");
  fscanpftreal(isout,file,&crop->laimax,pft->name,"maximal LAI");
  fscanpftreal(isout,file,&crop->laimin,pft->name,"minimal LAI");
  fscanpftreal(isout,file,&crop->hiopt,pft->name,"optimum harvest index");
  fscanpftreal(isout,file,&crop->himin,pft->name,"minimum harvest index");
  fscanpftreal(isout,file,&crop->shapesenescencenorm,pft->name,
               "shapesenescencenorm");
  fscancropphys2(isout,file,&crop->cn_ratio,pft->name,"cn_ratio");
  fscancropratio2(isout,file,&crop->ratio,pft->name,"ratio");
  crop->cn_ratio.leaf=1/crop->cn_ratio.leaf;
  crop->cn_ratio.root=1/crop->cn_ratio.root;
  crop->cn_ratio.so=1/crop->cn_ratio.so;
  crop->cn_ratio.pool=1/crop->cn_ratio.pool;
  return FALSE;
} /* of 'fscanpft_crop' */

/*
- called in fscanpftpar()
- scans crop specific parameters from pft.par
  -> struct Pftcroppar in crop.h
*/
