/***************************************************************************/
/**                                                                       **/
/**                d  a  i  l  y  c  l  i  m  a  t  e  .  c               **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran/C++ version     **/
/**                                                                       **/
/**     Function dailyclimate gets daily value for temperature,           **/
/**     precipitation, cloudiness and wet days.                           **/
/**     The pointer to the climate data has to be initialized by the      **/
/**     function getclimate for the specified year.                       **/
/**                                                                       **/
/**     written by Werner von Bloh, Sibyll Schaphoff                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: $Date:: 2018-05-16 15:15:17 +0200 (Wed, 16 May 201#$ **/
/**     By         : $Author:: herzfeld                        $          **/
/**                                                                       **/
/***************************************************************************/

#include "lpj.h"

void dailyclimate(Dailyclimate *daily,
                  const Climate *climate, /* climate data pointer */
                  Climbuf *climbuf,
                  int cell,               /* cell index */
                  int day,                /* day (1..365) */
                  int month,              /* month (0..11) */
                  int dayofmonth)
{
  Real tamp;
  int i;
  if(climate->file_temp.fmt!=FMS)
  {
    if(climate->file_temp.isdaily)
    {
      daily->temp=climate->data.temp[cell*NDAYYEAR+day-1];
      climbuf->mtemp+=daily->temp;
    }
    else
      daily->temp=interpolate(getcelltemp(climate,cell),month,dayofmonth);
  }
  else
    climbuf->mtemp+=daily->temp;
  if(climate->data.sun!=NULL)
    daily->sun=(climate->file_cloud.isdaily) ? climate->data.sun[cell*NDAYYEAR+day-1]
                      : interpolate(getcellsun(climate,cell),month,dayofmonth);
  if(climate->data.lwnet!=NULL)
    daily->lwnet=(climate->file_lwnet.isdaily) ? climate->data.lwnet[cell*NDAYYEAR+day-1]
                      : interpolate(getcelllwnet(climate,cell),month,dayofmonth);
  if(climate->data.swdown!=NULL)
    daily->swdown=(climate->file_swdown.isdaily) ? climate->data.swdown[cell*NDAYYEAR+day-1]
                      : interpolate(getcellswdown(climate,cell),month,dayofmonth);
  if(climate->data.wind!=NULL)
  {
    if(climate->file_wind.isdaily)
      daily->windspeed=climate->data.wind[cell*NDAYYEAR+day-1];
    else
      daily->windspeed=interpolate(getcellwind(climate,cell),month,dayofmonth);
  }
  if(climate->data.no3deposition!=NULL)
  {
    if(climate->file_no3deposition.isdaily)
      daily->no3deposition=climate->data.no3deposition[cell*NDAYYEAR+day-1];
    else
      daily->no3deposition=interpolate(getcellno3deposition(climate,cell),month,dayofmonth);
    //  daily->no3deposition+=1;
    //daily->no3deposition=min(0.0000273,daily->no3deposition);
    //daily->no3deposition=min(0.00273,daily->no3deposition);
    if(daily->no3deposition<0)
    {
      fprintf(stderr,"ERROR201: NO3 deposition %g<0 at day %d\n",daily->no3deposition,day);
      daily->no3deposition=0;
    }
  }

  if(climate->data.nh4deposition!=NULL)
  {
    if(climate->file_nh4deposition.isdaily)
      daily->nh4deposition=climate->data.nh4deposition[cell*NDAYYEAR+day-1];
    else
      daily->nh4deposition=interpolate(getcellnh4deposition(climate,cell),month,dayofmonth);
    //  daily->nh4deposition+=1;
    //daily->nh4deposition=min(0.0000273,daily->nh4deposition);
    //daily->nh4deposition=min(0.00273,daily->nh4deposition);
  if(daily->nh4deposition<0)
  {
    fprintf(stderr,"ERROR201: NH4 deposition %g<0 at day %d\n",daily->nh4deposition,day);
    for(i=0;i<NMONTH;i++)
      fprintf(stderr,"%g ",climate->data.nh4deposition[cell*NMONTH+i]);
    fprintf(stderr,"\n");
    daily->nh4deposition=0;
  }
  }
  if(climate->data.tamp!=NULL)
  {
#ifdef TMAX_SPITFIRE
    daily->tmin=(climate->file_tamp.isdaily) ?  climate->data.tamp[cell*NDAYYEAR+day-1] : interpolate(getcelltamp(climate,cell),month,dayofmonth);
    daily->tmax=(climate->file_tmax.isdaily) ?  climate->data.tmax[cell*NDAYYEAR+day-1] : interpolate(getcelltmax(climate,cell),month,dayofmonth);
#else
    tamp=(climate->file_tamp.isdaily) ?  climate->data.tamp[cell*NDAYYEAR+day-1] : interpolate(getcelltamp(climate,cell),month,dayofmonth);
    daily->tmin=daily->temp-tamp*0.5;
    daily->tmax=daily->temp+tamp*0.5;
#endif
  }
  if(climate->data.lightning!=NULL)
  {
    if(climate->file_lightning.isdaily)
      daily->lightning=climate->data.lightning[cell*NDAYYEAR+day-1];
    else
      daily->lightning=interpolate(getcelllightning(climate,cell),month,dayofmonth);
  }
  if(climate->file_prec.fmt!=FMS)
  {
    if(climate->file_prec.isdaily)
    {
      daily->prec=climate->data.prec[cell*NDAYYEAR+day-1];
      climbuf->mprec+=daily->prec;
    }
    else
    {
      daily->prec=(israndomprec(climate)) ? climbuf->dval_prec[dayofmonth+1] :
          interpolate(getcellprec(climate,cell),month,dayofmonth)*
                        ndaymonth1[month];
      daily->prec=(daily->prec>0.000001) ? daily->prec : 0.0;
    }
  }
  else
    climbuf->mprec+=daily->prec;
} /* of 'dailyclimate' */
