/***************************************************************************/
/**                                                                       **/
/**                        t  r  e  e  .  h                               **/
/**                                                                       **/
/**     C implementation of LPJ, derived from the Fortran version         **/
/**                                                                       **/
/**     Header file for tree PFTs                                         **/
/**                                                                       **/
/**     written by Werner von Bloh, Sibyll Schaphoff                      **/
/**     Potsdam Institute for Climate Impact Research                     **/
/**     PO Box 60 12 03                                                   **/
/**     14412 Potsdam/Germany                                             **/
/**                                                                       **/
/**     Last change: $Date:: 2017-02-23 12:38:53 +0100 (Thu, 23 Feb 201#$ **/
/**     By         : $Author:: bloh                            $          **/
/**                                                                       **/
/***************************************************************************/

#ifndef TREE_H /* Already included? */
#define TREE_H

/* Definition of constants */

#define reinickerp 1.6
#define k_latosa 4e3      /* leaf area to sapwood area */
#define wooddens 2e5      /* wood density (gC/m3) */
#define FPC_TREE_MAX 0.95 /* maximum foliage projective cover for trees */

/* Declaration of datatypes */

typedef struct
{
  Real sapwood,root;
} Treeratio;

typedef struct
{
  Real leaf,sapwood,root;
} Treephyspar;

typedef struct
{
  Stocks leaf,sapwood,root;
} Treephys;

typedef struct
{
  Stocks leaf,sapwood,heartwood,root,debt;
} Treephys2;

typedef struct
{
  int leaftype;                     /* par16*/
  Treephyspar turnover;             /* turnover period (years) (9,11,12)*/
  Treephyspar cn_ratio;             /* C:N mass ratio (13-15) */
  Treeratio ratio;                  /* C:N ratio relative to leaf ratio */
  Real crownarea_max;               /* tree maximum crown area (m2) (20)*/
  Treephys2 sapl;                   /* sapling */
  Real sapling_C;                   /* sapling C per (m2)*/
  Real allom1,allom2,allom3,allom4; /* allometry */
  Real aphen_min,aphen_max;
  Real height_max;                  /* maximum height of tree (m) */
  Real reprod_cost;                 /* reproduction cost */
  Real scorchheight_f_param;        /* scorch height (F)    benp */
  Real crownlength;                 /* crown length (CL)    benp */
  Real barkthick_par1;              /* bark thickness par1  benp */
  Real barkthick_par2;              /* bark thickness par2  benp */
  Real crown_mort_rck;              /* crown damage (rCK)   benp */
  Real crown_mort_p;                /* crown damage (p)     benp */
  Real fuelfrac[NFUELCLASS];        /* Fuel fractions */
  Real k_est;                       /* maximum overall sapling establishment rate (indiv/m2) */
  int rotation;                     /* coppicing interval from short ratation woody crops (yr) */
  int max_rotation_length;          /* lifetime of short rotation woody crop plantation (yr) */
} Pfttreepar;

typedef struct
{
  Real height;            /* tree height (m) */
  Real crownarea;         /* crown area (m2) */
  Real barkthickness;     /* bark thickness Kirsten */
  Real gddtw;
  Real aphen_raingreen;
  Bool isphen;            /* only used for summergreen trees*/
  Treephys turn;
  Treephys turn_litt;
  Treephys2 ind;
  Treephyspar falloc;
  Real turn_nbminc;       /* storage for recovered nitrogen from turnover */
  Real excess_carbon; /* gC/ind; storage for carbon that cannot be allocated under given N limitation */
} Pfttree;

/* Declaration of functions */

extern void new_tree(Pft *,int,int);
extern Pft *newpftage(Pftpar *,int);
extern Real npp_tree(Pft *,Real,Real,Real,Bool);
extern Bool mortality_tree(Litter *,Pft *,Real,Real);
extern Real fpc_tree(Pft *);
extern Real fpar_tree(const Pft *);
extern Real alphaa_tree(const Pft *,int);
extern void litter_update_tree(Litter *, Pft *,Real);
extern void allometry_tree(Pft *pft);
extern Bool allocation_tree(Litter *,Pft *,Real *);
extern Real lai_tree(const Pft *);
extern Real actual_lai_tree(const Pft *);
extern Stocks turnover_tree(Litter *,Pft *);
extern void phenology_tree(Pft *,Real,int);
extern Bool fwrite_tree(FILE *,const Pft *);
extern void fprint_tree(FILE *,const Pft *);
extern Bool fread_tree(FILE *,Pft *,Bool);
extern Bool fscanpft_tree(FILE *,Pftpar *,Bool);
extern Bool isneg_tree(const Pft *);
extern Stocks establishment_tree(Pft *,Real,Real,int);
extern void init_tree(Pft *);
extern Stocks fire_tree(Pft *,Real *);
extern Real vegc_sum_tree(const Pft *);
extern Real vegn_sum_tree(const Pft *);
extern void mix_veg_tree(Pft *,Real);
extern void free_tree(Pft *);
extern void light_tree(Litter *,Pft *,Real);
extern void adjust_tree(Litter *,Pft *,Real);
extern void reduce_tree(Litter *,Pft *,Real);
extern void fprintpar_tree(FILE *,const Pftpar *);
extern void output_daily_tree(Daily_outputs *,const Pft *);
extern Stocks livefuel_consum_tree(Litter *,Pft *,const Fuel *,Livefuel *,
                                 Bool *,Real,Real);
extern Bool annual_tree(struct stand *,Pft *,Bool,Real *);
extern Stocks coppice_tree(Pft *);
extern Real albedo_tree(const Pft *,Real,Real);
extern void turnover_monthly_tree(Litter *,Pft *);
extern void turnover_daily_tree(Litter *, Pft *, Real);
extern Real nuptake_tree(Soil *,Pft *,Real *,Real *,int,int,int);
extern Real ndemand_tree(const Pft *,Real *,Real,Real,Real,int,int,int);
extern Real vmaxlimit_tree(const Pft *,Real,Real);
extern void nitrogen_allocation_tree(Real *,Real *,Real *,Stocks,Stocks,Stocks,Real,Real,Real);

/* Definitions of macros */

#define istree(pft) (getpftpar(pft,type)==TREE)
#define israingreen(pft) getpftpar(pft,phenology)==RAINGREEN
#define fprinttreephys2(file,phys,nind) fprintf(file,"%6.2f %6.2f %6.2f %6.2f %6.2f (gC/m2) %6.2f %6.2f %6.2f %6.2f %6.2f (gN/m2)",phys.leaf.carbon*nind,phys.sapwood.carbon*nind,phys.heartwood.carbon*nind,phys.root.carbon*nind,phys.debt.carbon*nind,phys.leaf.nitrogen*nind,phys.sapwood.nitrogen*nind,phys.heartwood.nitrogen*nind,phys.root.nitrogen*nind,phys.debt.nitrogen*nind)
#define phys_sum_tree(ind) (ind.leaf.carbon+ind.root.carbon+ind.heartwood.carbon+ind.sapwood.carbon)
#define phys_sum_tree_n(ind) (ind.leaf.nitrogen+ind.root.nitrogen+ind.heartwood.nitrogen+ind.sapwood.nitrogen)

#endif
