/*
 *
 *    Copyright (c) 2016-2018
 *      SMASH Team
 *
 *    GNU General Public License (GPLv3 or later)
 *
 */
#ifndef SRC_INCLUDE_PARAMETRIZATIONS_DATA_H_
#define SRC_INCLUDE_PARAMETRIZATIONS_DATA_H_

#include <initializer_list>
#include <memory>

#include "interpolation.h"

namespace smash {

/// PDG data on K- p elastic cross section: momentum in lab frame.
const std::initializer_list<double> KMINUSP_ELASTIC_P_LAB = {
    0.03000,   0.05000,   0.06287,   0.07000,  0.07044,  0.07300,  0.08730,
    0.09000,   0.11000,   0.11000,   0.11210,  0.11300,  0.12262,  0.13000,
    0.13000,   0.13722,   0.14552,   0.15000,  0.15000,  0.15300,  0.15909,
    0.16269,   0.17000,   0.17000,   0.17470,  0.18768,  0.18916,  0.19000,
    0.19000,   0.19300,   0.20534,   0.21000,  0.21000,  0.21554,  0.22049,
    0.22500,   0.23000,   0.23000,   0.23300,  0.23500,  0.23944,  0.24500,
    0.24623,   0.25000,   0.25500,   0.26154,  0.26500,  0.27000,  0.27500,
    0.27618,   0.28227,   0.28500,   0.29000,  0.29290,  0.29300,  0.29500,
    0.30000,   0.30191,   0.30500,   0.31500,  0.32500,  0.33500,  0.34500,
    0.35000,   0.35012,   0.35500,   0.36500,  0.37500,  0.38500,  0.38700,
    0.38986,   0.39200,   0.39500,   0.40000,  0.40500,  0.41500,  0.42000,
    0.42500,   0.43400,   0.43407,   0.43500,  0.43600,  0.44500,  0.45500,
    0.45500,   0.46500,   0.47500,   0.49500,  0.51300,  0.51300,  0.51400,
    0.52228,   0.53400,   0.55400,   0.57300,  0.59700,  0.61000,  0.61700,
    0.62004,   0.63700,   0.64200,   0.65800,  0.66000,  0.67200,  0.67700,
    0.69900,   0.70000,   0.70300,   0.70800,  0.71900,  0.72500,  0.73000,
    0.74000,   0.74000,   0.74100,   0.75800,  0.75996,  0.76100,  0.76800,
    0.77300,   0.77700,   0.77700,   0.78000,  0.78500,  0.79300,  0.80200,
    0.80600,   0.80600,   0.81000,   0.82000,  0.82000,  0.83300,  0.83800,
    0.83800,   0.84999,   0.85300,   0.85300,  0.85600,  0.86000,  0.87000,
    0.87400,   0.87400,   0.87600,   0.89400,  0.89400,  0.89900,  0.90000,
    0.90400,   0.90400,   0.90500,   0.91600,  0.91600,  0.92200,  0.92500,
    0.93500,   0.93500,   0.93500,   0.94000,  0.94300,  0.94500,  0.95400,
    0.95400,   0.95500,   0.96000,   0.96500,  0.97000,  0.97000,  0.98000,
    0.99000,   0.99100,   0.99100,   1.00500,  1.02000,  1.02200,  1.02200,
    1.04000,   1.04400,   1.04400,   1.04500,  1.06000,  1.06100,  1.06100,
    1.08000,   1.08000,   1.08500,   1.10000,  1.10200,  1.10200,  1.11700,
    1.11700,   1.12500,   1.13400,   1.13400,  1.13800,  1.14000,  1.15000,
    1.15000,   1.15300,   1.15300,   1.16100,  1.16500,  1.17400,  1.17400,
    1.17900,   1.18000,   1.18300,   1.18300,  1.20100,  1.20500,  1.22000,
    1.22600,   1.22600,   1.23300,   1.24500,  1.25300,  1.26000,  1.26000,
    1.26300,   1.26300,   1.27600,   1.28500,  1.29600,  1.30000,  1.31600,
    1.31600,   1.32000,   1.32800,   1.34000,  1.35500,  1.36800,  1.36800,
    1.38000,   1.38300,   1.41500,   1.41500,  1.42300,  1.43300,  1.46200,
    1.46500,   1.48300,   1.51300,   1.51400,  1.53000,  1.53400,  1.54500,
    1.54600,   1.58400,   1.60600,   1.60600,  1.63400,  1.65200,  1.65300,
    1.68000,   1.68400,   1.70500,   1.70500,  1.73400,  1.73900,  1.74100,
    1.78400,   1.80000,   1.80000,   1.81500,  1.84300,  1.84300,  1.88400,
    1.93400,   1.93400,   1.98400,   2.00000,  2.03100,  2.03400,  2.08400,
    2.13400,   2.13500,   2.17500,   2.23400,  2.24000,  2.28400,  2.32500,
    2.33100,   2.37400,   2.41200,   2.51600,  2.66000,  2.66000,  2.98500,
    3.00000,   3.00000,   3.46000,   3.59000,  3.65000,  4.10000,  4.20000,
    4.60000,   5.00000,   5.50000,   6.00000,  7.20020,  9.00010,  10.12000,
    14.30000,  14.30000,  25.20000,  32.10000, 40.10000, 50.00000, 70.00000,
    100.00000, 140.00000, 147.00000, 175.00000};

/// PDG data on K- p elastic cross section: cross section.
const std::initializer_list<double> KMINUSP_ELASTIC_SIG = {
    313.50, 103.60, 113.00, 44.800, 58.500, 187.00, 92.000, 71.500, 92.800,
    87.290, 82.000, 105.00, 59.400, 40.400, 79.220, 82.000, 49.000, 41.400,
    69.610, 108.00, 53.900, 98.000, 75.760, 32.800, 45.000, 73.000, 66.000,
    59.090, 53.300, 68.000, 37.000, 53.300, 60.490, 48.000, 41.000, 65.410,
    62.900, 55.690, 50.000, 55.740, 41.200, 53.240, 37.000, 51.500, 49.220,
    43.600, 47.710, 58.060, 48.750, 30.000, 44.900, 39.420, 38.270, 47.800,
    48.200, 41.220, 44.500, 44.200, 40.360, 37.020, 40.280, 37.840, 37.260,
    34.000, 33.500, 34.770, 34.210, 36.670, 33.890, 31.900, 34.700, 34.000,
    38.470, 38.900, 32.060, 32.590, 48.400, 31.190, 30.600, 32.800, 26.830,
    25.800, 28.830, 23.800, 30.320, 31.990, 23.100, 21.500, 26.500, 27.600,
    21.700, 35.000, 19.300, 19.100, 17.500, 17.700, 17.660, 18.600, 16.000,
    16.000, 17.230, 16.400, 12.100, 16.220, 15.600, 15.200, 14.200, 15.220,
    13.500, 14.200, 11.500, 14.070, 15.900, 14.000, 11.600, 16.930, 16.700,
    17.300, 15.200, 18.600, 18.300, 18.700, 17.900, 19.370, 20.500, 19.300,
    20.000, 20.450, 20.670, 18.700, 19.300, 19.630, 19.800, 19.530, 22.400,
    19.100, 19.390, 19.780, 19.500, 21.360, 20.100, 20.310, 21.070, 21.600,
    21.660, 22.020, 21.500, 20.900, 20.710, 21.340, 20.850, 20.400, 22.340,
    20.940, 20.120, 20.100, 19.980, 21.800, 21.010, 19.330, 20.640, 20.700,
    20.610, 21.270, 20.340, 20.400, 20.720, 22.400, 21.220, 21.400, 22.150,
    21.750, 20.800, 22.100, 22.150, 23.300, 21.500, 21.460, 22.220, 21.200,
    20.600, 20.560, 18.700, 18.740, 19.830, 18.300, 18.600, 18.770, 17.600,
    17.820, 17.890, 17.000, 17.750, 15.700, 17.000, 18.300, 17.300, 17.200,
    17.230, 15.300, 15.390, 16.500, 16.460, 14.660, 16.800, 15.900, 15.890,
    12.680, 13.890, 15.700, 11.300, 11.810, 13.110, 12.320, 11.870, 15.200,
    14.000, 10.870, 10.870, 11.440, 10.960, 11.570, 12.000, 10.200, 11.200,
    10.260, 9.7400, 14.400, 9.5300, 10.300, 10.300, 16.600, 10.500, 8.8300,
    8.8300, 8.4200, 8.6000, 9.1100, 9.1100, 8.3000, 7.7000, 7.7000, 8.1500,
    8.8000, 8.0600, 8.0600, 8.7000, 8.8600, 8.8600, 9.2000, 8.4000, 8.4000,
    8.1900, 8.9000, 9.0800, 9.0800, 9.5000, 8.5100, 8.5100, 9.0000, 8.1300,
    8.1300, 6.9500, 7.8600, 7.8600, 9.0000, 7.9000, 7.4800, 7.8000, 7.4600,
    7.3100, 7.1000, 7.4000, 7.1000, 7.2600, 7.9000, 6.8700, 6.2000, 7.3000,
    6.5000, 6.7600, 6.6000, 6.3200, 6.2100, 5.7000, 6.0400, 4.9500, 5.0600,
    4.9500, 4.9400, 4.4000, 4.6000, 4.3000, 4.5000, 4.2000, 3.8400, 4.1000,
    3.6200, 4.2300, 3.9500, 3.2400, 2.9600, 3.0100, 2.4600, 2.5600, 2.3300,
    2.5400, 2.5300, 2.5100, 2.5200, 2.7400, 2.5900};

/// An interpolation that gets lazily filled using the KMINUSP_ELASTIC data.
static std::unique_ptr<InterpolateDataLinear<double>>
    kminusp_elastic_interpolation = nullptr;

/// PDG data on K- p total cross section: momentum in lab frame.
const std::initializer_list<double> KMINUSP_TOT_PLAB = {
    0.245,   0.255,   0.265,   0.275,   0.285,   0.293,   0.293,   0.295,
    0.300,   0.305,   0.315,   0.325,   0.335,   0.345,   0.350,   0.350,
    0.355,   0.360,   0.365,   0.375,   0.385,   0.385,   0.387,   0.390,
    0.392,   0.395,   0.400,   0.405,   0.405,   0.415,   0.419,   0.425,
    0.434,   0.434,   0.435,   0.436,   0.440,   0.445,   0.446,   0.455,
    0.455,   0.464,   0.475,   0.475,   0.483,   0.494,   0.495,   0.506,
    0.506,   0.513,   0.513,   0.514,   0.520,   0.534,   0.534,   0.536,
    0.547,   0.554,   0.561,   0.566,   0.573,   0.574,   0.586,   0.590,
    0.596,   0.597,   0.599,   0.604,   0.612,   0.617,   0.620,   0.623,
    0.625,   0.627,   0.630,   0.637,   0.639,   0.653,   0.655,   0.657,
    0.658,   0.667,   0.677,   0.680,   0.686,   0.687,   0.693,   0.699,
    0.707,   0.708,   0.712,   0.719,   0.720,   0.725,   0.732,   0.736,
    0.740,   0.741,   0.744,   0.752,   0.756,   0.760,   0.761,   0.767,
    0.768,   0.768,   0.773,   0.777,   0.782,   0.793,   0.794,   0.795,
    0.802,   0.806,   0.819,   0.820,   0.822,   0.838,   0.839,   0.840,
    0.848,   0.850,   0.853,   0.860,   0.874,   0.874,   0.881,   0.894,
    0.894,   0.899,   0.900,   0.900,   0.904,   0.916,   0.925,   0.931,
    0.935,   0.936,   0.952,   0.954,   0.965,   0.969,   0.970,   0.975,
    0.978,   0.980,   0.987,   0.991,   0.991,   1.020,   1.022,   1.022,
    1.023,   1.030,   1.043,   1.044,   1.061,   1.067,   1.068,   1.070,
    1.080,   1.080,   1.093,   1.100,   1.101,   1.102,   1.115,   1.117,
    1.119,   1.130,   1.134,   1.144,   1.150,   1.153,   1.169,   1.169,
    1.174,   1.179,   1.183,   1.194,   1.219,   1.220,   1.226,   1.230,
    1.230,   1.244,   1.263,   1.270,   1.274,   1.283,   1.306,   1.316,
    1.320,   1.325,   1.328,   1.336,   1.350,   1.368,   1.370,   1.375,
    1.383,   1.384,   1.406,   1.415,   1.433,   1.434,   1.442,   1.462,
    1.480,   1.483,   1.485,   1.488,   1.509,   1.514,   1.521,   1.534,
    1.546,   1.549,   1.562,   1.583,   1.584,   1.600,   1.606,   1.614,
    1.634,   1.637,   1.653,   1.663,   1.684,   1.687,   1.705,   1.710,
    1.730,   1.734,   1.735,   1.741,   1.763,   1.784,   1.785,   1.800,
    1.805,   1.835,   1.840,   1.843,   1.855,   1.879,   1.884,   1.885,
    1.903,   1.924,   1.929,   1.934,   1.950,   1.953,   1.984,   1.985,
    2.005,   2.031,   2.034,   2.035,   2.052,   2.080,   2.084,   2.104,
    2.124,   2.134,   2.135,   2.159,   2.164,   2.175,   2.206,   2.207,
    2.234,   2.240,   2.250,   2.257,   2.284,   2.301,   2.308,   2.325,
    2.331,   2.350,   2.351,   2.356,   2.374,   2.392,   2.400,   2.404,
    2.423,   2.450,   2.467,   2.480,   2.500,   2.514,   2.550,   2.558,
    2.600,   2.608,   2.630,   2.650,   2.657,   2.660,   2.660,   2.670,
    2.700,   2.700,   2.750,   2.800,   2.850,   2.880,   2.900,   2.950,
    2.970,   2.980,   3.000,   3.000,   3.050,   3.090,   3.100,   3.150,
    3.190,   3.200,   3.250,   3.300,   3.300,   3.500,   3.710,   3.920,
    3.980,   4.000,   4.130,   4.200,   4.250,   4.340,   4.760,   5.180,
    5.500,   6.000,   6.650,   7.000,   7.000,   8.000,   8.500,   10.000,
    10.000,  10.000,  10.000,  10.000,  10.100,  11.500,  12.000,  12.000,
    12.600,  13.300,  13.400,  14.000,  14.500,  15.000,  15.800,  16.000,
    17.500,  18.000,  19.100,  20.000,  20.000,  20.000,  21.000,  21.000,
    23.000,  25.000,  25.000,  25.000,  27.500,  30.000,  30.000,  30.000,
    31.500,  32.000,  33.500,  35.000,  35.000,  35.500,  37.500,  40.000,
    40.000,  40.000,  40.000,  42.500,  45.000,  45.000,  45.000,  47.500,
    50.000,  50.000,  50.000,  50.000,  50.000,  52.500,  55.000,  55.000,
    55.000,  70.000,  100.000, 100.000, 100.000, 120.000, 147.000, 150.000,
    150.000, 170.000, 175.000, 200.000, 200.000, 240.000, 280.000, 310.000};

/// PDG data on K- p total cross section: cross section.
const std::initializer_list<double> KMINUSP_TOT_SIG = {
    113.80, 98.00, 94.00, 96.70, 75.10, 89.30, 90.70, 82.50, 79.40, 78.60,
    70.90,  76.00, 71.50, 71.30, 70.20, 69.30, 68.80, 69.32, 70.00, 81.80,
    100.19, 88.30, 69.60, 77.80, 80.60, 91.60, 76.20, 69.80, 76.60, 70.70,
    58.61,  66.70, 61.50, 62.80, 54.90, 56.70, 58.41, 55.70, 50.27, 57.20,
    48.60,  46.82, 45.30, 48.45, 44.19, 43.19, 42.30, 42.70, 44.71, 50.10,
    49.50,  40.70, 40.11, 40.50, 39.55, 42.15, 38.34, 39.30, 37.78, 39.10,
    35.60,  36.74, 36.32, 32.99, 37.80, 36.50, 36.03, 38.59, 35.07, 36.30,
    32.40,  34.40, 34.75, 36.36, 35.40, 32.70, 34.41, 34.31, 34.44, 35.40,
    34.90,  34.38, 33.70, 34.34, 35.33, 34.16, 34.81, 33.10, 34.63, 32.60,
    35.37,  35.40, 34.69, 33.30, 35.26, 35.64, 36.80, 31.90, 37.65, 36.40,
    38.84,  40.10, 38.10, 38.57, 38.50, 39.87, 40.60, 40.70, 40.31, 40.90,
    40.96,  40.53, 40.20, 41.30, 40.67, 39.70, 40.29, 37.30, 39.86, 36.10,
    40.58,  40.60, 38.10, 41.13, 41.75, 41.30, 42.36, 43.70, 42.70, 43.41,
    43.41,  52.00, 42.10, 43.20, 44.86, 44.10, 42.40, 45.46, 46.24, 42.90,
    47.33,  47.16, 43.20, 47.19, 47.96, 47.10, 45.30, 45.80, 49.79, 51.70,
    51.27,  47.70, 52.19, 48.40, 52.17, 51.90, 50.30, 49.40, 51.46, 49.30,
    50.51,  46.90, 46.13, 43.60, 44.50, 43.50, 44.18, 40.90, 42.23, 43.74,
    39.30,  39.80, 45.00, 40.50, 39.18, 38.67, 42.00, 39.01, 40.70, 36.92,
    35.21,  35.51, 33.40, 35.41, 33.80, 33.39, 32.36, 32.52, 32.11, 32.50,
    31.17,  29.68, 31.03, 30.60, 30.50, 31.12, 31.20, 32.61, 31.27, 30.90,
    31.20,  31.09, 31.63, 29.78, 31.90, 31.83, 32.28, 31.68, 32.50, 32.90,
    33.35,  32.83, 33.47, 29.87, 33.53, 33.80, 32.16, 34.00, 33.97, 34.17,
    34.20,  32.50, 33.19, 34.42, 34.20, 34.07, 32.79, 34.19, 33.80, 33.84,
    33.20,  33.66, 32.50, 33.20, 33.19, 33.42, 32.77, 32.30, 32.26, 32.94,
    31.92,  31.53, 31.19, 31.32, 30.72, 30.85, 30.40, 30.81, 30.18, 30.40,
    30.49,  30.14, 30.50, 30.46, 30.10, 30.17, 30.37, 30.71, 29.90, 29.99,
    29.86,  29.89, 29.80, 29.70, 29.81, 29.60, 31.77, 29.83, 29.79, 29.70,
    29.87,  30.05, 30.75, 29.79, 29.79, 29.46, 29.50, 29.63, 29.37, 29.30,
    31.89,  29.44, 29.38, 28.81, 28.90, 29.07, 29.10, 28.86, 28.88, 28.87,
    28.50,  26.90, 28.46, 28.37, 28.30, 28.13, 28.21, 27.88, 30.30, 28.10,
    27.96,  30.40, 30.20, 27.41, 28.02, 30.90, 27.83, 27.72, 27.73, 26.65,
    27.58,  27.51, 25.30, 27.76, 24.90, 27.42, 27.20, 26.25, 27.00, 26.98,
    26.75,  26.78, 26.70, 26.73, 27.00, 27.15, 26.90, 25.95, 25.40, 28.20,
    25.14,  25.40, 25.80, 25.84, 24.74, 24.03, 24.30, 24.00, 23.80, 25.00,
    23.50,  23.60, 24.60, 23.20, 22.90, 22.60, 22.67, 22.50, 22.50, 23.30,
    22.90,  21.60, 21.60, 22.10, 21.50, 21.50, 21.60, 21.60, 21.90, 21.30,
    21.90,  21.00, 21.60, 21.20, 21.02, 22.40, 21.16, 21.09, 20.89, 20.72,
    20.70,  20.81, 20.72, 21.30, 20.84, 20.57, 20.50, 19.90, 20.55, 20.80,
    20.54,  20.43, 20.39, 20.90, 20.59, 20.62, 20.49, 20.35, 20.60, 20.57,
    20.45,  20.45, 21.00, 20.48, 20.30, 20.25, 20.40, 20.32, 21.50, 20.49,
    20.23,  20.38, 20.80, 20.45, 20.41, 20.59, 21.50, 20.60, 20.57, 20.65,
    21.40,  20.79, 20.84, 21.30, 21.32, 21.45};

/// Center-of-mass energy list for K̅⁻ N⁺
const std::initializer_list<double> KMINUSP_RES_SQRTS = {
    1.4355, 1.4530, 1.4730, 1.4929, 1.5129, 1.5329, 1.5529, 1.5729,
    1.5929, 1.6128, 1.6328, 1.6528, 1.6728, 1.6928, 1.7127, 1.7327,
    1.7527, 1.7727, 1.7927, 1.8127, 1.8326, 1.8526, 1.8726, 1.8926,
    1.9126, 1.9325, 1.9525, 1.9725, 1.9925, 2.0125, 2.0325, 2.0524,
    2.0724, 2.0924, 2.1124, 2.1324, 2.1523, 2.1723, 2.1923, 2.2123,
    2.2323, 2.2523, 2.2722, 2.2922, 2.3122, 2.3322, 2.3522, 2.3721,
    2.3921, 2.4121, 2.4321, 2.4521, 2.4721, 2.4920, 2.5120, 2.5320};

/**
 * Elastic K̅⁻ N⁺ cross section contributions from decays.
 *
 * These need to be subtracted from the interpolation of the PDG data on
 * elastic cross sections. This data was generated using the SMASH analysis
 * suite and should be updated when strange resonances are changed or added.
 */
const std::initializer_list<double> KMINUSP_RES_SIG = {
    0.46856081097,  0.68994120313, 1.00475205957,  1.66469547655,
    5.63530052434,  4.17372028288, 4.50737036469,  8.00913400697,
    0.29205365102,  2.72859364291, 3.30822314603,  4.44740017628,
    4.95697831919,  5.05350905117, 4.87562017799,  7.48383422000,
    8.29845755438,  9.71940157530, 11.10200040600, 12.00610574630,
    10.06137989140, 7.47886042856, 6.11390219499,  5.19531126779,
    4.38090191191,  3.95316327084, 3.53446044755,  3.46497827089,
    3.63741875589,  3.77762079044, 0.87409952036,  4.19070149234,
    4.38097308237,  4.27752586136, 4.12637945445,  3.70027602474,
    3.31806303484,  2.88526838044, 2.58141493751,  2.36391939397,
    2.18133708906,  1.39193162095, 2.03247269918,  2.00726146262,
    2.13817978212,  2.16907178433, 2.08118209913,  1.83166338166,
    1.56038155638,  1.27216056674, 1.03167072054,  0.85006416230,
    0.39627220898,  0.57172926654, 0.51129452389,  0.44626386026};

/// An interpolation that gets lazily filled using the KMINUSP_RES data.
static std::unique_ptr<InterpolateDataSpline>
    kminusp_elastic_res_interpolation = nullptr;

/**
 * PDG data on K+ n total cross section: momentum in lab frame.
 * One data point is ignored because it is an outlier and messes up the
 * smoothing.
 */
// clang-format off
const std::initializer_list<double> KPLUSN_TOT_PLAB = {
    0.770,   0.888,   0.939,   0.970,   0.989,   1.040,   1.091,   1.141,
    /*1.170,*/ 1.191,   1.242,   1.292,   1.300,   1.342,   1.392,   1.440,
    1.442,   1.492,   1.550,   1.593,   1.600,   1.643,   1.690,   1.693,
    1.700,   1.743,   1.750,   1.793,   1.800,   1.850,   1.893,   1.900,
    1.950,   1.970,   1.993,   2.000,   2.050,   2.093,   2.100,   2.150,
    2.193,   2.200,   2.260,   2.300,   2.350,   2.393,   2.400,   2.450,
    2.500,   2.550,   2.550,   2.600,   2.650,   2.700,   2.750,   2.800,
    2.830,   2.850,   2.900,   2.950,   3.000,   3.050,   3.100,   3.150,
    3.200,   3.250,   3.300,   6.000,   8.000,   10.000,  12.000,  14.000,
    15.000,  16.000,  18.000,  20.000,  20.000,  25.000,  30.000,  35.000,
    35.000,  40.000,  45.000,  50.000,  50.000,  50.000,  55.000,  70.000,
    100.000, 100.000, 120.000, 150.000, 150.000, 170.000, 200.000, 200.000,
    240.000, 280.000, 310.000};
// clang-format on

/**
 * PDG data on K+ n total cross section: cross section.
 * One data point is ignored because it is an outlier and messes up the
 * smoothing.
 */
const std::initializer_list<double> KPLUSN_TOT_SIG = {
    15.50, 16.85, 17.60, 17.80, 18.53, 18.91, 20.61, 21.25, /*18.20,*/ 20.87,
    20.26, 19.68, 18.50, 19.32, 19.22, 18.10, 19.07, 18.95, 18.91,
    18.79, 18.89, 18.67, 18.50, 18.69, 18.83, 18.88, 18.86, 18.73,
    18.53, 18.66, 18.50, 18.69, 18.70, 18.60, 18.55, 18.79, 18.54,
    18.67, 18.49, 18.43, 18.40, 18.40, 17.70, 18.27, 18.26, 18.63,
    18.09, 18.25, 18.11, 17.10, 18.17, 18.09, 18.02, 18.11, 18.06,
    18.01, 17.50, 17.95, 17.85, 17.81, 17.81, 17.83, 17.85, 17.61,
    17.61, 17.66, 17.55, 17.50, 17.60, 17.50, 17.60, 17.50, 17.87,
    17.40, 17.60, 17.94, 17.70, 17.78, 17.69, 18.29, 18.12, 18.15,
    18.30, 18.66, 18.56, 18.02, 18.43, 18.60, 19.04, 18.99, 19.23,
    19.63, 19.55, 19.74, 19.72, 19.82, 20.37, 20.61, 20.80};

/// An interpolation that gets lazily filled using the KPLUSN_TOT data.
static std::unique_ptr<InterpolateDataLinear<double>>
    kplusn_total_interpolation = nullptr;

/// PDG data on K+ p total cross section: momentum in lab frame.
const std::initializer_list<double> KPLUSP_TOT_PLAB = {
    0.178,   0.265,   0.321,   0.351,   0.366,   0.405,   0.440,   0.451,
    0.475,   0.475,   0.506,   0.522,   0.536,   0.566,   0.569,   0.588,
    0.589,   0.592,   0.593,   0.596,   0.618,   0.620,   0.627,   0.643,
    0.644,   0.657,   0.668,   0.686,   0.698,   0.713,   0.717,   0.727,
    0.757,   0.768,   0.770,   0.786,   0.817,   0.823,   0.846,   0.864,
    0.864,   0.881,   0.891,   0.900,   0.904,   0.916,   0.938,   0.942,
    0.951,   0.969,   0.969,   0.970,   0.970,   0.985,   0.992,   1.020,
    1.029,   1.043,   1.055,   1.060,   1.084,   1.090,   1.094,   1.125,
    1.130,   1.140,   1.144,   1.160,   1.170,   1.189,   1.194,   1.207,
    1.210,   1.238,   1.245,   1.250,   1.293,   1.295,   1.300,   1.320,
    1.345,   1.347,   1.380,   1.395,   1.408,   1.440,   1.445,   1.455,
    1.468,   1.480,   1.495,   1.550,   1.563,   1.596,   1.600,   1.646,
    1.690,   1.696,   1.700,   1.746,   1.750,   1.796,   1.800,   1.850,
    1.896,   1.900,   1.945,   1.950,   1.960,   1.970,   1.996,   2.000,
    2.050,   2.096,   2.100,   2.150,   2.196,   2.200,   2.260,   2.300,
    2.350,   2.396,   2.400,   2.450,   2.473,   2.500,   2.530,   2.550,
    2.550,   2.600,   2.650,   2.700,   2.750,   2.760,   2.800,   2.830,
    2.850,   2.900,   2.950,   3.000,   3.050,   3.100,   3.150,   3.200,
    3.200,   3.250,   3.250,   3.300,   3.700,   4.000,   4.200,   4.750,
    5.000,   5.500,   6.000,   7.000,   7.000,   8.000,   8.200,   8.500,
    10.000,  10.000,  10.000,  10.000,  10.900,  11.500,  12.000,  12.500,
    13.400,  14.000,  15.000,  15.000,  16.000,  16.000,  16.900,  18.000,
    19.000,  20.000,  20.000,  25.000,  30.000,  32.000,  35.000,  35.000,
    40.000,  42.500,  45.000,  50.000,  50.000,  50.000,  52.200,  55.000,
    70.000,  100.000, 100.000, 100.000, 100.000, 120.000, 147.000, 150.000,
    150.000, 170.000, 175.000, 200.000, 200.000, 240.000, 280.000, 310.000};

/// PDG data on K+ p total cross section: cross section.
const std::initializer_list<double> KPLUSP_TOT_SIG = {
    11.40, 13.00, 14.00, 12.20, 13.20, 13.69, 12.81, 16.30, 12.70, 13.58, 13.02,
    15.20, 12.09, 13.20, 12.70, 12.60, 16.30, 14.36, 13.05, 13.04, 12.65, 12.91,
    12.18, 12.50, 12.88, 12.43, 13.10, 11.25, 12.60, 11.14, 11.10, 12.45, 12.65,
    11.65, 13.00, 12.80, 13.20, 12.97, 13.45, 14.07, 13.21, 13.90, 14.39, 13.10,
    14.23, 14.20, 14.59, 15.57, 14.95, 15.28, 15.63, 15.40, 15.25, 16.20, 15.97,
    16.10, 15.69, 17.39, 16.95, 16.40, 17.04, 17.60, 17.12, 17.55, 18.08, 18.02,
    18.09, 17.95, 18.10, 18.06, 18.47, 19.85, 18.58, 18.11, 18.54, 20.71, 18.44,
    18.61, 17.90, 19.33, 18.44, 18.27, 18.64, 18.27, 17.97, 18.10, 18.04, 18.20,
    17.94, 18.04, 17.93, 17.70, 17.66, 17.75, 17.71, 17.86, 17.50, 17.85, 17.73,
    17.80, 17.83, 17.80, 17.98, 17.77, 17.81, 17.79, 17.41, 17.75, 19.40, 16.90,
    17.60, 17.63, 17.72, 17.51, 17.56, 17.57, 17.54, 17.60, 17.10, 17.44, 17.52,
    17.55, 17.56, 17.48, 17.25, 17.49, 17.47, 17.10, 17.44, 17.50, 17.47, 17.41,
    17.41, 17.41, 17.40, 16.70, 17.30, 17.34, 17.30, 17.19, 17.14, 17.08, 17.15,
    17.13, 17.13, 17.13, 17.50, 17.14, 21.00, 17.60, 17.10, 21.30, 17.20, 17.90,
    17.00, 17.20, 18.40, 17.30, 17.20, 18.70, 17.20, 17.30, 18.80, 17.30, 18.10,
    19.00, 17.30, 18.30, 17.50, 17.40, 18.50, 17.31, 17.10, 17.00, 18.80, 17.10,
    17.30, 17.50, 17.42, 17.68, 17.72, 18.40, 17.82, 17.80, 18.05, 17.91, 17.88,
    18.06, 18.03, 18.37, 18.28, 18.17, 18.52, 18.40, 18.88, 18.70, 18.85, 19.14,
    19.52, 19.36, 19.33, 19.64, 18.20, 19.91, 19.84, 20.22, 20.45, 20.67};

/// An interpolation that gets lazily filled using the KPLUSP_TOT data.
static std::unique_ptr<InterpolateDataLinear<double>>
    kplusp_total_interpolation = nullptr;

/// PDG data on pi- p elastic cross section: momentum in lab frame.
const std::initializer_list<double> PIMINUSP_ELASTIC_P_LAB = {
    0.09875, 0.14956, 0.21648, 0.21885, 0.22828, 0.24684, 0.25599, 0.26733,
    0.27071, 0.2752,  0.29303, 0.29303, 0.33138, 0.33571, 0.33788, 0.35052,
    0.378,   0.38261, 0.404,   0.40626, 0.408,   0.42188, 0.427,   0.44888,
    0.452,   0.471,   0.49008, 0.49008, 0.49008, 0.509,   0.523,   0.52845,
    0.53155, 0.547,   0.54911, 0.54911, 0.556,   0.565,   0.57281, 0.582,
    0.586,   0.609,   0.6139,  0.61698, 0.625,   0.64054, 0.657,   0.65793,
    0.658,   0.6753,  0.683,   0.687,   0.69061, 0.699,   0.70692, 0.71399,
    0.72628, 0.731,   0.73257, 0.74257, 0.75,    0.76189, 0.775,   0.77714,
    0.77714, 0.77827, 0.798,   0.82586, 0.83803, 0.848,   0.84954, 0.854,
    0.87466, 0.90386, 0.91903, 0.924,   0.95947, 1.03016, 1.106,   1.12339,
    1.164,   1.165,   1.174,   1.214,   1.21659, 1.23,    1.25,    1.26,
    1.279,   1.323,   1.339,   1.347,   1.355,   1.365,   1.375,   1.385,
    1.395,   1.405,   1.415,   1.425,   1.435,   1.445,   1.455,   1.465,
    1.475,   1.485,   1.495,   1.497,   1.499,   1.503,   1.5031,  1.509,
    1.567,   1.59,    1.6,     1.603,   1.71,    1.85,    2.1,     2.14,
    2.26,    2.29,    2.7,     2.75,    2.77,    2.7999,  3.,      3.15};

/// PDG data on pi- p elastic cross section: cross section
const std::initializer_list<double> PIMINUSP_ELASTIC_SIG = {
    1.847,  2.9,    9.6,    11.3,   12.8,   17.,    20.1,   21.4,   22.5,
    21.2,   22.5,   18.2,   19.5,   16.,    17.4,   15.1,   12.29,  12.4,
    10.1,   13.8,   10.41,  11.4,   9.,     10.3,   8.9,    9.2,    10.42,
    10.9,   10.8,   9.15,   9.8,    11.4,   11.4,   9.99,   13.,    13.,
    10.24,  10.8,   12.19,  16.2,   11.34,  12.86,  13.71,  13.9,   12.19,
    14.8,   13.92,  16.2,   15.32,  16.98,  18.9,   17.07,  18.86,  19.07,
    19.95,  20.5,   19.87,  18.9,   16.6,   19.4,   19.91,  18.94,  17.56,
    17.19,  17.82,  16.1,   14.91,  15.75,  14.9,   13.2,   14.1,   14.47,
    14.4,   14.8,   14.1,   18.8,   18.,    18.6,   17.95,  17.7,   13.66,
    15.01,  15.73,  12.45,  14.1,   14.6,   13.31,  13.8,   12.8,   13.09,
    12.627, 12.987, 12.763, 12.367, 12.852, 12.67,  12.126, 12.972, 12.478,
    12.594, 12.532, 11.801, 11.568, 11.413, 11.119, 11.643, 11.368, 11.523,
    11.163, 11.69,  10.,    10.39,  10.21,  9.65,   9.,     9.82,   10.4,
    11.1,   9.69,   9.3,    8.91,   8.5,    7.7,    7.2,    7.2,    7.8,
    7.57,   6.1};

/// An interpolation that gets lazily filled using the PIMINUSP_ELASTIC data.
static std::unique_ptr<InterpolateDataLinear<double>>
    piminusp_elastic_interpolation = nullptr;

/// PDG data on pi- p to Lambda K0 cross section: momentum in lab frame.
const std::initializer_list<double> PIMINUSP_LAMBDAK0_P_LAB = {
    0.904, 0.91,  0.919, 0.922, 0.926, 0.93,  0.931, 0.942, 0.945, 0.958, 0.964,
    0.968, 0.98,  0.98,  0.983, 0.993, 0.997, 1.001, 1.002, 1.007, 1.012, 1.017,
    1.02,  1.02,  1.021, 1.022, 1.023, 1.027, 1.031, 1.035, 1.04,  1.04,  1.04,
    1.043, 1.048, 1.054, 1.059, 1.065, 1.078, 1.081, 1.091, 1.091, 1.094, 1.095,
    1.097, 1.116, 1.129, 1.13,  1.135, 1.144, 1.159, 1.194, 1.223, 1.235, 1.277,
    1.287, 1.326, 1.331, 1.332, 1.395, 1.433, 1.455, 1.5,   1.508, 1.515, 1.575,
    1.59,  1.6,   1.615, 1.645, 1.69,  1.69,  1.705, 1.775, 1.85,  1.875, 1.94,
    1.95,  1.98,  2.,    2.05,  2.05,  2.055, 2.115, 2.14,  2.15,  2.235, 2.25,
    2.35,  2.375, 2.494, 2.605, 2.7,   2.75,  2.75,  2.86,  3.01,  3.125, 3.21,
    3.885, 3.9,   3.95,  4.16,  4.5,   6.,    8.,    10.};

/// PDG data on pi- p to Lambda K0 cross section: cross section
const std::initializer_list<double> PIMINUSP_LAMBDAK0_SIG = {
    0.056, 0.122,  0.18,  0.14,  0.227, 0.212,  0.13,  0.3,   0.336, 0.43,
    0.427, 0.52,   0.467, 0.45,  0.576, 0.59,   0.652, 0.56,  0.588, 0.634,
    0.686, 0.665,  0.67,  0.69,  0.809, 0.675,  0.94,  0.737, 0.734, 0.73,
    0.926, 0.59,   0.92,  0.57,  0.568, 0.651,  0.899, 0.64,  0.794, 0.58,
    0.82,  0.58,   0.7,   0.68,  0.729, 0.575,  0.592, 0.462, 0.541, 0.64,
    0.48,  0.58,   0.46,  0.485, 0.447, 0.25,   0.367, 0.32,  0.29,  0.25,
    0.32,  0.29,   0.334, 0.214, 0.22,  0.21,   0.214, 0.238, 0.208, 0.16,
    0.199, 0.174,  0.14,  0.13,  0.181, 0.16,   0.185, 0.182, 0.184, 0.15,
    0.182, 0.179,  0.11,  0.16,  0.162, 0.192,  0.15,  0.172, 0.174, 0.12,
    0.16,  0.106,  0.12,  0.09,  0.09,  0.109,  0.084, 0.094, 0.087, 0.067,
    0.058, 0.0644, 0.049, 0.054, 0.038, 0.0221, 0.0157};

/// An interpolation that gets lazily filled using the PIMINUSP_LAMBDAK0 data.
static std::unique_ptr<InterpolateDataLinear<double>>
    piminusp_lambdak0_interpolation = nullptr;

/// PDG data on pi- p to Sigma- K+ cross section: momentum in lab frame
const std::initializer_list<double> PIMINUSP_SIGMAMINUSKPLUS_P_LAB = {
    1.091, 1.128, 1.17, 1.22,  1.235, 1.284, 1.326, 1.5,  1.59,
    1.615, 1.69,  1.69, 1.85,  1.94,  1.95,  1.98,  2.05, 2.14,
    2.15,  2.25,  2.35, 2.494, 2.61,  2.7,   2.75,  2.86, 3.,
    3.01,  3.13,  3.21, 3.89,  3.95,  4.,    4.16};

/// PDG data on pi- p to Sigma- K+ cross section: cross section
const std::initializer_list<double> PIMINUSP_SIGMAMINUSKPLUS_SIG = {
    0.25,  0.218,  0.231,  0.27,   0.235,  0.209, 0.245, 0.242, 0.262,
    0.18,  0.153,  0.19,   0.099,  0.098,  0.099, 0.09,  0.087, 0.069,
    0.065, 0.057,  0.053,  0.051,  0.03,   0.031, 0.032, 0.022, 0.015,
    0.022, 0.0155, 0.0145, 0.0085, 0.0096, 0.005, 0.0045};

/**
 * An interpolation that gets lazily filled using the
 * PIMINUSP_SIGMAMINUSKPLUS data.
 */
static std::unique_ptr<InterpolateDataLinear<double>>
    piminusp_sigmaminuskplus_interpolation = nullptr;

/// pi- p to Sigma0 K0 cross section: square root s
const std::initializer_list<double> PIMINUSP_SIGMA0K0_RES_SQRTS = {
    1.5,   1.516, 1.532, 1.548, 1.564, 1.58,  1.596, 1.612, 1.628, 1.644, 1.66,
    1.676, 1.692, 1.708, 1.724, 1.74,  1.756, 1.772, 1.788, 1.804, 1.82,  1.836,
    1.852, 1.868, 1.884, 1.9,   1.916, 1.932, 1.948, 1.964, 1.98,  1.996, 2.012,
    2.028, 2.044, 2.06,  2.076, 2.092, 2.108, 2.124, 2.14,  2.156, 2.172, 2.188,
    2.204, 2.22,  2.236, 2.252, 2.268, 2.284, 2.3,   2.316, 2.332, 2.348, 2.364,
    2.38,  2.396, 2.412, 2.428, 2.444, 2.46,  2.476, 2.492, 2.508, 2.524, 2.54,
    2.556, 2.572, 2.588, 2.604, 2.62,  2.636, 2.652, 2.668, 2.684, 2.7,   2.716,
    2.732, 2.748, 2.764, 2.78,  2.796, 2.812, 2.828, 2.844, 2.86,  2.876, 2.892,
    2.908, 2.924, 2.94,  2.956, 2.972, 2.988, 3.004, 3.02,  3.036, 3.052, 3.068,
    3.084, 3.1,   3.116, 3.132, 3.148, 3.164, 3.18};

/**
 * pi- p to Sigma0 K0 cross section: cross section
 *
 * The experimental data is missing, so the cross section is obtained
 * by running the simulation, and purely contributed by the resonances
 */
const std::initializer_list<double> PIMINUSP_SIGMA0K0_RES_SIG = {
    0.,         0.,         0.,         0.,         0.,         0.,
    0.,         0.,         0.,         0.,         0.,         0.,
    0.0386981,  0.09589789, 0.11956695, 0.11685363, 0.12053117, 0.13208736,
    0.14949223, 0.16688579, 0.18830654, 0.20611132, 0.22231072, 0.23099061,
    0.23734563, 0.23334048, 0.22794051, 0.21559531, 0.20134617, 0.18763246,
    0.1723282,  0.15814744, 0.14757816, 0.13750278, 0.12698656, 0.11719809,
    0.11024985, 0.1044732,  0.09623321, 0.09092108, 0.08670191, 0.08147493,
    0.0772165,  0.07346243, 0.0719974,  0.06805902, 0.06496733, 0.06264939,
    0.05904799, 0.05762721, 0.05588871, 0.05393479, 0.0517673,  0.05165839,
    0.05087591, 0.04885535, 0.04730724, 0.04651682, 0.04604065, 0.04529776,
    0.04406593, 0.04367817, 0.04230014, 0.04144308, 0.04171145, 0.04073006,
    0.03996921, 0.03902337, 0.03939531, 0.03895125, 0.03904553, 0.03816119,
    0.03772662, 0.03710955, 0.0361001,  0.03632378, 0.03549849, 0.03549987,
    0.03527251, 0.034535,   0.03314715, 0.0335742,  0.03326698, 0.0330181,
    0.0324203,  0.03227253, 0.0315376,  0.03065083, 0.03041305, 0.03023753,
    0.03008669, 0.02900321, 0.02827017, 0.02805024, 0.02785525, 0.02753706,
    0.02692862, 0.02603758, 0.02591122, 0.02537291, 0.02467199, 0.02466657,
    0.02370074, 0.02353027, 0.02362089, 0.0230085};

/**
 * An interpolation that gets lazily filled using the
 * PIMINUSP_SIGMA0K0_RES data.
 */
static std::unique_ptr<InterpolateDataLinear<double>>
    piminusp_sigma0k0_interpolation = nullptr;

/// Center-of-mass energy.
const std::initializer_list<double> PIMINUSP_RES_SQRTS = {
    1.1438620, 1.1482410, 1.1514750, 1.1566800, 1.1572040, 1.1579910, 1.1665900,
    1.1679230, 1.1701290, 1.1720720, 1.1764440, 1.1765450, 1.1773210, 1.1786720,
    1.1823280, 1.1845210, 1.1869010, 1.1881750, 1.1884820, 1.1895390, 1.1931590,
    1.1955810, 1.1984320, 1.1998050, 1.2002920, 1.2018580, 1.2047590, 1.2057700,
    1.2077800, 1.2087440, 1.2101220, 1.2117760, 1.2119900, 1.2150870, 1.2151980,
    1.2156390, 1.2158600, 1.2178480, 1.2190910, 1.2203690, 1.2204730, 1.2204940,
    1.2228010, 1.2255380, 1.2260010, 1.2271080, 1.2274050, 1.2291260, 1.2294580,
    1.2308410, 1.2338980, 1.2342780, 1.2350390, 1.2363740, 1.2374810, 1.2398320,
    1.2411660, 1.2412150, 1.2447480, 1.2455710, 1.2456470, 1.2464000, 1.2483350,
    1.2501600, 1.2508920, 1.2515830, 1.2516660, 1.2519140, 1.2539170, 1.2554980,
    1.2569070, 1.2579900, 1.2584040, 1.2603780, 1.2605570, 1.2613850, 1.2619360,
    1.2628740, 1.2633150, 1.2653620, 1.2658450, 1.2666570, 1.2677530, 1.2688130,
    1.2701420, 1.2708920, 1.2722680, 1.2726330, 1.2750260, 1.2774170, 1.2799030,
    1.2801710, 1.2812000, 1.2820710, 1.2823940, 1.2825720, 1.2860820, 1.2885750,
    1.2897730, 1.2914760, 1.2934590, 1.2971470, 1.3008370, 1.3017160, 1.3045220,
    1.3052980, 1.3053930, 1.3082150, 1.3112370, 1.3119160, 1.3119980, 1.3156860,
    1.3173860, 1.3194780, 1.3196130, 1.3213980, 1.3231680, 1.3247750, 1.3268590,
    1.3302420, 1.3302480, 1.3315140, 1.3344390, 1.3382320, 1.3407860, 1.3415830,
    1.3420180, 1.3477680, 1.3509370, 1.3516040, 1.3530900, 1.3547200, 1.3609120,
    1.3616290, 1.3636960, 1.3648880, 1.3675350, 1.3675540, 1.3685060, 1.3753530,
    1.3767680, 1.3780830, 1.3889330, 1.3911860, 1.3920020, 1.3956730, 1.3973910,
    1.3990370, 1.3997060, 1.4003760, 1.4009540, 1.4041980, 1.4090600, 1.4106690,
    1.4157070, 1.4171170, 1.4177600, 1.4179530, 1.4190210, 1.4223220, 1.4252400,
    1.4275900, 1.4289080, 1.4299430, 1.4387270, 1.4420430, 1.4442120, 1.4471120,
    1.4484790, 1.4534290, 1.4564910, 1.4577870, 1.4581640, 1.4600930, 1.4647410,
    1.4653680, 1.4677830, 1.4698700, 1.4734890, 1.4759800, 1.4773430, 1.4790890,
    1.4805180, 1.4846720, 1.4852910, 1.4864480, 1.4868440, 1.4877650, 1.4889450,
    1.4896180, 1.4912540, 1.4914700, 1.4918830, 1.4931410, 1.4952340, 1.4962800,
    1.4970120, 1.4988550, 1.5025360, 1.5039210, 1.5049860, 1.5056590, 1.5062770,
    1.5068210, 1.5070170, 1.5098750, 1.5120090, 1.5143540, 1.5147500, 1.5180690,
    1.5194050, 1.5201020, 1.5202180, 1.5208240, 1.5217760, 1.5220360, 1.5303840,
    1.5317880, 1.5323060, 1.5353150, 1.5359820, 1.5383300, 1.5407180, 1.5413170,
    1.5425990, 1.5431130, 1.5439750, 1.5493680, 1.5502810, 1.5508760, 1.5516930,
    1.5560580, 1.5586050, 1.5607390, 1.5633430, 1.5651170, 1.5668890, 1.5680450,
    1.5704270, 1.5757210, 1.5798270, 1.5817360, 1.5827540, 1.5840110, 1.5862590,
    1.5912250, 1.5944120, 1.5970180, 1.5973140, 1.5990530, 1.5994180, 1.6016820,
    1.6048380, 1.6088750, 1.6111040, 1.6133580, 1.6152010, 1.6168420, 1.6186420,
    1.6232190, 1.6249900, 1.6261880, 1.6295560, 1.6323380, 1.6368790, 1.6374460,
    1.6376790, 1.6398760, 1.6404590, 1.6421720, 1.6456730, 1.6495730, 1.6515580,
    1.6518560, 1.6518680, 1.6541320, 1.6549290, 1.6603630, 1.6631810, 1.6644430,
    1.6655580, 1.6676540, 1.6688130, 1.6689020, 1.6693530, 1.6699370, 1.6716500,
    1.6722390, 1.6794480, 1.6800020, 1.6800180, 1.6800570, 1.6822520, 1.6828430,
    1.6860770, 1.6932300, 1.6933350, 1.6955350, 1.6961330, 1.6977940, 1.6987140,
    1.7044260, 1.7069070, 1.7093030, 1.7115550, 1.7123480, 1.7126470, 1.7159200,
    1.7199390, 1.7204800, 1.7235710, 1.7252330, 1.7264200, 1.7268130, 1.7273740,
    1.7339520, 1.7344240, 1.7403830, 1.7473240, 1.7478990, 1.7479100, 1.7500400,
    1.7531770, 1.7605990, 1.7611280, 1.7664090, 1.7691810, 1.7737780, 1.7764030,
    1.7766390, 1.7795470, 1.7797780, 1.7807510, 1.7829640, 1.7868640, 1.7925930,
    1.7998580, 1.8007640, 1.8055470, 1.8127620, 1.8178990, 1.8193350, 1.8209750,
    1.8214510, 1.8214820, 1.8255780, 1.8286820, 1.8367850, 1.8383080, 1.8414410,
    1.8469150, 1.8509520, 1.8554840, 1.8584990, 1.8635130, 1.8710100, 1.8752210,
    1.8759920, 1.8774990, 1.8787820, 1.8804650, 1.8854230, 1.8883910, 1.8911140,
    1.8972700, 1.9007120, 1.9061080, 1.9129540, 1.9183170, 1.9251210, 1.9270610,
    1.9280640, 1.9303930, 1.9372130, 1.9400900, 1.9449130, 1.9492320, 1.9497830,
    1.9611780, 1.9645110, 1.9688770, 1.9730540, 1.9756480, 1.9796740, 1.9878660,
    1.9881530, 2.0115230, 2.0159570, 2.0332500, 2.0519420, 2.0579350, 2.0633810,
    2.0796300, 2.0796340, 2.0930840, 2.1011130, 2.1056180, 2.1219370, 2.1238140,
    2.1373130, 2.1443060, 2.1587020, 2.1655840, 2.1802270, 2.1875130, 2.2024370,
    2.2099850, 2.2303110, 2.2507840, 2.2686000, 2.2710750, 2.3119360, 2.3312970,
    2.3536850, 2.3734980, 2.3742870, 2.3923640, 2.4103070, 2.4126380, 2.4300470,
    2.4511570, 2.4625960, 2.4890840, 2.5264450, 2.5632640, 2.5923800, 2.5995640,
    2.6103560, 2.6353660, 2.6706900, 2.6986180, 2.7055550, 2.7399780, 2.7739760,
    2.8075630, 2.8407540, 2.8670310, 2.8768230, 2.9092270, 2.9316960, 2.9412740,
    2.9729770, 3.0043460, 3.0353910, 3.0569360, 3.0661230, 3.0965510, 3.0988120,
    3.1266830, 3.1565270, 3.1772520, 3.1860930, 3.1919740, 3.2153880, 3.2444180,
    3.2731920, 3.3017150, 3.3328080, 3.3608260, 3.3886130, 3.4161730, 3.4189170,
    3.4435140, 3.4706390, 3.4868130, 3.4975540, 3.5507730, 3.6032070, 3.6548900,
    3.6574550, 3.7058520, 3.7310720, 3.7461230, 3.7955240, 3.9886240, 4.0003630,
    4.2614590, 4.4340360, 4.6809920, 4.7288360, 4.8386340};
/// Elastic π⁻N⁺ cross section contributions from decays.
///
/// These need to be subtracted from the interpolation of the PDG data on
/// elastic cross sections. This data was generated using the SMASH analysis
/// suite and should be updated when strange resonances are changed or added.
const std::initializer_list<double> PIMINUSP_RES_SIG = {
    2.727771,  3.571329,  4.034613,  4.882370,  4.975210,  5.117328,  6.891753,
    7.205112,  7.746671,  8.247897,  9.456023,  9.485245,  9.711213,  10.112155,
    11.241098, 11.944095, 12.723382, 13.144982, 13.246803, 13.598222, 14.798669,
    15.586762, 16.480923, 16.893047, 17.035961, 17.481312, 18.242040, 18.484793,
    18.928599, 19.122060, 19.375409, 19.642156, 19.673564, 20.048232, 20.058785,
    20.099053, 20.118021, 20.253818, 20.307050, 20.336885, 20.338216, 20.338463,
    20.326376, 20.215612, 20.187220, 20.108831, 20.085299, 19.929646, 19.896012,
    19.744140, 19.348620, 19.294422, 19.183131, 18.979071, 18.802318, 18.407836,
    18.174742, 18.166186, 17.527526, 17.376048, 17.362015, 17.222750, 16.864140,
    16.526686, 16.391817, 16.265062, 16.249887, 16.204412, 15.841202, 15.559769,
    15.313297, 15.126607, 15.055993, 14.725313, 14.695755, 14.560464, 14.471319,
    14.321746, 14.252233, 13.937248, 13.864858, 13.744437, 13.585398, 13.434918,
    13.251312, 13.150076, 12.968950, 12.921955, 12.624439, 12.345808, 12.074788,
    12.046720, 11.940788, 11.853593, 11.821900, 11.804500, 11.481213, 11.273034,
    11.179088, 11.052115, 10.913921, 10.686029, 10.491401, 10.449671, 10.328092,
    10.297759, 10.294122, 10.194918, 10.106365, 10.088903, 10.086866, 10.009062,
    9.982275,  9.955883,  9.954419,  9.937770,  9.926040,  9.919670,  9.917476,
    9.925773,  9.925804,  9.932451,  9.954887,  9.999073,  10.035233, 10.047444,
    10.054284, 10.154801, 10.215371, 10.228372, 10.257565, 10.289514, 10.404053,
    10.416507, 10.451255, 10.469744, 10.506548, 10.506807, 10.518891, 10.583211,
    10.588553, 10.591846, 10.551600, 10.528871, 10.519808, 10.468089, 10.437574,
    10.407237, 10.394638, 10.381921, 10.370850, 10.307754, 10.204285, 10.171181,
    10.069177, 10.040061, 10.027313, 10.023558, 10.003399, 9.948213,  9.908627,
    9.883310,  9.872680,  9.866237,  9.877111,  9.919510,  9.961662,  10.035629,
    10.078320, 10.277399, 10.432873, 10.507124, 10.529744, 10.652615, 10.997871,
    11.049584, 11.260341, 11.456495, 11.825114, 12.094558, 12.246925, 12.447007,
    12.613749, 13.107598, 13.181143, 13.361513, 13.418153, 13.534148, 13.667426,
    13.738379, 13.898388, 13.918290, 13.955613, 14.063025, 14.218190, 14.284287,
    14.326422, 14.417247, 14.531076, 14.550199, 14.554711, 14.553183, 14.549119,
    14.543450, 14.540937, 14.476394, 14.395865, 14.277562, 14.253650, 14.025681,
    13.921779, 13.865112, 13.855605, 13.804923, 13.723295, 13.700529, 12.899844,
    12.759289, 12.707480, 12.406185, 12.339343, 12.108105, 11.880783, 11.825126,
    11.708153, 11.661998, 11.585849, 11.139385, 11.070268, 11.026165, 10.967131,
    10.676650, 10.527975, 10.416330, 10.296090, 10.224045, 10.160037, 10.122722,
    10.056953, 9.963186,  9.940076,  9.944406,  9.950619,  9.962028,  9.992798,
    10.107907, 10.216027, 10.324590, 10.338056, 10.421938, 10.440565, 10.563991,
    10.758763, 11.046275, 11.219098, 11.406161, 11.568358, 11.719190, 11.891153,
    12.356378, 12.545937, 12.676903, 13.054393, 13.374187, 13.902515, 13.968362,
    13.995323, 14.248969, 14.315704, 14.510032, 14.894839, 15.294828, 15.483193,
    15.510420, 15.511442, 15.708221, 15.771604, 16.129790, 16.258504, 16.302184,
    16.332919, 16.371389, 16.381668, 16.382132, 16.383765, 16.384106, 16.373504,
    16.365846, 16.094248, 16.061361, 16.060351, 16.057988, 15.912247, 15.869115,
    15.604058, 14.881278, 14.869656, 14.615127, 14.544006, 14.343831, 14.231480,
    13.518719, 13.205142, 12.904069, 12.624201, 12.524907, 12.487428, 12.080104,
    11.596000, 11.532586, 11.165458, 10.970463, 10.834819, 10.790496, 10.727909,
    10.033831, 9.986213,  9.422328,  8.843849,  8.799297,  8.798477,  8.637944,
    8.413399,  7.933449,  7.901865,  7.604069,  7.459614,  7.237359,  7.119512,
    7.109214,  6.986207,  6.976764,  6.937338,  6.850621,  6.707226,  6.516829,
    6.307029,  6.283124,  6.165071,  6.009385,  5.914360,  5.889920,  5.863003,
    5.855379,  5.854890,  5.792956,  5.750120,  5.653524,  5.637576,  5.606512,
    5.557455,  5.524890,  5.491104,  5.470122,  5.436638,  5.387926,  5.360343,
    5.355253,  5.345255,  5.336686,  5.325224,  5.289797,  5.267817,  5.247120,
    5.194926,  5.164390,  5.113899,  5.046572,  4.990838,  4.916877,  4.895353,
    4.884239,  4.858425,  4.779720,  4.745950,  4.688505,  4.637428,  4.630918,
    4.491197,  4.449922,  4.395158,  4.342866,  4.310165,  4.260304,  4.156622,
    4.153074,  3.863210,  3.807705,  3.599068,  3.381739,  3.313336,  3.252640,
    3.078632,  3.078586,  2.942304,  2.865732,  2.822595,  2.680616,  2.664734,
    2.559076,  2.508390,  2.409106,  2.363398,  2.270298,  2.225212,  2.133668,
    2.087464,  1.964694,  1.842465,  1.737800,  1.723454,  1.497293,  1.398065,
    1.291078,  1.202957,  1.199580,  1.125166,  1.056454,  1.047924,  0.986448,
    0.917631,  0.882818,  0.808313,  0.716179,  0.638085,  0.583949,  0.571516,
    0.553453,  0.514354,  0.465001,  0.430238,  0.422140,  0.384707,  0.351896,
    0.323011,  0.297502,  0.279222,  0.272812,  0.253013,  0.240484,  0.235418,
    0.219741,  0.205756,  0.193259,  0.185301,  0.182074,  0.172035,  0.171327,
    0.163000,  0.154840,  0.149582,  0.147433,  0.146033,  0.140669,  0.134455,
    0.128697,  0.123321,  0.117770,  0.112988,  0.108418,  0.104023,  0.103592,
    0.099782,  0.095677,  0.093278,  0.091704,  0.084136,  0.077092,  0.070601,
    0.070291,  0.064685,  0.061942,  0.060365,  0.055497,  0.040625,  0.039905,
    0.027723,  0.022456,  0.017122,  0.016299,  0.014606};

/// An interpolation that gets lazily filled using the PIMINUSP_RES data.
static std::unique_ptr<InterpolateDataSpline>
    piminusp_elastic_res_interpolation = nullptr;

/// PDG data on pi+ p elastic cross section: momentum in lab frame.
const std::initializer_list<double> PIPLUSP_ELASTIC_P_LAB = {
    0.09875, 0.13984, 0.14956, 0.33138, 0.378,   0.408,   0.4093,  0.427,
    0.42736, 0.453,   0.471,   0.49008, 0.509,   0.5214,  0.53155, 0.547,
    0.57281, 0.574,   0.586,   0.5891,  0.59,    0.591,   0.6139,  0.625,
    0.625,   0.635,   0.645,   0.645,   0.657,   0.65793, 0.6753,  0.687,
    0.698,   0.7,     0.70692, 0.725,   0.72628, 0.72628, 0.752,   0.752,
    0.77714, 0.807,   0.809,   0.82,    0.82586, 0.85,    0.895,   0.895,
    0.9,     0.91,    0.93926, 0.945,   0.945,   0.995,   0.995,   1.0029,
    1.02,    1.04,    1.04,    1.0402,  1.05,    1.12091, 1.12091, 1.195,
    1.207,   1.2217,  1.232,   1.375,   1.3925,  1.44384, 1.46,    1.493,
    1.5,     1.53,    1.585,   1.585,   1.6,     1.68881, 1.69,    1.77,
    1.8,     1.869,   1.9,     1.99,    2.,      2.0199,  2.07,    2.077,
    2.11,    2.19,    2.3,     2.35,    2.5,     2.77,    2.9,     3.,
    3.05,    3.56,    3.63,    3.65,    3.67,    3.9,     4.,      5.,
    5.,      6.,      6.8001,  8.,      8.04,    8.8,     10.8,    11.7,
    12.8,    14.8,    16.,     16.2,    16.7,    29.,     32.1,    43.,
    50.,     60.,     70.,     100.,    140.,    147.,    175.,    200.,
    250.};

/// PDG data on pi+ p elastic cross section: cross section
const std::initializer_list<double> PIPLUSP_ELASTIC_SIG = {
    6.15,  15.8,  20.4,  140.9, 91.6,  71.6,  67.5,  57.3,  60.19, 46.6,  40.8,
    38.74, 30.6,  29.6,  30.59, 24.74, 24.31, 28.16, 19.83, 20.4,  20.64, 20.63,
    19.55, 15.75, 18.5,  17.2,  14.85, 16.16, 14.71, 15.32, 14.38, 12.2,  12.93,
    12.96, 12.17, 11.6,  11.06, 11.5,  10.62, 10.55, 8.82,  9.36,  8.97,  9.1,
    8.02,  8.38,  8.37,  8.14,  11.1,  9.3,   11.,   10.32, 9.87,  11.7,  11.15,
    12.05, 11.1,  12.37, 11.83, 10.3,  12.8,  14.54, 15.3,  14.3,  13.59, 12.6,
    12.3,  17.87, 16.5,  19.31, 18.73, 16.68, 13.8,  15.86, 16.7,  16.7,  15.05,
    13.04, 13.57, 12.46, 12.3,  11.81, 10.9,  9.84,  10.6,  9.1,   9.52,  9.46,
    9.44,  9.15,  8.45,  10.2,  6.9,   7.7,   8.3,   7.84,  7.02,  6.93,  7.07,
    6.88,  7.15,  6.5,   6.4,   5.85,  5.79,  5.33,  5.47,  4.9,   4.9,   5.02,
    4.75,  4.2,   4.54,  4.46,  4.21,  4.21,  3.98,  3.19,  3.37,  3.16,  3.29,
    3.1,   3.35,  3.3,   3.39,  3.24,  3.37,  3.17,  3.3};

/// An interpolation that gets lazily filled using the PIPLUSP_ELASTIC_SIG data.
static std::unique_ptr<InterpolateDataLinear<double>>
    piplusp_elastic_interpolation = nullptr;

/// PDG data on pi+ p to Sigma+ K+ cross section: momentum in lab frame.
const std::initializer_list<double> PIPLUSP_SIGMAPLUSKPLUS_P_LAB = {
    1.041, 1.105, 1.111, 1.15,  1.157, 1.17,  1.195, 1.206, 1.218, 1.222, 1.265,
    1.28,  1.282, 1.328, 1.34,  1.377, 1.39,  1.41,  1.419, 1.43,  1.456, 1.49,
    1.508, 1.518, 1.549, 1.55,  1.58,  1.582, 1.614, 1.63,  1.68,  1.687, 1.7,
    1.712, 1.76,  1.77,  1.775, 1.808, 1.84,  1.879, 1.906, 1.95,  1.971, 1.997,
    2.067, 2.08,  2.099, 2.152, 2.197, 2.241, 2.291, 2.344, 2.379, 2.437, 2.473,
    2.77,  3.23,  3.71,  4.,    5.,    5.5,   7.,    10.3,  12.,   16.};

/// PDG data on pi+ p to Sigma+ K+ section: cross section
const std::initializer_list<double> PIPLUSP_SIGMAPLUSKPLUS_SIG = {
    0.034,  0.146,  0.144, 0.214,  0.248, 0.205, 0.24,  0.214, 0.242, 0.25,
    0.278,  0.34,   0.369, 0.412,  0.4,   0.467, 0.44,  0.49,  0.523, 0.51,
    0.662,  0.529,  0.692, 0.545,  0.604, 0.53,  0.53,  0.465, 0.494, 0.47,
    0.505,  0.434,  0.47,  0.436,  0.38,  0.415, 0.418, 0.393, 0.405, 0.332,
    0.37,   0.31,   0.338, 0.298,  0.3,   0.29,  0.28,  0.273, 0.26,  0.25,
    0.23,   0.242,  0.22,  0.217,  0.234, 0.165, 0.168, 0.104, 0.059, 0.059,
    0.0297, 0.0371, 0.02,  0.0202, 0.0143};

/**
 * An interpolation that gets lazily filled using the
 * PIPLUSP_SIGMAPLUSKPLUS_SIG data.
 */
static std::unique_ptr<InterpolateDataLinear<double>>
    piplusp_sigmapluskplus_interpolation = nullptr;

/// Center-of-mass energy.
const std::initializer_list<double> PIPLUSP_RES_SQRTS = {
    1.1173610, 1.1241380, 1.1358180, 1.1371030, 1.1380990, 1.1424360, 1.1457360,
    1.1465540, 1.1482410, 1.1566800, 1.1579910, 1.1583200, 1.1604390, 1.1665900,
    1.1679230, 1.1685180, 1.1749480, 1.1757030, 1.1768490, 1.1773210, 1.1786720,
    1.1829250, 1.1881750, 1.1884820, 1.1892660, 1.1892730, 1.1895390, 1.1913820,
    1.1924340, 1.1940070, 1.1947940, 1.1955810, 1.1984320, 1.1998050, 1.2002920,
    1.2010810, 1.2018580, 1.2048200, 1.2049860, 1.2087440, 1.2090200, 1.2096600,
    1.2101220, 1.2112110, 1.2119900, 1.2127620, 1.2143280, 1.2156390, 1.2158600,
    1.2166330, 1.2189530, 1.2190910, 1.2204730, 1.2212600, 1.2231810, 1.2251030,
    1.2255180, 1.2260010, 1.2262570, 1.2266370, 1.2274050, 1.2294580, 1.2300810,
    1.2308410, 1.2342780, 1.2350390, 1.2358000, 1.2363740, 1.2398320, 1.2412150,
    1.2426250, 1.2428740, 1.2456470, 1.2508920, 1.2515830, 1.2531650, 1.2602810,
    1.2605570, 1.2606400, 1.2619360, 1.2621290, 1.2688130, 1.2708920, 1.2722680,
    1.2743520, 1.2792100, 1.2812000, 1.2825720, 1.2908540, 1.2914760, 1.2921600,
    1.3000800, 1.3017160, 1.3112370, 1.3119160, 1.3174950, 1.3196130, 1.3213980,
    1.3247750, 1.3252880, 1.3315140, 1.3382320, 1.3415830, 1.3442600, 1.3463790,
    1.3516040, 1.3602490, 1.3609120, 1.3616290, 1.3648880, 1.3657620, 1.3698540,
    1.3761100, 1.3780830, 1.3808030, 1.3889330, 1.3911860, 1.3943310, 1.3944480,
    1.4041980, 1.4077270, 1.4106690, 1.4157070, 1.4171170, 1.4210000, 1.4282490,
    1.4293040, 1.4299430, 1.4328100, 1.4341490, 1.4419790, 1.4420430, 1.4465240,
    1.4490100, 1.4534290, 1.4536500, 1.4553190, 1.4647410, 1.4672450, 1.4759800,
    1.4824160, 1.4855750, 1.4865280, 1.4877650, 1.4912540, 1.4956530, 1.4988550,
    1.4994080, 1.5025360, 1.5093620, 1.5098750, 1.5118750, 1.5141410, 1.5208240,
    1.5279280, 1.5321730, 1.5323060, 1.5347140, 1.5425990, 1.5431130, 1.5462770,
    1.5508760, 1.5510910, 1.5526630, 1.5564860, 1.5633430, 1.5651170, 1.5668890,
    1.5702910, 1.5715220, 1.5757210, 1.5786540, 1.5862590, 1.5961540, 1.5973140,
    1.6058020, 1.6129040, 1.6139380, 1.6186080, 1.6284110, 1.6323380, 1.6404590,
    1.6416010, 1.6467170, 1.6498710, 1.6518170, 1.6529970, 1.6581190, 1.6626220,
    1.6666740, 1.6671250, 1.6688130, 1.6722390, 1.6822130, 1.6850290, 1.6961330,
    1.6981660, 1.7011210, 1.7175570, 1.7204800, 1.7235710, 1.7392590, 1.7425210,
    1.7500400, 1.7607570, 1.7632420, 1.7713600, 1.7766390, 1.7790240, 1.7903550,
    1.7924370, 1.7946710, 1.8028500, 1.8096730, 1.8194380, 1.8235340, 1.8286820,
    1.8301710, 1.8434770, 1.8494390, 1.8536470, 1.8642640, 1.8700130, 1.8737520,
    1.8787820, 1.8824500, 1.8908620, 1.8912220, 1.8936740, 1.8977620, 1.9035830,
    1.9158620, 1.9182680, 1.9183170, 1.9280640, 1.9328680, 1.9381770, 1.9421740,
    1.9517610, 1.9568860, 1.9613690, 1.9706840, 1.9756480, 1.9777850, 1.9801460,
    1.9893750, 1.9983050, 2.0031410, 2.0085470, 2.0341700, 2.0359620, 2.0383030,
    2.0561620, 2.0633810, 2.0810910, 2.0815680, 2.0930840, 2.1104420, 2.1256810,
    2.1276610, 2.1386260, 2.1520030, 2.1603920, 2.1736890, 2.1767910, 2.1802270,
    2.1815150, 2.1917870, 2.2020120, 2.2058340, 2.2164220, 2.2478710, 2.2625000,
    2.2932290, 2.3035850, 2.3337060, 2.3479530, 2.3532870, 2.3536850, 2.3734980,
    2.3875670, 2.4126380, 2.4265400, 2.4511570, 2.4686940, 2.4890840, 2.5064020,
    2.5264450, 2.5435540, 2.5449200, 2.5632640, 2.5801700, 2.5995640, 2.6162720,
    2.6353660, 2.6554170, 2.6706900, 2.7055550, 2.7251500, 2.7399780, 2.7604270,
    2.7739760, 2.7965000, 2.8075630, 2.8407540, 2.8660730, 2.8768230, 2.8995430,
    2.9092270, 2.9339980, 2.9412740, 2.9729770, 3.0003830, 3.0043460, 3.0353910,
    3.0569360, 3.0653340, 3.0661230, 3.0965510, 3.1266830, 3.1565270, 3.1860930,
    3.2153880, 3.2444180, 3.2731920, 3.3017150, 3.3328080, 3.3608260, 3.3886130,
    3.4161730, 3.4189170, 3.4435140, 3.4706390, 3.4868130, 3.4975540, 3.5242640,
    3.5507730, 3.6032070, 3.6548900, 3.7058520, 3.7461230, 3.9246250, 3.9886240,
    4.4000620, 4.4340360, 4.8192100, 4.8386340};
/// Elastic π⁺N⁺ cross section contributions from decays.
///
/// These need to be subtracted from the interpolation of the PDG data on
/// elastic cross sections. This data was generated using the SMASH analysis
/// suite and should be updated when strange resonances are changed or added.
const std::initializer_list<double> PIPLUSP_RES_SIG = {
    6.022200,   10.038871,  17.441789,  18.453186,  19.268833,  23.150742,
    26.498913,  27.386442,  29.291426,  40.496896,  42.513388,  43.030735,
    46.492651,  57.799559,  60.507365,  61.744072,  76.313703,  78.161300,
    81.021590,  82.218538,  85.698091,  97.146898,  112.028626, 112.912122,
    115.173370, 115.193049, 115.960670, 121.274197, 124.294133, 128.769360,
    130.983774, 133.177973, 140.888488, 144.431558, 145.658198, 147.606924,
    149.472691, 156.080287, 156.422132, 163.346112, 163.784724, 164.765987,
    165.439046, 166.911553, 167.863958, 168.723549, 170.204028, 171.170636,
    171.308879, 171.736988, 172.504248, 172.525712, 172.593592, 172.514719,
    171.976396, 170.970613, 170.695374, 170.349193, 170.155496, 169.854117,
    169.198023, 167.145300, 166.443383, 165.538602, 160.877328, 159.734532,
    158.557081, 157.647233, 151.838017, 149.388278, 146.835348, 146.380094,
    141.238822, 131.363425, 130.066638, 127.112163, 114.248942, 113.768689,
    113.624927, 111.391996, 111.062599, 100.192520, 97.031881,  94.998359,
    92.007883,  85.447008,  82.920313,  81.231779,  71.911700,  71.268389,
    70.570322,  63.112285,  61.706466,  54.325885,  53.847853,  50.135114,
    48.820524,  47.750605,  45.816616,  45.532806,  42.281080,  39.137415,
    37.695516,  36.598844,  35.763674,  33.820390,  30.931186,  30.724707,
    30.504071,  29.529065,  29.275637,  28.130677,  26.504830,  26.021034,
    25.375549,  23.583128,  23.119865,  22.496127,  22.473352,  20.695879,
    20.104852,  19.631763,  18.860276,  18.652732,  18.099067,  17.131947,
    16.997986,  16.917638,  16.564509,  16.403533,  15.511183,  15.504274,
    15.029072,  14.775835,  14.342938,  14.321885,  14.164381,  13.330021,
    13.123245,  12.447103,  11.992319,  11.781862,  11.720067,  11.640951,
    11.424298,  11.165000,  10.986072,  10.955990,  10.790078,  10.454817,
    10.431082,  10.340373,  10.241359,  9.973671,   9.727943,   9.600744,
    9.597007,   9.531720,   9.352521,   9.342579,   9.286710,   9.220119,
    9.217412,   9.198613,   9.160760,   9.116223,   9.108690,   9.102419,
    9.092648,   9.089575,   9.079161,   9.070168,   9.030578,   8.920964,
    8.903042,   8.745744,   8.587201,   8.563189,   8.451960,   8.222839,
    8.135503,   7.964442,   7.941527,   7.840761,   7.779875,   7.743029,
    7.721027,   7.623950,   7.539104,   7.460968,   7.452091,   7.418960,
    7.352368,   7.151795,   7.095082,   6.872018,   6.831788,   6.775597,
    6.499647,   6.458988,   6.421107,   6.281186,   6.267541,   6.253040,
    6.283141,   6.299494,   6.373666,   6.441742,   6.477662,   6.688097,
    6.733830,   6.785977,   6.995237,   7.190716,   7.500324,   7.636831,
    7.812619,   7.864044,   8.315853,   8.503866,   8.626044,   8.879732,
    8.974573,   9.017728,   9.051747,   9.056952,   9.008060,   9.004429,
    8.972971,   8.906696,   8.781147,   8.419200,   8.335117,   8.333399,
    7.962610,   7.768317,   7.542001,   7.372803,   6.958122,   6.731881,
    6.543614,   6.150650,   5.943860,   5.858059,   5.766199,   5.407669,
    5.077684,   4.907944,   4.721214,   3.920965,   3.868515,   3.802953,
    3.320499,   3.139608,   2.718237,   2.708015,   2.448100,   2.100843,
    1.843201,   1.814261,   1.669739,   1.527892,   1.453866,   1.356321,
    1.336277,   1.315603,   1.308211,   1.252123,   1.203159,   1.186153,
    1.142877,   1.035775,   0.993647,   0.915748,   0.892059,   0.828894,
    0.801562,   0.791697,   0.790969,   0.756008,   0.732619,   0.693611,
    0.673343,   0.639635,   0.617218,   0.592709,   0.573136,   0.551832,
    0.534734,   0.533411,   0.516219,   0.501298,   0.485222,   0.472219,
    0.458270,   0.444610,   0.434844,   0.414467,   0.404081,   0.396683,
    0.387082,   0.381073,   0.371641,   0.367241,   0.354831,   0.346043,
    0.342460,   0.335135,   0.332105,   0.324567,   0.322406,   0.313236,
    0.305601,   0.304518,   0.296208,   0.290622,   0.288484,   0.288284,
    0.280729,   0.273519,   0.266620,   0.259982,   0.253547,   0.247248,
    0.241018,   0.234792,   0.227884,   0.221501,   0.214989,   0.208334,
    0.207660,   0.201536,   0.194611,   0.190406,   0.187587,   0.180499,
    0.173394,   0.159321,   0.145738,   0.132952,   0.123434,   0.088815,
    0.079356,   0.042881,   0.041067,   0.026625,   0.026107};

/// A null interpolation that gets filled using the PIPLUSP_RES data
static std::unique_ptr<InterpolateDataSpline>
    piplusp_elastic_res_interpolation = nullptr;
}  // namespace smash

#endif  // SRC_INCLUDE_PARAMETRIZATIONS_DATA_H_
