% Authors: Hannah Schug, Anze Zupanic 
% Dbendorf, 05.10.2018
%
% This run sheet is designed to execute the kinetic models for
% cell-free and cell-containing experiments for the following thesis
% chapter:
%
% "An intestinal fish cell barrier model to assess transfer of organic
% chemicals in vitro - an experimental and computational study"
%
% The models are built based on ordinary differential equations (ODE) in
% Potterswheel, a matlab toolbox for dynamic modelling (version 4.0).
% For more information please refer to www.potterswheel.de.
%
%% Model Description
%
% All the models in this family describe the dynamics of organic 
%   chemicals in the TransFEr exposure chamber. The chamber consist of 
%   two main compartments, the apical (=upper) and basolateral (=lower) 
%   compartment. Both compartments are separated by a permeable membrane 
%   which in the following is referred to as "filter" to avoid confusion 
%   with the cell membrane. The chemicals are applied to the apical compartment 
%   either via active or passive dosing. The chemicals can bind to the 
%   plastic and to the filter, can passage into the cells or through the 
%   space between the cells to the basolateral compartment. Within the 
%   cells, the chemicals can be bound or biotransformed. Experiments were 
%   conducted in cell-free and cell-containing setups. In total, 4 models 
%   describing the chemical dynamics in cell-free conditions were built. 
%   For cell-containing, in total 14 sets of models are built. These 
%   describe the chemical dynamics in the presence of a cell layer and 
%   contain different mechanisms on how the chemicals can transfer across
%   the cell monolayer. Each model set was built in the absence and in the
%   presence of a chemical sink in the basolateral compartment.

% Cell-free models:
% Model 1: - Permeation from the apical to the basolateral compartment 
%          - active dosing
% Model 2: - Permeation from the apical to the basolateral compartment
%          - passive dosing
% Model 3: - Permeation from the apical to the basolateral compartment
%          - active dosing
%          - interaction with plastic and filter
% Model 4: - Permeation from the apical to the basolateral compartment
%          - passive dosing
%          - interaction with plastic and filter
%
% Cell-containing models:
% Note: Interaction with plastic and filter was not modelled for the cell-containing models. 
%       Model 1 (for active dosing) and Model 2 (for passive
%       dosing) were further developed including potential mechanisms of
%       chemical transfer across the cell layer.
%
% In the following the abbreviations will be briefly explained:
%
% Para:     "Paracellular". Diffusion from the apical to the 
%           basolateral compartment through the space between the cells 
%           along the concentration gradient
%
% Cyto:     "Cytosolic". Diffusion from the apical across the apical cell 
%           membrane into the cytosol and permeation from the cytosol 
%           across the basolateral cell membrane and the filter to the 
%           basolateral compartment along the concentration gradient
%
% Bind:     "Binding". Intracellular binding of the chemical. Does not 
%           result in loss of the chemical.
%
% Biotrans: "Biotransformation". Intracellular biotransformation of the
%           chemical. Does result in loss of the chemical.
%
% Lateral:  "Lateral". Diffusion from the apical into the apical cell
%           membrane and diffusion from the basolateral cell membrane 
%           through the filter into the basolateral compartment along the 
%           concentration gradient
%
% Uptake:   "Active uptake". Active transport of from the apical 
%           compartment into the cytosol of the cell. Can occur against 
%           the concentration gradient.
%
% Efflux:   "Active efflux". Active transport from the cytosol of the cell
%           across the filter into the basolateral compartment. Can occur 
%           against the concentration gradient. 

%% Clearance of the workspace
clear
clc
pwCloseFigures;
pwClear;

%% Configurations for potterswheel execution
% only needs to be done once in the beginning
config = pwGetConfig;
config.integration.integrator = 14; % change the integrator for 64 bit systems! cvodes
config.integration.calcJacobian = true;
config.integration.useJacobian = true;
% choose your optimization methods: 1 - fminsearch, 2 trust region, 3 GA, 
% 4 SA 5 smarq 6 ssmis  7 ssmdn
config.optimization.method = 2;     
config.optimization.fitInLogParameterSpace = true;
config.optimization.calcJacobian = true;
config.optimization.useJacobian = true; 
config.optimization.levelOfHistoryStorage = 3;   % to store trajectories during fitting
config.integration.calcSensitivities = true;
config.model.xMaxValue = 1e7;
pwSetConfig(config);

%% Model and Data
% load the respective model using the complete path to model
pwAddModel('insert_the_path_and_name_of_model_file_here'); 
% load the respective data set using the complete path to the data set
% the data need to be stored in a special format for the potterswheel , see attached files. 
pwAddData('insert_the_path_and_name_of_excel_sheet_here');

% Combine the data with the model
% an extra window opens: assign the data to the respective input
% variables. Save the assignment for next times. After confirming the 
% assignments, the data will be plotted. Confirm the correctness of the
% data structure by visual inspection of the graphs
pwCombine;

%% Fitting
% setting that individual fittings are not shown during the fitting process
pwShowFitting(false); 
% fit the model 100 times
% at the end of the fitting process, an extra window opens showing the
% result of the fit.
% save the chi-square and akaike information criterion (AIC) separately
pwF2(100);

% to extract the best fit from last sequence after closing the extra window
pwGetBestFit

%% Saving of parameters 
% get all the fits
[IDs, values] = pwGetFitParameters();
fit = pwFitHistoryGetFits;
% save them to an excel file
pwSaveFitsToFile('fittedParametersOfSingleFit.xls', fit);

% Note: Saving the fit is creating a large file. After saving the
% parameters of the single fit, the "fit" matrix can be deleted again from 
% the workspace

%% Confidence intervals
% the confidence intervals are determined by the profile likelihood (PLE)
% method. The intervals can be set to different confidence levels by 
% commenting / uncommenting the following lines
% initialize the parameters of the profile likelihood
config = pwGetConfig;
config.analyses.PLE.steppingMode = 1; % direct stepping mode
config.analyses.PLE.confLevelInUnitsOfStd = 1.96; % 95 confidence intervals
% config.analyses.PLE.confLevelInUnitsOfStd = 2.576; % 99% confidence intervals
% config.analyses.PLE.confLevelInUnitsOfStd = 2.807; % 99.5% confidence intervals
% config.analyses.PLE.confLevelInUnitsOfStd = 3.291; % 99.9% confidence intervals
config.analyses.PLE.allowBetterOptimum = 1
pwSetConfig(config);

% open the profile likelihood GUI
% to run the PLE, click in the new opened window on "Analysis" and 
% "Estimate profile likelihood" depending on the number of parameters 
% The PLE might take a while. In case the PLE gets stuck, a new PLE can be
% run, leaving the problematic parameter out of the analysis by unticking
% the box
pwPLEgui;

% Save the PLE-Data
% c = chiSquareValues
% p = parameter Values
% t = threshold
pwPleGetProfileLikelihoodData
% run as many lines as parameters are in the current model
[c1,p1,t1] = pwPleGetProfileLikelihoodData(1); %for parameter 1
[c2,p2,t2] = pwPleGetProfileLikelihoodData(2); %for parameter 2
[c3,p3,t3] = pwPleGetProfileLikelihoodData(3); %for parameter 3
[c4,p4,t4] = pwPleGetProfileLikelihoodData(4); %for parameter 4
[c5,p5,t5] = pwPleGetProfileLikelihoodData(5); %for parameter 5
[c6,p6,t6] = pwPleGetProfileLikelihoodData(6); %for parameter 6
[c7,p7,t7] = pwPleGetProfileLikelihoodData(7); %for parameter 7
[c8,p8,t8] = pwPleGetProfileLikelihoodData(8); %for parameter 8

% Re-plot the PLE data
figure(11); 
semilogx(p2,c2);
hold on;
semilogx(p2,t2*ones(size(p2)),'r--')

%% Save parameters into model file 
% always the best fit will be saved into model file named for this chemical
% can be opened later again to look at the results
pwSaveParametersIntoModelFile

save Insert_useful_name_here
