% Authors: Hannah Schug, Anze Zupanic 
% Dbendorf, 05.10.2018
%
% The model is built based on ordinary differential equations (ODE) in
% Potterswheel, a matlab toolbox for dynamic modelling (version 4.0).
% The returned variable "m" contains all required information of the model.
% For more model description and to execute this model file, 
% please use the provided run sheet.
% For more information visit www.potterswheel.de

function m = Model_1B_laterStart_CELL_sink_para_lateral()

% get empty model
m= pwGetEmptyModel();

%% Meta information
m.name        = 'epithelialCellBarrier';
m.description = 'a';
m.authors     = {'Anze Zupanic; Hannah Schug'};
m.dates       = {'2017-07-21'};
m.modelFormat = 3.0;

%% Default sampling time points
m.t = 0:0.01:72;

%% X - Dynamic variables 
% % m = pwAddX(m, *ID, *startValue, fitSetting, minValue, maxValue, unit, compartment, name, description, typeOfStartValue, designerProps, classname)
% ID:              Unique name, e.g. 'ProtA'
% startValue:      The value of X at integration start time (default 0)
% fitSetting:      'global', 'local' (default), 'fix' (startValue will be fitted globally or locally or is fixed)
% min/maxValue:    Minimum and maximum of the startValue during fitting
% unit:            Unit of X
% compartment:     Compartment of X. Default: first given compartment
% typeOfStartValue: 'amount' or 'concentration' (default)
% designerProps:   Designer information, e.g. position and coloring
% classname:       Subgroup of the X variable, e.g. 'protein.receptor'.

% Notation example
% m = pwAddX(m, 'name_variable'      ,  starting_amount, 'fix', 0);

m = pwAddX(m, 'conc_api' ,  0, 'fix', 0); % concentration in the apical compartment [ug/ml]
m = pwAddX(m, 'conc_baso',  0, 'fix', 0); % concentration in the basolateral compartment [ug/ml]
m = pwAddX(m, 'conc_cell',  0, 'fix', 0); % concentration associated with the cell layer [ug/ml]
m = pwAddX(m, 'conc_cell_membrane',  0, 'fix', 0); % concentration associated with the membrane of the cell layer [ug/ml]
m = pwAddX(m, 'conc_twister',  0, 'fix', 0); % concentration associated with the sink [ug/ml]

%% K - Dynamic parameters (reaction rates, etc)
% m = pwAddK(m, *ID, *value, fitSetting, minValue, maxValue, unit, name, description)
% ID:         ID of the parameter
% value:      Value of the parameter
% fitSetting: 'global', 'local', 'fix' (during fitting)
% min/maxValue Limits for the parameter during parameter calibration

% the are the parameter values which need to be changed for each chemical 
% insert prior optima and lower and upper levels
m = pwAddK(m, 'permeability', 0.0507   , 'local', 5e-15, 0.065926594); % permeability of the filter [cm/h]
m = pwAddK(m, 'k_a_twister' , 177.999999383929  , 'local', 101    , 178   ); % Rate of chemical absorption to stir bar PDMS
m = pwAddK(m, 'Klipw'          , 42747.8006410584   , 'local', 1820     , 726000); % partition coefficient between liposome and water [unitless]
m = pwAddK(m, 'P_lateral_api'  , 1.1713480879949e-06, 'local', 1e-15    , 1e+15 ); % lateral permeability from apical side to half of cell height [cm/h]
m = pwAddK(m, 'P_lateral_baso' , 26381213358.6887   , 'local', 1e-15    , 1e+15 ); % lateral permeability from half of cell to basolateral side [cm/h]

% these are constants
m = pwAddK(m, 'V_api'       , 3.5                 , 'fix'  ); % Volume of the apical compartment [ml]
m = pwAddK(m, 'V_baso'      , 12                  , 'fix'  ); % Volume of the basolateral compartment [ml]
m = pwAddK(m, 'A_mem'       , 4.524               , 'fix'  ); % Surface area of the filter [cm2]
m = pwAddK(m, 'V_cell'       , 6.97e-04              , 'fix'  ); % Volume of the cell layer [ml]
m = pwAddK(m, 'V_cell_membrane', 2.09e-06           , 'fix'); % Volume of the membrane of the cell layer [ml]
m = pwAddK(m, 'V_twister'   , 0.0393            , 'fix'  ); % Volume of the PDMS of the chemical sink [ml]

%% V - reaction rates and fluxes 
% reaction rates are pre-defined here to make writing of the ODEs easier
% diffusion through insert pores
m = pwAddV(m, 'D_I', 'permeability*(conc_api-conc_baso)*A_mem');
% adsorption from apical compartment to the cell membrane
m = pwAddV(m, 'late_api', 'P_lateral_api*(Klipw*conc_api - conc_cell_membrane)');
% adsorption from basolatelal compartment to the cell membrane
m = pwAddV(m, 'late_baso', 'P_lateral_baso*(Klipw*conc_baso - conc_cell_membrane)');
% absorption to the PDMS of the chemical sink
m = pwAddV(m, 'A_tw', 'k_a_twister*conc_baso*V_twister');


%% Z - derived variable
% flux into basolateral because of diffusion (in units mass per time)
m = pwAddZ(m, 'flux_aroundCell', 'permeability*(conc_api-conc_baso)*A_mem');
% flux into basolateral because of lateral memrbane diffusion
m = pwAddZ(m, 'flux_lateral', '-P_lateral_baso*(Klipw*conc_baso - conc_cell_membrane)');
%% ODEs
% ODEs for each compartments using the pre-defined reaction rates
m = pwAddODE(m, 'conc_api'        , '(1/V_api)*(-D_I-late_api)');
m = pwAddODE(m, 'conc_baso'        , '(1/V_baso)*(D_I-late_baso - A_tw)');
m = pwAddODE(m, 'conc_cell'           ,  '0');
m = pwAddODE(m, 'conc_cell_membrane'        , '(1/V_cell_membrane)*(late_api+late_baso)');
m = pwAddODE(m, 'conc_twister'      , '(1/V_twister)*A_tw');

%% U - Driving inputs
% m = pwAddU(m, *ID, *uType, *uTimes, *uValues, compartment, name, description, u2Values, alternativeIDs, designerProps, classname, referenceXID, unit, uFormula)
% Some entities like the ligand concentration can be controlled externally.
% Here you can specify the default dependency on time of these players.
% When loading an experiment, the default dependency is usually overwritten.
% m = pwAddU(m, 'name_of_driving_variable', 'what kind of driver e.g. injection', when (this means time 0)[-1 0]  , step size from to [0 30]  , 'dish');

m = pwAddU(m, 'conc_api' , 'injection', [-100 0]  , [0 0.0144857142857143]);
m = pwAddU(m, 'conc_baso' , 'injection', [-100 0]  , [0 0.004875]);
m = pwAddU(m, 'conc_cell' , 'injection', [-100 0]  , [0 22000]);
m = pwAddU(m, 'conc_twister' , 'injection', [-100 0]  , [0 0.977099236641221]);

%% Y - Observables (what was measured)
% m = pwAddY(m, *ID, *rhs, errorModelRhs, noiseType, unit, name, description, alternativeIDs, designerProps, classname)
% ID:          ID of the observable
% rhs:         right-hand side of the observation, i.e. a function of all dynamic variables X, dynamic parameters K, and observation parameters S
% errorModel:  formula to calculate the standard deviation depending on measurements y
% noiseType:   'Gaussian' (other noise types will be implemented later)
% Example with Gaussian noise with a standard deviation of 10 % relative to y plus 5 % absolute (relative to max(y) over all y):
% m = pwAddY(m, 'totalStat_obs', 'scale_totalStat_obs * (Stat + pStat + 2 * pStatDimer)', '0.10 * y + 0.05 * max(y)', 'Gaussian');

% you can define what was measured here, also if it's different from the
% defined variables. for example, a mesaurement could be a sum of two
% variables e.g. m = pwAddY(m, 'external_selenium', 'selenate+selenite+metabolites');
m = pwAddY(m, 'Y_conc_api', 'conc_api');
m = pwAddY(m, 'Y_conc_baso', 'conc_baso');
m = pwAddY(m, 'Y_conc_cell'     , 'conc_cell+conc_cell_membrane*(V_cell_membrane/V_cell)');
m = pwAddY(m, 'Y_conc_twister', 'conc_twister');


%% Designer layout properties (default)
m.designerPropsM = [1 1 1 0 0 0 400 250 600 400 1 1 1 0 0 0 0 0 0];
