% This file is constructed as a Matlab function.
% The returned variable "m" contains all required information of the model.
% For more information visit www.potterswheel.de
%
% All the models in this family describe the fate of hydrophobic chemicals 
%   in a developed exposure chamber. the chanmber has a passive dosing
%   systems and two compartments: apical and basolateral). between the
%   compartments, there is a plstic mebrane, on which RTgut cells are grown
%   in a monolayer (or no cells). The chemical is loaded into the apical
%   compartment, where it can bind to plastics, go into the cells (where it
%   can be biotransformed), or cross the cell barrier into the basolatereal
%   compartment. The models are intended to describe this process in
%   various degrees of detail, and to be used to determine the
%   identifiability of the system and inference of the process parameters. 

% MODEL 4B
%   Model describes the procees without cell (blanks = B):
%   - chemical exchange between passive doser and apical compartment
%   - diffusion through the plastic membrane between the apical and basolateral
%   compartments
%   - absorption and desorption to plastic parts
%   - absorption and desorption to the plastic membrane. The assumption
%   here is that the half of the membrane is accesible to the apical
%   chemical, while the other half to basolateral chemical, while there is
%   NO diffusion amongst the two. 
%   - this is a simple model of absorption to plastic, in which the
%   saturation of plastic with the chemical is not taken into account
%   - it is assumed that binding to the plastic and to the membrane is
%   not the same, therefore different parameters are taken for the two
%   processes
%   - absorption of chemical on twister in the basolateral compartment

% Prior Data for the model for individual parameters: 
%   P           - permeability - mu = 2.86e-2, sd = 1.42e-2 [cm/h]
%   k_d_pdms    - desorption to passive doser - mu = 3.22e-5, SD = 1.11e-3 [1/h]
%   k_a_plastic - absorption to plastic - mu = 7.66e-2, SD = 2.95e-2 [cm3/cm2*h]
%   k_d_plastic - desorption from pastic - mu = 2.96e-1; SD = 7.72e-1 [1/h]
%   k_a_membrane - absorption to membrane - mu = 7.66e-2, SD = 2.95e-2 [cm3/cm2*h]
%   k_d_membrane - desorption from membrane - mu = 2.96e-1; SD = 7.72e-1 [1/h]
%   k_a_twister    - absorption to passive doser - mu = 2.38, SD = 1.88 [1/h]
%
%
% priors from Hannah are changed into uniform priors which cover a 100 fold
% from the mean value Hannah calculated 
%
% other constant values (Volumes, surfaces)
% V_api         <- 3.5       # Volume in the apical [ml]
% V_baso        <- 12        # Volume in the basolateral [ml]
% A_mem         <- 4.524     # Area of the Membrane [cm2] (producer information)
% V_pdms        <- 0.05      # calculated based on the weight of the tubes and the density of the PDMS [ml]
% A_plastic_med <- 5.63      # contact area of plastic with the medium in the apical compartment [cm2]
% A_mem_total   <- 12.46     # total membrane are available to chemical binding [cm2]
% V_twister     <- 0.0393    # volume of twister in ml

function m = Model_4B_tw_laterStart()

% get empty model
m= pwGetEmptyModel();

%% Meta information
m.name        = 'epithelialCellBarrier';
m.description = 'a';
m.authors     = {'Anze Zupanic; Hannah Schug'};
m.dates       = {'2017-07-21'};
m.modelFormat = 3.0;

%% Default sampling time points
m.t = 0:0.01:72;

%% X - Dynamic variables 
% % m = pwAddX(m, *ID, *startValue, fitSetting, minValue, maxValue, unit, compartment, name, description, typeOfStartValue, designerProps, classname)
% ID:              Unique name, e.g. 'ProtA'
% startValue:      The value of X at integration start time (default 0)
% fitSetting:      'global', 'local' (default), 'fix' (startValue will be fitted globally or locally or is fixed)
% min/maxValue:    Minimum and maximum of the startValue during fitting
% unit:            Unit of X
% compartment:     Compartment of X. Default: first given compartment
% typeOfStartValue: 'amount' or 'concentration' (default)
% designerProps:   Designer information, e.g. position and coloring
% classname:       Subgroup of the X variable, e.g. 'protein.receptor'.

% concentration of chemical in the apical compartment
% concentration of chemical in the basolateral compartment
% m = pwAddX(m, 'name_variable'      ,  starting_amount, 'fix', 0);
m = pwAddX(m, 'conc_api'          ,  0, 'fix', 0);
m = pwAddX(m, 'conc_baso'         ,  0, 'fix', 0);
m = pwAddX(m, 'conc_pdms'         ,  0, 'fix', 0);
m = pwAddX(m, 'conc_plastic'      ,  0, 'fix', 0);
m = pwAddX(m, 'conc_membrane_api' ,  0, 'fix', 0);
m = pwAddX(m, 'conc_membrane_baso',  0, 'fix', 0);
m = pwAddX(m, 'conc_twister'      ,  0, 'fix', 0);

%% K - Dynamic parameters (reaction rates, etc)
% m = pwAddK(m, *ID, *value, fitSetting, minValue, maxValue, unit, name, description)
% ID:         ID of the parameter
% value:      Value of the parameter
% fitSetting: 'global', 'local', 'fix' (during fitting)
% min/maxValue Limits for the parameter during parameter calibration

% permeability - permeability of the plastic membrane (with epithelial
%   barrier) [cm/h]
% k_d_pdms - desorption from passive sampler [1/h]
% k_a_pdms - absorption to passive sampler [1/h]

m = pwAddK(m, 'permeability' , 0.0487  , 'local', 2.86e-05, 28.6  );
m = pwAddK(m, 'k_d_pdms'     , 3.42256395100507e-06, 'local', 3.22e-09, 0.322 );
m = pwAddK(m, 'k_d_plastic'  , 3.97532071912339    , 'local', 2.96e-05, 296   );
m = pwAddK(m, 'k_a_plastic'  , 0.350623094091531   , 'local', 7.66e-06, 76.6  );
m = pwAddK(m, 'k_d_membrane' , 8.37228101992562    , 'local', 2.96e-05, 296   );
m = pwAddK(m, 'k_a_membrane' , 0.282211178264217   , 'local', 7.66e-06, 76.6  );
m = pwAddK(m, 'k_a_twister'  , 44.6566114611963    , 'local', 0.000639, 99200 );

% these are constants
m = pwAddK(m, 'V_api'        , 3.5                 , 'fix'  );
m = pwAddK(m, 'V_baso'       , 12                  , 'fix'  );
m = pwAddK(m, 'V_pdms'       , 0.05                , 'fix'  );
m = pwAddK(m, 'A_mem'        , 4.524               , 'fix'  );
m = pwAddK(m, 'A_plastic_med', 5.63                , 'fix'  );
m = pwAddK(m, 'A_mem_total'  , 12.46               , 'fix'  );
m = pwAddK(m, 'V_twister'    , 0.0393              , 'fix'  );

%% V - reaction rates and fluxes (to make writing the ODEs easier)
% diffusion through insert
m = pwAddV(m, 'D_I', 'permeability*(conc_api-conc_baso)*A_mem');
% desorption from passive doser
m = pwAddV(m, 'dePD', 'k_d_pdms*conc_pdms*V_pdms');
% absorption to plastic
m = pwAddV(m, 'A_P', 'k_a_plastic*conc_api');
% desorption from plastic
m = pwAddV(m, 'D_P', 'k_d_plastic*conc_plastic');
% absorption to membrane, apical side
m = pwAddV(m, 'A_M_api', 'k_a_membrane*conc_api');
% desorption from membrane, apical side
m = pwAddV(m, 'D_M_api', 'k_d_membrane*conc_membrane_api');
% absorption to membrane, basolateral side
m = pwAddV(m, 'A_M_baso', 'k_a_membrane*conc_baso');
% desorption from membrane, basolateral side
m = pwAddV(m, 'D_M_baso', 'k_d_membrane*conc_membrane_baso');
% absorption to twister
m = pwAddV(m, 'A_tw', 'k_a_twister*conc_baso*V_twister');

%% Z - derived variable
% flux into basolateral because of diffusion (in units mass per time)
m = pwAddZ(m, 'flux1', 'permeability*(conc_api-conc_baso)*A_mem');

%% ODEs
% some constant values have been added to the equations: 
%   size of plastic membrane
%   volume of apical compartment
%   volume of basolateral compartment
%   volume of passive doser
%   average concentration on passive doser
%   the total cell volume (~200 fl* 10'000'000 cells), the units used in the equations are dm for space and h
%   for time
m = pwAddODE(m, 'conc_api'          , '(1/V_api)*(-D_I + dePD + D_P*A_plastic_med - A_P*A_plastic_med + D_M_api*0.5*A_mem_total - A_M_api*0.5*A_mem_total)');
m = pwAddODE(m, 'conc_baso'         , '(1/V_baso)*(D_I + D_M_baso*0.5*A_mem_total - A_M_baso*0.5*A_mem_total-A_tw)');
m = pwAddODE(m, 'conc_pdms'         , '0');
m = pwAddODE(m, 'conc_plastic'      , 'A_P - D_P');
m = pwAddODE(m, 'conc_membrane_api' , 'A_M_api - D_M_api');
m = pwAddODE(m, 'conc_membrane_baso', 'A_M_baso - D_M_baso');
m = pwAddODE(m, 'conc_twister'      , '(1/V_twister)*A_tw');

%% U - Driving inputs
% m = pwAddU(m, *ID, *uType, *uTimes, *uValues, compartment, name, description, u2Values, alternativeIDs, designerProps, classname, referenceXID, unit, uFormula)
% Some entities like the ligand concentration can be controlled externally.
% Here you can specify the default dependency on time of these players.
% When loading an experiment, the default dependency is usually overwritten.
% m = pwAddU(m, 'name_of_driving_variable', 'what kind of driver e.g. injection', when (this means time 0)[-1 0]  , step size from to [0 30]  , 'dish');
m = pwAddU(m, 'conc_pdms', 'injection', [-100 0]  , [0 5953.8]);
m = pwAddU(m, 'conc_api' , 'injection', [-100 0]  , [0 0.0144857142857143]);
m = pwAddU(m, 'conc_baso', 'injection', [-100 0]  , [0 0.004875]);
m = pwAddU(m, 'conc_twister' , 'injection', [-100 0]  , [0 0.977099236641221]);
m = pwAddU(m, 'conc_plastic' , 'injection', [-100 0]  , [0 0.0012753108348135]);
m = pwAddU(m, 'conc_membrane_api', 'injection', [-100 0]  , [0 0.00147271268057785/2]);
m = pwAddU(m, 'conc_membrane_baso' , 'injection', [-100 0]  , [0 0.00147271268057785/2]);

%% Y - Observables (what was measured)
% m = pwAddY(m, *ID, *rhs, errorModelRhs, noiseType, unit, name, description, alternativeIDs, designerProps, classname)
% ID:          ID of the observable
% rhs:         right-hand side of the observation, i.e. a function of all dynamic variables X, dynamic parameters K, and observation parameters S
% errorModel:  formula to calculate the standard deviation depending on measurements y
% noiseType:   'Gaussian' (other noise types will be implemented later)
% Example with Gaussian noise with a standard deviation of 10 % relative to y plus 5 % absolute (relative to max(y) over all y):
% m = pwAddY(m, 'totalStat_obs', 'scale_totalStat_obs * (Stat + pStat + 2 * pStatDimer)', '0.10 * y + 0.05 * max(y)', 'Gaussian');

% you can define what was measured here, also if it's different from the
% defined variables. for example, a mesaurement could be a sum of two
% variables
% e.g. m = pwAddY(m, 'external_selenium', 'selenate+selenite+metabolites');
m = pwAddY(m, 'Y_conc_api', 'conc_api');
m = pwAddY(m, 'Y_conc_baso', 'conc_baso');
m = pwAddY(m, 'Y_conc_pdms', 'conc_pdms');
m = pwAddY(m, 'Y_conc_plastic', 'conc_plastic');
m = pwAddY(m, 'Y_conc_membrane', '(conc_membrane_api + conc_membrane_baso)/2');
m = pwAddY(m, 'Y_conc_twister', 'conc_twister');

%% Designer layout properties (default)
m.designerPropsM = [1 1 1 0 0 0 400 250 600 400 1 1 1 0 0 0 0 0 0];
