% This file is constructed as a Matlab function.
% The returned variable "m" contains all required information of the model.
% For more information visit www.potterswheel.de
%
% All the models in this family describe the fate of hydrophobic chemicals 
%   in a developed exposure chamber. the chanmber has a passive dosing
%   systems and two compartments: apical and basolateral). between the
%   compartments, there is a plstic mebrane, on which RTgut cells are grown
%   in a monolayer (or no cells). The chemical is loaded into the apical
%   compartment, where it can bind to plastics, go into the cells (where it
%   can be biotransformed), or cross the cell barrier into the basolatereal
%   compartment. The models are intended to describe this process in
%   various degrees of detail, and to be used to determine the
%   identifiability of the system and inference of the process parameters. 

% MODEL 2B_CELL1
%   Model describes the procees without cells:
%   - chemical exchange between passive doser and apical compartment
%   - diffusion through the membrane between the apical and basolateral
%   compartments

% Prior Data for the model for individual parameters: 
%   - P - permeability - mu = 2.86e-2, sd = 1.42e-2 [cm/h]
%   - k_d - desorption to passive doser - mu = 3.22e-5, SD = 1.11e-3 [1/h]
%   - k_cell       - [0,1] the cell monolayer decreases the permeability of the plastic membrane
%   - P_cell permeability of the cell to the chemical mu = ??, sd = ??
%
% other constant values (Volumes, surfaces)
% V_api         <- 3.5       # Volume in the apical [ml]
% V_baso        <- 12        # Volume in the basolateral [ml]
% A_mem         <- 4.524     # Area of the Membrane [cm2] (producer information)
% V_pdms        <- 0.05      # calculated based on the weight of the tubes and the density of the PDMS [ml]
% V_cell        <- 1.65e-6   # volume of all cells in the monolayer [ml]


% BIOLOGICAL PART OF THE MODEL
%   - chemical can enter the cell from apical and basolateral compartment
%   through diffusion


function m = Model_2B_laterStart()

% get empty model
m= pwGetEmptyModel();

%% Meta information
m.name        = 'epithelialCellBarrier';
m.description = 'a';
m.authors     = {'Anze Zupanic; Hannah Schug'};
m.dates       = {'2017-07-21'};
m.modelFormat = 3.0;

%% Default sampling time points
m.t = 0:0.01:72;

%% X - Dynamic variables 
% % m = pwAddX(m, *ID, *startValue, fitSetting, minValue, maxValue, unit, compartment, name, description, typeOfStartValue, designerProps, classname)
% ID:              Unique name, e.g. 'ProtA'
% startValue:      The value of X at integration start time (default 0)
% fitSetting:      'global', 'local' (default), 'fix' (startValue will be fitted globally or locally or is fixed)
% min/maxValue:    Minimum and maximum of the startValue during fitting
% unit:            Unit of X
% compartment:     Compartment of X. Default: first given compartment
% typeOfStartValue: 'amount' or 'concentration' (default)
% designerProps:   Designer information, e.g. position and coloring
% classname:       Subgroup of the X variable, e.g. 'protein.receptor'.

% concentration of chemical in the apical compartment
% concentration of chemical in the basolateral compartment
% m = pwAddX(m, 'name_variable'      ,  starting_amount, 'fix', 0);

m = pwAddX(m, 'conc_api' ,  0, 'fix', 0);
m = pwAddX(m, 'conc_baso',  0, 'fix', 0);
m = pwAddX(m, 'conc_pdms',  0, 'fix', 0);
%% K - Dynamic parameters (reaction rates, etc)
% m = pwAddK(m, *ID, *value, fitSetting, minValue, maxValue, unit, name, description)
% ID:         ID of the parameter
% value:      Value of the parameter
% fitSetting: 'global', 'local', 'fix' (during fitting)
% min/maxValue Limits for the parameter during parameter calibration

% permeability - permeability of the plastic membrane (with epithelial
%   barrier) [cm/h]
% k_d_pdms - desorption from passive sampler [1/h]

m = pwAddK(m, 'permeability', 0.060426091302074   , 'local', 2.86e-05, 28.6  );
m = pwAddK(m, 'k_d_pdms'    , 2.00899852893066e-05, 'local', 3.22e-09, 0.322 );

% these are constants
m = pwAddK(m, 'V_api'       , 3.5                 , 'fix'  );
m = pwAddK(m, 'V_baso'      , 12                  , 'fix'  );
m = pwAddK(m, 'V_pdms'      , 0.05                , 'fix'  );
m = pwAddK(m, 'A_mem'       , 4.524               , 'fix'  );

%% V - reaction rates and fluxes (to make writing the ODEs easier)
% diffusion through insert
m = pwAddV(m, 'D_I', 'permeability*(conc_api-conc_baso)*A_mem');
% desorption from passive doser
m = pwAddV(m, 'dePD', 'k_d_pdms*conc_pdms*V_pdms');

%% Z - derived variable
% flux into basolateral because of diffusion (in units mass per time)
m = pwAddZ(m, 'flux1', 'permeability*(conc_api-conc_baso)*A_mem');

%% ODEs
% some constant values have been added to the equations: 
%   size of plastic membrane
%   volume of apical compartment
%   volume of basolateral compartment
%   volume of passive doser
%   average concentration on passive doser
%   the total cell volume (~200 fl* 10'000'000 cells), the units used in the equations are dm for space and h
%   for time
m = pwAddODE(m, 'conc_api'        , '(1/V_api)*(-D_I + dePD)');
m = pwAddODE(m, 'conc_baso'        , '(1/V_baso)*D_I');
m = pwAddODE(m, 'conc_pdms'        , '0');

%% U - Driving inputs
% m = pwAddU(m, *ID, *uType, *uTimes, *uValues, compartment, name, description, u2Values, alternativeIDs, designerProps, classname, referenceXID, unit, uFormula)
% Some entities like the ligand concentration can be controlled externally.
% Here you can specify the default dependency on time of these players.
% When loading an experiment, the default dependency is usually overwritten.
% m = pwAddU(m, 'name_of_driving_variable', 'what kind of driver e.g. injection', when (this means time 0)[-1 0]  , step size from to [0 30]  , 'dish');
m = pwAddU(m, 'conc_pdms', 'injection', [-100 0]  , [0 5953.8]);
m = pwAddU(m, 'conc_api' , 'injection', [-100 0]  , [0 0.0144857142857143]);
m = pwAddU(m, 'conc_baso' , 'injection', [-100 0]  , [0 0.004875]);

%% Y - Observables (what was measured)
% m = pwAddY(m, *ID, *rhs, errorModelRhs, noiseType, unit, name, description, alternativeIDs, designerProps, classname)
% ID:          ID of the observable
% rhs:         right-hand side of the observation, i.e. a function of all dynamic variables X, dynamic parameters K, and observation parameters S
% errorModel:  formula to calculate the standard deviation depending on measurements y
% noiseType:   'Gaussian' (other noise types will be implemented later)
% Example with Gaussian noise with a standard deviation of 10 % relative to y plus 5 % absolute (relative to max(y) over all y):
% m = pwAddY(m, 'totalStat_obs', 'scale_totalStat_obs * (Stat + pStat + 2 * pStatDimer)', '0.10 * y + 0.05 * max(y)', 'Gaussian');

% you can define what was measured here, also if it's different from the
% defined variables. for example, a mesaurement could be a sum of two
% variables
% e.g. m = pwAddY(m, 'external_selenium', 'selenate+selenite+metabolites');
m = pwAddY(m, 'Y_conc_api', 'conc_api');
m = pwAddY(m, 'Y_conc_baso', 'conc_baso');
m = pwAddY(m, 'Y_conc_pdms', 'conc_pdms');


%% Designer layout properties (default)
m.designerPropsM = [1 1 1 0 0 0 400 250 600 400 1 1 1 0 0 0 0 0 0];
