#!/usr/bin/env bash

## "PPHPC NetLogo Datasets" by Nuno Fachada are licensed under the 
## Creative Commons Attribution 4.0 International License. To view a 
## copy of this license, visit http://creativecommons.org/licenses/by/4.0/ 
## or send a letter to Creative Commons, PO Box 1866, Mountain View, 
## CA 94042, USA.

# ################################################ #
# Script used to create the PPHPC NetLogo Datasets #
# ################################################ #

# NetLogo location
NLDIR="${HOME}/LocalPrograms/netlogo-5.1.0"

# Location of pphpc.nlogo file (PPHPC NetLogo implementation)
PPDIR="${HOME}/workspace/pphpc/netlogo"

# Location of output files
OUTDIR="."

# Create output directories if they don't yet exist
mkdir -p ${OUTDIR}/v1
mkdir -p ${OUTDIR}/v2

# Start runs
for RUN in {1..30}
do
    # Determine seed for this run
    SEED=`echo ${RUN} | md5sum | cut -f 1 -d ' '`
    SEED=`echo ${SEED} | awk '{print toupper($0)}'`
    SEED=`echo "ibase=16; ${SEED}" | bc`
    let "SEED=SEED%2147483647"

    # Cycle through model sizes
    for SIZE in 100 200 400 800 1600
    do
        let "MULT=2*(SIZE/100)"
        let "INITWOLVES=MULT*SIZE"
        let "INITSHEEP=INITWOLVES*2"
        sed "s/INITSHEEP/${INITSHEEP}/g" nlexps.in.xml > nlexps.xml
        sed -i "s/INITWOLVES/${INITWOLVES}/g" nlexps.xml
        sed -i "s/MAXPXCOR/$(($SIZE-1))/g" nlexps.xml
        sed -i "s/MAXPYCOR/$(($SIZE-1))/g" nlexps.xml
        sed -i "s/RNGSEED/${SEED}/g" nlexps.xml

        # Cycle through parameter sets
        for VERS in 1 2
        do
            OUTFILE="${OUTDIR}/v${VERS}/exp${SIZE}v${VERS}r${RUN}.csv"
            if [ ! -f ${OUTFILE} ]; then

                # Print info about current run
                echo "Run: ${RUN}, Param.Set: ${VERS}, Size: ${SIZE}, SEED: ${SEED}"

                # Perform run
                java -Xms2048m -Xmx4096m -Dfile.encoding=UTF-8 -cp ${NLDIR}/NetLogo.jar org.nlogo.headless.Main --model ${PPDIR}/pphpc.nlogo --setup-file ./nlexps.xml --threads 1 --experiment v${VERS} --table ${OUTFILE}

                # Convert NetLogo 5.1.0 output to TSV (this may fail with other NetLogo versions)
                TSVFILE="${OUTDIR}/v${VERS}/stats${SIZE}v${VERS}r${RUN}.txt"
                awk -F, 'NR>7 { gsub("\"", " "); print $19, $20, $21, $22, $23, $24 }' ${OUTFILE} > ${TSVFILE}.tmp

                # Replace spaces with tabs
                sed 's/^ //g' ${TSVFILE}.tmp | sed  's/ \+$//g' | sed 's/ \+/\t/g' > ${TSVFILE}

                # Remove temporary files
                rm ${TSVFILE}.tmp
                rm ${OUTFILE}

            fi
        done
    done

    # Remove temporary configuration file
    rm nlexps.xml

done

