#!/usr/bin/env bash

## "PPHPC ParStrat Datasets" by Nuno Fachada are licensed under the 
## Creative Commons Attribution 4.0 International License. To view a 
## copy of this license, visit http://creativecommons.org/licenses/by/4.0/ 
## or send a letter to Creative Commons, PO Box 1866, Mountain View, 
## CA 94042, USA.

# ######################################################## #
# Script used to create the PPHPC ParStrat Datasets (Java) #
# ######################################################## #

# Location of the Java PPHPC implementation
PPDIR="${HOME}/workspace/pphpc/java"

# Location of output files
OUTDIR="."

# Location of simulation configuration files
CFGDIR="${HOME}/workspace/pphpc/configs"

# Java command to run PPHPC
PPCMD="java -Xms2048m -Xmx4096m -cp ${PPDIR}/bin:${PPDIR}/lib/* org.laseeb.pphpc.PredPrey"

# Create output directories if they don't yet exist
for ODIR in ST EQ EX ER OD
do
    mkdir -p ${OUTDIR}/simout/${ODIR}
    mkdir -p ${OUTDIR}/times/${ODIR}
done

# Start runs
for RUN in {1..10}
do
    # Determine seed for this run
    SEED=`echo ${RUN} | md5sum | cut -f 1 -d ' '`
    SEED=`echo ${SEED} | awk '{print toupper($0)}'`
    SEED=`echo "ibase=16; ${SEED}" | bc`

    # Cycle through model sizes
    for SIZE in 100 200 400 800 1600
    do

        # Cycle through parameter sets
        for VERS in 1 2
        do

            # Single-thread strategy (ST)
            OUTFILE="${OUTDIR}/simout/ST/stats${SIZE}v${VERS}pSTr${RUN}.txt"
            TIMEFILE="${OUTDIR}/times/ST/time${SIZE}v${VERS}pSTr${RUN}.txt"
            if [ ! -f ${OUTFILE} ] || [ ! -f ${TIMEFILE} ]; then
                echo "${OUTFILE}"
                /usr/bin/time -o ${TIMEFILE} ${PPCMD} -p ${CFGDIR}/config${SIZE}v${VERS}.txt -s ${OUTFILE} -r ${SEED} -ps ST
            fi
            
            # Multi-threaded strategies
            for THREADS in 1 2 4 6 8 12 16 24
            do

                # Equal work parallelization strategy (EQ)
                OUTFILE="${OUTDIR}/simout/EQ/stats${SIZE}v${VERS}pEQt${THREADS}r${RUN}.txt"
                TIMEFILE="${OUTDIR}/times/EQ/time${SIZE}v${VERS}pEQt${THREADS}r${RUN}.txt"
                if [ ! -f ${OUTFILE} ] || [ ! -f ${TIMEFILE} ]; then
                    echo "${OUTFILE}"
                    /usr/bin/time -o ${TIMEFILE} ${PPCMD} -p ${CFGDIR}/config${SIZE}v${VERS}.txt -s ${OUTFILE} -r ${SEED} -n ${THREADS} -ps EQ
                fi

                # Equal with reproducibility parallelization strategy (EX)
                OUTFILE="${OUTDIR}/simout/EX/stats${SIZE}v${VERS}pEXt${THREADS}r${RUN}.txt"
                TIMEFILE="${OUTDIR}/times/EX/time${SIZE}v${VERS}pEXt${THREADS}r${RUN}.txt"
                if [ ! -f ${OUTFILE} ] || [ ! -f ${TIMEFILE} ]; then
                    echo "${OUTFILE}"
                    /usr/bin/time -o ${TIMEFILE} ${PPCMD} -p ${CFGDIR}/config${SIZE}v${VERS}.txt -s ${OUTFILE} -r ${SEED} -n ${THREADS} -ps EX
                fi

                # Equal row sync parallelization strategy (ER)
                OUTFILE="${OUTDIR}/simout/ER/stats${SIZE}v${VERS}pERt${THREADS}r${RUN}.txt"
                TIMEFILE="${OUTDIR}/times/ER/time${SIZE}v${VERS}pERt${THREADS}r${RUN}.txt"
                if [ ! -f ${OUTFILE} ] || [ ! -f ${TIMEFILE} ]; then
                    echo "${OUTFILE}"
                    /usr/bin/time -o ${TIMEFILE} ${PPCMD} -p ${CFGDIR}/config${SIZE}v${VERS}.txt -s ${OUTFILE} -r ${SEED} -n ${THREADS} -ps ER
                fi

                # On demand parallelization strategy (OD)
                for BLKSIZE in 20 50 100 200 500 1000 2000 5000
                do
                    OUTFILE="${OUTDIR}/simout/OD/stats${SIZE}v${VERS}pODb${BLKSIZE}t${THREADS}r${RUN}.txt"
                    TIMEFILE="${OUTDIR}/times/OD/time${SIZE}v${VERS}pODb${BLKSIZE}t${THREADS}r${RUN}.txt"
                    if [ ! -f ${OUTFILE} ] || [ ! -f ${TIMEFILE} ]; then
                        echo "${OUTFILE}"
                        /usr/bin/time -o ${TIMEFILE} ${PPCMD} -p ${CFGDIR}/config${SIZE}v${VERS}.txt -s ${OUTFILE} -r ${SEED} -b ${BLKSIZE} -n ${THREADS} -ps OD
                    fi
                done

            done
        done
    done
done
