#ifndef SCHNYDER_H
#define SCHNYDER_H

#include "planarmap.h"

#include <random>
#include <stack>


using namespace planar;

struct EdgeData {
	bool visited = false;
};

typedef typename Map<EdgeData>::EdgeHandle EdgeHandle;

enum walkSteps : uint8_t {
	RIGHT, LEFT, UP, DOWN
};

template<typename RNG>
std::vector<int> randomDyckPath(int halflength, RNG &rng) {
	std::vector<int> steps;
	steps.reserve(2*halflength);
	int q = halflength, p = halflength;

	for (int i = 0; i < 2*halflength; ++i) {
		// Cannot use uniform_int_distribution here, since (q+p)*(q-p+1) may overflow.
		std::bernoulli_distribution dist(double(q+1)*(q-p)/(double(q+p)*(q-p+1)));
		if (dist(rng)) {
			steps.push_back(-1);
			--q;
		} else {
			steps.push_back(1);
			--p;
		}
	}
	return steps;
}

std::vector<int> matchingFromDyckPath(std::vector<int> &path);

template<typename RNG>
Map<EdgeData> makeUniformQuadrangulation(int size, RNG &rng) {
	std::vector<int> dyckpath = randomDyckPath(size, rng);
	std::vector<int> matching = matchingFromDyckPath(dyckpath);

	std::uniform_int_distribution<> signdist(-1, 1);
	std::vector<int> edgelabels;
	edgelabels.reserve(2*size);
	for (int i = 0; i < 2*size; ++i) {
		int match = matching[i];
		if (match > i) {
			edgelabels.push_back(signdist(rng));
		} else {
			edgelabels.push_back(-edgelabels[matching[i]]);
		}
	}

	Map<EdgeData> map;
	map.reserve(4*size);

	int curLabel = 0;
	std::stack<std::pair<int, EdgeHandle>> stack;
	std::vector<EdgeHandle> cornerEdges;
	stack.push({0, map.newDoubleEdge()});
	cornerEdges.push_back(stack.top().second);
	for (int i = 0; i < 2*size - 1; ++i)
	{
		int nextLabel = curLabel + edgelabels[i];
		if (nextLabel >= curLabel) {
			if (matching[i] >= i) {
				if (!stack.empty() && stack.top().first == nextLabel) {
					stack.top().second = map.insertEdge(stack.top().second->getNext())->getAdjacent();
				} else {
					stack.push({nextLabel, map.newDoubleEdge()});
				}
			} else {
				if (!stack.empty() && stack.top().first == nextLabel) {
					stack.top().second = map.insertEdge(cornerEdges[matching[i]], stack.top().second->getNext());
				} else {
					stack.push({nextLabel, map.insertEdge(cornerEdges[matching[i]])});
				}
			}
		} else {
			assert(stack.top().first == nextLabel + 1);

			EdgeHandle attachEdge = stack.top().second->getNext();
			stack.pop();

			if (!stack.empty() && stack.top().first == nextLabel) {
				stack.top().second = map.insertEdge(attachEdge,stack.top().second->getNext());
			} else {
				stack.push({nextLabel, map.insertEdge(attachEdge)});
			}

			if (matching[i] < i) {
				map.contractVertices(stack.top().second, cornerEdges[matching[i]]);
			}
		}
		cornerEdges.push_back(stack.top().second);

		curLabel += edgelabels[i];
	}
	// Have to contract the vertices that remain in the stack and circle around the root.
	EdgeHandle contractEdge = cornerEdges[0]->getAdjacent()->getPrevious(1-stack.top().first);
	while (stack.size() > 1) {
		map.contractVertices(stack.top().second->getNext(), contractEdge->getNext());
		contractEdge = contractEdge->getPrevious();
		stack.pop();
	}

	return map;
}

void addFirstSchnyderTree(std::vector<walkSteps> const &walk, Map<EdgeData> &map);
std::vector<EdgeHandle> addSecondSchnyderTree(std::vector<walkSteps> const &walk, Map<EdgeData> &map);

template <typename RNG>
std::vector<walkSteps> randomWalkInCone(int halflength, RNG &rng) {
	// produce a walk in 45° cone, i.e. (0, 0) → (0, 0) such that 0 ≤ y ≤ x.

	using dist = std::discrete_distribution<>;
	dist select;

	std::vector<walkSteps> walk;
	walk.reserve(2*halflength);

	int x = 0, y = 0;
	for (int n = 2*halflength; n != 0; --n) {
		// Probabilities chosen so all walks have equal probability. Although these are integers, they are way too big for an uint64_t.
		auto pRight = double(3 + x)*(n - x - y)*(2 + x - y)*(1 + y)*(2 + n - x + y)*(4 + x + y);
		auto pLeft = double(1 + x)*(x - y)*(4 + n + x - y)*(1 + y)*(2 + x + y)*(6 + n + x + y);
		auto pUp = double(2 + x)*(x - y)*(4 + n + x - y)*(2 + y)*(4 + x + y)*(n - x - y);
		auto pDown = double(2 + x)*(2 + x - y)*y*(2 + n - x + y)*(2 + x + y)*(6 + n + x + y);
		auto dir = (walkSteps)select(rng, dist::param_type{pRight, pLeft, pUp, pDown});
		walk.push_back(dir);
		switch (dir) {
			case RIGHT: ++x; break;
			case LEFT: --x; break;
			case UP: ++y; break;
			case DOWN: --y; break;
		}
	}

	//char const *d[] = {"→", "←", "↑", "↓"}; for (auto s : walk) std::cerr << d[s]; std::cerr << " LENGTH = " << walk.size() << std::endl;

	return walk;
}

template <typename RNG>
Map<EdgeData> makeSchnyderMap(int size, RNG &rng) {
	if (size%2 == 1) {
		++size;
	}
	size = size/2 - 1;
	std::vector<walkSteps> walk = randomWalkInCone(size, rng);

	Map<EdgeData> map;
	map.reserve(2*(size + 1)*3);
	map.makePolygon(3);
	addFirstSchnyderTree(walk,map);
	std::vector<EdgeHandle> greencorners = addSecondSchnyderTree(walk,map);
	for (auto corner = greencorners.begin(); corner != greencorners.end(); ++corner) {
		map.insertEdge((*corner)->getNext(),(*corner)->getPrevious());
	}
	return map;
}


template <typename RNG>
std::vector<walkSteps> randomSkewWalkInCone(int length, RNG &rng) {
	// produce a uniform walk in the first quadrant with steps UP=(0, 1), LEFT=(-1, 0), RIGHT(down)=(1, -1)
	// starting and ending at the origin

	using dist = std::discrete_distribution<>;
	dist select;

	std::vector<walkSteps> walk;
	walk.reserve(length);

	int x = 0, y = 0;
	for (int n = length; n != 0; --n) {
		auto pRightDown = double(2 + x)*y*(3 + n - x + y)*(2 + x + y);
		auto pLeft = double(1 + y)*x*(1 + x + y)*(6 + n + 2*x + y);
		auto pUp = double(1 + x)*(n - x - 2*y)*(2 + y)*(3 + x + y);
		auto dir = (walkSteps)select(rng, dist::param_type{pRightDown, pLeft, pUp, 0.0});
		walk.push_back(dir);
		switch (dir) {
			case RIGHT: ++x; --y; break; // RIGHT is right & down
			case LEFT: --x; break;
			case UP: ++y; break;
			case DOWN: break;
		}
	}

	return walk;
}

void walkToBipolarMap(std::vector<walkSteps> const &walk, Map<EdgeData> &map);

template<typename RNG>
Map<EdgeData> makeBipolarMap(int size, RNG &rng) {
	Map<EdgeData> map;
	map.reserve(3*size);

	int numEdges = 3*size/2;

	std::vector<walkSteps> walk = randomSkewWalkInCone(numEdges, rng);

	walkToBipolarMap(walk,map);

	return map;
}

template<typename RNG>
std::vector<bool> randomSubsetIndicator(int length, int number, RNG &rng) {
	std::vector<bool> indicator;
	indicator.reserve(length);
	int rest = number;
	for (int l = length; l > 0; --l) {
		std::uniform_int_distribution<> dist(1, l);
		if (dist(rng) <= rest) {
			indicator.push_back(true);
			rest--;
		} else {
			indicator.push_back(false);
		}
	}
	return indicator;
}

template<typename RNG>
std::vector<walkSteps> randomWalkInQuarterPlane(int halflength, RNG &rng) {
	// We want to generate a walk in qplane by shuffling two walks on half-line.
	// The precise distribution of the first walk is binom{2n}{2k}*cat(n-k)*cat(k)/(cat(n)*cat(n+1)).
	// To approximate this we generate a normal random variable with mean n/2 and st.dev. sqrt(n/8).
	std::normal_distribution<> normaldistribution(0.5*halflength, std::sqrt(halflength/8.0));
	int k = static_cast<int>(0.5 + normaldistribution(rng));
	//k = std::clamp(k, 0, halflength);
	k = std::min(std::max(k, 0), halflength);

	std::vector<int> steps1 = randomDyckPath(k, rng);
	std::vector<int> steps2 = randomDyckPath(halflength - k, rng);
	std::vector<bool> indicator = randomSubsetIndicator(2*halflength, 2*k, rng);

	int num1 = 0, num2 = 0;
	std::vector<walkSteps> steps;
	steps.reserve(2*halflength);
	for (int i = 0; i < 2*halflength; ++i) {
		if (indicator[i]) {
			steps.push_back(steps1[num1] == 1? UP : DOWN);
			++num1;
		} else {
			steps.push_back(steps2[num2] == 1? RIGHT : LEFT);
			++num2;
		}
	}
	return steps;
}

void walkToMap(std::vector<walkSteps> const &walk, Map<EdgeData> &map);

template <typename RNG>
Map<EdgeData> makeSpanningTreeMap(int size, RNG &rng) {
	Map<EdgeData> map;
	map.reserve(4*size);
	std::vector<walkSteps> walk = randomWalkInQuarterPlane(size, rng);

	walkToMap(walk, map);

	return map;
}

#endif
