#ifndef PLANARMAP_H
#define PLANARMAP_H

#include <vector>
#include <set>
#include <queue>
#include <assert.h>

namespace planar {

template<typename Data>
class Edge {
public:
	using Id = int;

	class Handle;
	class HandleBase {
	public:
		using Id = Edge::Id;

		std::vector<Edge<Data>> *edges;
		Id id;

		HandleBase() = default;
		HandleBase(std::vector<Edge<Data>> *edges, Id id) : edges(edges), id(id) {}

		bool operator ==(HandleBase const &other) const { return id == other.id; }
		bool operator !=(HandleBase const &other) const { return id != other.id; }

		Edge<Data> &v() { return (*edges)[id]; }
		Edge<Data> &v(Id id) { return (*edges)[id]; }

		Handle getNext() {
			return {edges, v().next_};
		}

		Handle getNext(int n) {
			if (n < 0) {
				return getPrevious(-n);
			}
			Id edge = id;
			for (int i = 0; i < n; ++i) {
				edge = v(edge).next_;
			}
			return {edges, edge};
		}

		void setNext(Handle next) {
			v().next_ = next.id;
		}

		Handle getPrevious() {
			return {edges, v().previous_};
		}

		Handle getPrevious(int n) {
			if (n < 0) {
				return getNext(-n);
			}
			Id edge = id;
			for (int i = 0; i < n; ++i) {
				edge = v(edge).previous_;
			}
			return {edges, edge};
		}

		void setPrevious(Handle previous) {
			v().previous_ = previous.id;
		}

		Handle getAdjacent() {
			return {edges, v().adj_};
		}

		void setAdjacent(Handle adj) {
			v().adj_ = adj.id;
		}

		Handle getRotateCCW() {
			return getPrevious()->getAdjacent();
		}

		Handle getRotateCCW(int n) {
			if (n < 0) {
				return getRotateCW(-n);
			}
			Handle edge = this;
			for (int i = 0; i < n; ++i) {
				edge = edge->getRotateCCW();
			}
			return edge;
		}

		Handle getRotateCW() {
			return getAdjacent()->getNext();
		}

		Handle getRotateCW(int n) {
			if (n < 0) {
				return getRotateCCW(-n);
			}
			Handle edge = this;
			for (int i = 0; i < n; ++i) {
				edge = edge->getRotateCW();
			}
			return edge;
		}

		Id getId() const {
			return id;
		}

		Data &data() {
			return v().data_;
		}
	};

	class Handle : public HandleBase {
	public:
		Handle() = default;
		Handle(std::vector<Edge<Data>> *edges, int id) : HandleBase(edges, id) {}
		HandleBase *operator ->() { return this; }  // Pretend we are a pointer.
		Handle &operator *() { return *this; }  // Pretend we are an iterator.
		Handle &operator ++() { ++HandleBase::id; return *this; }
		Handle &operator --() { --HandleBase::id; return *this; }
		Handle operator ++(int) { return {HandleBase::edges, HandleBase::id++}; }
		Handle operator --(int) { return {HandleBase::edges, HandleBase::id--}; }
	};

private:
	Id next_;
	Id previous_;
	Id adj_;
	Data data_;
};

template<typename Data>
class Map {
public:
	using EdgeHandle = typename Edge<Data>::Handle;
	using EdgeIterator = typename Edge<Data>::Handle;
	using Id = typename Edge<Data>::Id;
private:
	std::vector<Edge<Data>> edges_;
	EdgeHandle root_;

public:
	EdgeIterator begin() {
		return {&edges_, 0};
	}
	EdgeIterator end() {
		return {&edges_, Id(edges_.size())};
	}

	EdgeHandle getRoot() {
		return root_;
	}
	void setRoot(EdgeHandle root) {
		root_ = root;
	}
	EdgeHandle newEdge() {
		edges_.push_back({});
		return {&edges_, Id(edges_.size() - 1)};
	}
	EdgeHandle newDoubleEdge() {
		EdgeHandle edge1 = newEdge();
		EdgeHandle edge2 = newEdge();
		edge1->setAdjacent(edge2);
		edge1->setNext(edge2);
		edge1->setPrevious(edge2);
		edge2->setAdjacent(edge1);
		edge2->setNext(edge1);
		edge2->setPrevious(edge1);
		return edge1;
	}
	void reserve(Id count) {
		edges_.reserve(count);
	}
	int numHalfEdges() const {
		return edges_.size();
	}
	void clearMap() {
		edges_.clear();
	}
	EdgeHandle getEdge(int id) {
		return {&edges_, id};
	}
	void makePolygon(int n) {
		clearMap();
		for (int i = 0; i < 2*n; ++i) {
			newEdge();
		}
		for (int i = 0; i < n; ++i) {
			getEdge(i)->setAdjacent(getEdge(i + n));
			getEdge(i)->setNext(getEdge((i + 1)%n));
			getEdge(i)->setPrevious(getEdge((i + n - 1)%n));
			getEdge(i + n)->setAdjacent(getEdge(i));
			getEdge(i + n)->setNext(getEdge((i + n - 1)%n + n));
			getEdge(i + n)->setPrevious(getEdge((i + 1)%n + n));
		}
		setRoot(getEdge(0));
	}
	EdgeHandle insertEdge(EdgeHandle edge) {
		EdgeHandle newedge = newDoubleEdge();
		newedge->getAdjacent()->setNext(edge);
		newedge->setPrevious(edge->getPrevious());
		edge->getPrevious()->setNext(newedge);
		edge->setPrevious(newedge->getAdjacent());
		return newedge;
	}
	void contractVertices(EdgeHandle edge1, EdgeHandle edge2) {
		if (edge1 == edge2) {
			return;
		}

		EdgeHandle edge1prev = edge1->getPrevious();
		EdgeHandle edge2prev = edge2->getPrevious();
		edge1prev->setNext(edge2);
		edge2prev->setNext(edge1);
		edge1->setPrevious(edge2prev);
		edge2->setPrevious(edge1prev);
	}
	EdgeHandle insertEdge(EdgeHandle edge1, EdgeHandle edge2) {
		EdgeHandle newedge = insertEdge(edge1);
		contractVertices(newedge->getNext(), edge2);
		return newedge;
	}
	void reattachEdge(EdgeHandle edge, EdgeHandle target) {
		// change the origin of edge to the origin of target
		// (assuming target and edge are in the same face)
		EdgeHandle adjEdge = edge->getAdjacent();
		if (adjEdge->getNext() != edge) {
			adjEdge->getNext()->setPrevious(edge->getPrevious());
			edge->getPrevious()->setNext(adjEdge->getNext());
		}
		edge->setPrevious(target->getPrevious());
		adjEdge->setNext(target);
		target->getPrevious()->setNext(edge);
		target->setPrevious(adjEdge);
	}
	void detachEdge(EdgeHandle edge) {
		edge->getPrevious()->setNext(edge->getAdjacent()->getNext());
		edge->getAdjacent()->getNext()->setPrevious(edge->getPrevious());
		edge->setPrevious(edge->getAdjacent());
		edge->getAdjacent()->setNext(edge);
	}
	void removeEdge(EdgeHandle edge) {
		EdgeHandle adjEdge = edge->getAdjacent();
		detachEdge(edge);
		detachEdge(adjEdge);
		deleteEdge(edge);
		deleteEdge(adjEdge);
	}
	void insertFace(EdgeHandle edge, int degree) {
		EdgeHandle newedge = edge;
		for (int i = 0; i < degree - 1; ++i)
		{
			newedge = insertEdge(newedge)->getNext();
		}
		contractVertices(newedge, edge->getNext());
	}
	int numFaces() {
		int num = 0;
		std::vector<bool> visited(numHalfEdges(), false);
		for (int i = 0, endi = edges_.size(); i < endi; ++i) {
			EdgeHandle edge = edges_[i];
			if (!visited[edge->getId()]) {
				++num;
				while (!visited[edge->getId()]) {
					visited[edge->getId()] = true;
					edge = edge->getNext();
				}
			}
		}
		return num;
	}
	int numVertices() {
		int num = 0;
		std::vector<bool> visited(numHalfEdges(), false);
		for (int i = 0, endi = edges_.size(); i < endi; ++i) {
			EdgeHandle edge = edges_[i];
			if (!visited[edge->getId()]) {
				++num;
				while (!visited[edge->getId()]) {
					visited[edge->getId()] = true;
					edge = edge->getRotateCCW();
				}
			}
		}
		return num;
	}
	int faceDegree(EdgeHandle edge) {
		int degree = 0;
		EdgeHandle faceedge = edge;
		do {
			++degree;
			faceedge = faceedge->getNext();
		} while (faceedge != edge);
		return degree;
	}
	int vertexDegree(EdgeHandle edge) {
		int degree = 0;
		EdgeHandle vertedge = edge;
		do {
			++degree;
			vertedge = vertedge->getRotateCCW();
		} while (vertedge != edge);
		return degree;
	}
	bool isPlanarMap() {
		for (int i = 0, endi = edges_.size(); i < endi; ++i) {
			if (edges_[i]->getNext()->getPrevious() != edges_[i] ||
			    edges_[i]->getPrevious()->getNext() != edges_[i] ||
			    edges_[i]->getAdjacent()->getAdjacent() != edges_[i])
				return false;
		}
		return numVertices() - numHalfEdges()/2 + numFaces() == 2;
	}
};

}

#endif
