#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <time.h>
#include <cmath>
#include <algorithm>

#ifdef _WIN32
#include <windows.h>
#endif

#include "boost/program_options.hpp"

#include "liouvillefpp.h"


unsigned long seedMix(unsigned long a, unsigned long b, unsigned long c) {
    a=a-b;  a=a-c;  a=a^(c >> 13);
    b=b-c;  b=b-a;  b=b^(a << 8);
    c=c-a;  c=c-b;  c=c^(b >> 13);
    a=a-b;  a=a-c;  a=a^(c >> 12);
    b=b-c;  b=b-a;  b=b^(a << 16);
    c=c-a;  c=c-b;  c=c^(b >> 5);
    a=a-b;  a=a-c;  a=a^(c >> 3);
    b=b-c;  b=b-a;  b=b^(a << 10);
    c=c-a;  c=c-b;  c=c^(b >> 15);
    return c;
}

unsigned long getseed()
{
#ifdef _WIN32
	unsigned int pid = GetCurrentProcessId();
#else
	unsigned int pid = getpid();
#endif
	return seedMix(clock(), time(nullptr), pid);
}

void outputImage(std::ostream &of, std::vector<double> const &exp, int width, RNG &rng) {
	std::vector<uint8_t> data(width*width*3);
	for (int y = 0; y < width; ++y) {
		for (int x = 0; x < width; ++x) {
			auto f = exp[x*width + y];
			auto p = &data[3*(width*y + x)];
			p[0] = 255*f/(f + 1) + .5;
			p[1] = 255*f/(f + 2) + .5;
			p[2] = 255*f/(f + 4) + .5;
		}
	}
	std::uniform_int_distribution<> randomcoordinate(0, width - 1);
	//auto paths = twopointfunctionpaths(exp, width, randomcoordinate(rng), randomcoordinate(rng));
	auto paths = twopointfunctionpaths(exp, width, width/2, width/2);
	for (int num = 0; num < 30; ++num) {
		int x = randomcoordinate(rng), y = randomcoordinate(rng);
		while (true) {
			uint8_t dxy = paths[x*width + y];
			int dx = (dxy&3) - 1;
			int dy = (dxy>>2) - 1;
			auto p = &data[3*(width*y + x)];
			if (dxy == 5) {
				p[0] = 255; p[1] = 0; p[2] = 255;
			} else {
				p[0] = 0; p[1] = 128; p[2] = 255;
			}
			x = (x + dx + width)%width;
			y = (y + dy + width)%width;
			if (dxy == 5) {
				break;
			}
		}
	}
	of << "P6\n" << width << ' ' << width << "\n255\n";
	of.write((char const *)data.data(), data.size());
}

int main(int argc, char **argv) {
	RNG rng;
	rng.seed(getseed());

	int width = 512;
	double xival = 0.9, ximin = 0.9, ximax = 0.9, deltaxi = 0.1;
	int num;
	std::string filename;
	int freq = 100;
	std::string imageout;

	namespace po = boost::program_options;
	po::options_description desc("Allowed options");
	desc.add_options()
		("help", "produce help message")
		("verbose,v", "print debug information")
		("width,w", po::value<int>(&width)->default_value(512), "set lattice width")
		("runs,n", po::value<int>(&num), "number of runs")
		("output,o", po::value<std::string>(&filename), "output file name")
		("outputfreq,f", po::value<int>(&freq)->default_value(100), "output frequency: number of field generations between file writing")
		("text,t", "output as plain text instead of Mathematica format")
		("xi,x", po::value<double>(&xival), "set ξ parameter in e^(ξ*√(π/2)*DGFF)")
		("ximin,l", po::value<double>(&ximin), "set lowest value of ξ")
		("ximax,u", po::value<double>(&ximax), "set highest value of ξ")
		("deltaxi,p", po::value<double>(&deltaxi), "set step size of ξ")
		("twopoint,2", "measure full two-point function")
		("imageout,O", po::value<std::string>(&imageout), "output image of one DGFF instead of histogram")
	;

	po::variables_map vm;
	po::store(po::parse_command_line(argc, argv, desc), vm);
	po::notify(vm);

	if (vm.count("help")) {
		std::cerr << desc << "\n";
		return 1;
	}
	if (!vm.count("runs")) {
		num = 10000000;
	}

	bool output = vm.count("output");
	bool text = vm.count("text");
	bool twopoint = vm.count("twopoint");
	bool multiplexi = vm.count("ximin") && vm.count("ximax") && vm.count("deltaxi");
	if (!multiplexi && !vm.count("xi")) {
		std::cerr << "Provide value or range of xi.\n";
		return 1;
	}
	std::vector<double> xis;
	if (!multiplexi) {
		xis.push_back(xival);
		ximin = xival;
		ximax = xival;
		deltaxi = 1.0;
	} else {
		for (double xi = ximin; xi < ximax + 1.0e-6; xi += deltaxi) {
			xis.push_back(xi);
		}
	}
	int numxis = xis.size();

	int histogramSize = 2*width;
	std::vector<Histogram> histogram(numxis, Histogram(histogramSize, 0));

	// Read outputfile if present, to resume collecting data.
	uint64_t initialCount = 0;
	if (text && output) {
		std::ifstream inputfile(filename);
		std::vector<uint64_t> counts(numxis*histogramSize, 0);
		struct {
			int width;
			double xi;
			uint64_t count;
			int r;
			uint64_t samples;
		} line;
		while (inputfile >> line.width >> line.xi >> line.count >> line.r >> line.samples) {
			int xi = round((line.xi - ximin)/deltaxi);
			if (xi < 0 || xi >= numxis || fabs(xis[xi] - line.xi) > 1e-6 || width != line.width || (initialCount != 0 && initialCount != line.count) || line.r < 0 || line.r >= histogramSize) {
				std::cerr << "Can't understand outputfile to resume.\n";
				return 1;
			}
			initialCount = line.count;
			counts[xi*histogramSize + line.r] += line.count;
			histogram[xi][line.r] += line.samples;
		}
		if (!counts.empty()) {
			initialCount = counts.front();
			
			if (std::any_of(counts.begin(), counts.end(), [&](uint64_t val) { return val != initialCount; })) {
				std::cerr << "Not all data has the same count.\n";
				return 1;
			}
		}
	}

	std::vector<double> average(numxis, 0.);

	GaussianField gaussian(width);
	std::vector<double> exp(width*width);

	auto generateData = [&] {
		gaussian.GenerateRandom(rng);

		for (int xi = 0; xi < numxis; ++xi) {
			// the factor multiplying the DGFF should be \xi\sqrt{\pi/2}, see Ang, arXiv:1904.09285
			double factor = xis[xi] * std::sqrt(M_PI / 2.);

			for (int x = 0; x < width; ++x) {
				for (int y = 0; y < width; ++y) {
					exp[x*width + y] = std::exp(factor*gaussian.getField(x, y));
				}
			}
			if (!imageout.empty()) {
				{
					std::ofstream of;
					of.open(imageout);
					outputImage(of, exp, width, rng);
				}
				std::cout << "Wrote \"" << imageout << "\"\n";
				exit(0);
			} else if (twopoint) {
				std::uniform_int_distribution<> randomcoordinate(0, width - 1);
				twopointfunction(exp, width, histogram[xi], randomcoordinate(rng), randomcoordinate(rng));
			} else {
				double dist = shortestpath(exp, width, false);
				average[xi] += dist;
				int bin = static_cast<int>(dist);
				if (bin >= 0 && bin < 2*width) {
					++histogram[xi][bin];
				}
			}
		}
	};

	auto saveData = [&](uint64_t n) {
		std::ofstream outputfile;
		std::ostream *file = &std::cout;
		if (output) {
			outputfile.open(filename);
			file = &outputfile;
		}
		if (text) {
			file->precision(16);
			for (int xi = 0; xi < numxis; ++xi) {
				for (int r = 0; r < histogramSize; ++r) {
					*file << width << ' ' << xis[xi] << ' ' << n + 1 << ' ' << r << ' ' << histogram[xi][r] << '\n';
				}
			}
		} else {
			*file << "{ width -> " << width;
			*file << ", twopoint -> " << (twopoint? "True" : "False");
			*file << ", measurements -> " << n + 1;
			*file << ", data -> {";
			for (int xi = 0; xi < numxis; ++xi) {
				*file << (xi > 0? "," : "") << "{ xi -> " << std::fixed << xis[xi] << ", average -> " << average[xi]/(n + 1) << ", histogram -> {";
				for (int j = 0; j < histogramSize; ++j) {
					*file << (j > 0? "," : "") << histogram[xi][j];
				}
				*file << "}}";
			}
			*file << "}}\n";
		}
	};

	for (int n = initialCount; n < num; ++n) {
		generateData();
		if ((output && (n + 1)%freq == 0) || n == num - 1) {
			saveData(n);
		}
	}
}
