#pragma once

#include "Array.h"
#include "fftw++.h"

#include <random>
#include <vector>


using RNG = std::mt19937_64;


static inline Complex RandomComplexNormal(RNG &rng) {
	std::normal_distribution<> normal_distribution;
	return {normal_distribution(rng), normal_distribution(rng)};
}

class GaussianField {
public:
	GaussianField(int width) :
		field_(width, width, sizeof(Complex)),
		width_(width),
		usereal_(false),
		spectrum_(width*width, 0),
		fourier_(1, field_) {
		FillSpectrum();
	}

	void GenerateRandom(RNG &rng) {
		if (usereal_) {
			// Use the imaginary part of the field.
			usereal_ = false;
		} else {
			// Generate a new complex gaussian field
			// and use its real part.
			usereal_ = true;
			double rescaling = 2.0/width_;
			for (int p1 = 0; p1 < width_; ++p1) {
				for (int p2 = 0; p2 < width_; ++p2) {
					field_(p1, p2) = rescaling*spectrum_[p1*width_ + p2]*RandomComplexNormal(rng);
				}
			}
			fourier_.fft(field_);
		}
	}

	double getField(int x, int y) const {
		return usereal_? field_(x,y).real() : field_(x,y).imag();
	}
	int getWidth() const {
		return width_;
	}

private:
	Array::array2<Complex> field_;
	int width_;
	bool usereal_;
	std::vector<double> spectrum_;
	fftwpp::fft2d fourier_;

	void FillSpectrum() {
		std::vector<double> sines(width_);
		for (int p = 0; p < width_; ++p) {
			double sine = std::sin(static_cast<double>(p) * M_PI / static_cast<double>(width_));
			sines[p] = 4.0*sine*sine; // 2.0 / M_PI * sine * sine;
		}

		for (int p1 = 0; p1 < width_; ++p1) {
			for (int p2 = 0; p2 < width_; ++p2) {
				if (p1 != 0 || p2 != 0) {
					spectrum_[p1*width_ + p2] = 1.0/std::sqrt(sines[p1] + sines[p2]);
				} else {
					spectrum_[p1*width_ + p2] = 0.0;
				}
			}
		}
	}
};

/*
 * Yields the Discrete GFF on a box with corners at (0,0) and (width,width) with Dirichlet
 * boundary conditions, i.e. the field vanishes at x==0 or x==width or y==0 or y==width.
 * The law of the GFF corresponds to
 *
 * P(field(x,y)) = (1/Z) * exp( - sum_{ x1=0,...,w, y1=0,...w, x2=0,...,w, y2=0,...w, x~y } (field(x1,y1)-field(x2,y2))^2 )
 *
 * It has covariance E( field(x1,y1)*field(x2,y2) ) =  E( #visits (including start) of SRW started at (x1,y1) to (x2,y2) before hitting boundary)
*/
class DirichletGaussianField {
public:
	DirichletGaussianField(int width) :
		field_(2*width, 2*width, sizeof(Complex)),
		width_(2*width),
		usereal_(false),
		spectrum_(width*width),
		fourier_(1, field_) {
		FillSpectrum();
	}

	void GenerateRandom(RNG &rng) {
		if (usereal_) {
			// Use the imaginary part of the field.
			usereal_ = false;
		} else {
			// Generate a new complex gaussian field
			// and use its real part.
			usereal_ = true;
			double rescaling = 2.0/width_;

			for (int p = 0; p < width_; ++p) {
				field_(p, 0) = 0;
				field_(p, width_/2) = 0;
				field_(0, p) = 0;
				field_(width_/2, p) = 0;
			}

			for (int p1 = 1; 2*p1 < width_; ++p1) {
				for (int p2 = 1; 2*p2 < width_; ++p2) {
					Complex normal = rescaling * spectrum_[p1*width_/2 + p2] * RandomComplexNormal(rng);
					field_(p1, p2) = normal;
					field_(width_ - p1, p2) = -normal;
					field_(p1, width_ - p2) = -normal;
					field_(width_ - p1, width_ - p2) = normal;
				}
			}
			fourier_.fft(field_);
		}
	}

	double getField(int x, int y) const {
		return usereal_? field_(x, y).real() : field_(x, y).imag();
	}
	int getWidth() const {
		return width_;
	}

private:
	Array::array2<Complex> field_;
	int width_;
	bool usereal_;
	std::vector<double> spectrum_;
	fftwpp::fft2d fourier_;

	void FillSpectrum() {
		std::vector<double> sines(width_/2);
		for (int p = 0; p < width_/2; ++p) {
			double sine = std::sin(static_cast<double>(p) * M_PI / static_cast<double>(width_));
			sines[p] = 4.0*sine*sine; // 2.0 / M_PI * sine * sine;
		}

		for (int p1 = 0; p1 < width_/2; ++p1) {
			for (int p2 = 0; p2 < width_/2; ++p2) {
				if (p1 != 0 && p2 != 0) {
					spectrum_[p1*width_/2 + p2] = 1.0/std::sqrt(sines[p1] + sines[p2]);
				} else {
					spectrum_[p1*width_/2 + p2] = 0.0;
				}
			}
		}
	}
};

double shortestpath(std::vector<double> const &time, int width, bool point);

using Histogram = std::vector<uint64_t>;
void twopointfunction(std::vector<double> const &time, int width, Histogram &histogram, int startx, int starty);

std::vector<uint8_t> twopointfunctionpaths(std::vector<double> const &time, int width, int startx, int starty);
