/**
 * \copyright
 * Copyright (c) 2012-2019, OpenGeoSys Community (http://www.opengeosys.org)
 *            Distributed under a Modified BSD License.
 *              See accompanying file LICENSE.txt or
 *              http://www.opengeosys.org/project/license
 *
 */

#pragma once

#include "NumLib/IndexValueVector.h"
#include "ProcessLib/ProcessVariable.h"

#include "BoundaryCondition.h"

namespace ProcessLib
{
class BoundaryConditionCollection final
{
public:
    explicit BoundaryConditionCollection(
        std::vector<std::unique_ptr<ParameterLib::ParameterBase>> const&
            parameters)
        : _parameters(parameters)
    {
    }

    void applyNaturalBC(const double t, GlobalVector const& x, GlobalMatrix& K,
                        GlobalVector& b, GlobalMatrix* Jac);

    std::vector<NumLib::IndexValueVector<GlobalIndexType>> const*
    getKnownSolutions(double const t, GlobalVector const& x) const
    {
        auto const n_bcs = _boundary_conditions.size();
        for (std::size_t i=0; i<n_bcs; ++i) {
            auto const& bc = *_boundary_conditions[i];
            auto& dirichlet_storage = _dirichlet_bcs[i];
            bc.getEssentialBCValues(t, x, dirichlet_storage);
        }
        return &_dirichlet_bcs;
    }

    void addBCsForProcessVariables(
        std::vector<std::reference_wrapper<ProcessVariable>> const&
            process_variables,
        NumLib::LocalToGlobalIndexMap const& dof_table,
        unsigned const integration_order, Process const& process);

    void addBoundaryCondition(std::unique_ptr<BoundaryCondition>&& bc)
    {
        _boundary_conditions.push_back(std::move(bc));
    }

    void preTimestep(const double t, GlobalVector const& x)
    {
        for (auto const& bc_ptr : _boundary_conditions)
        {
            bc_ptr->preTimestep(t, x);
        }
    }

private:
    mutable std::vector<NumLib::IndexValueVector<GlobalIndexType>> _dirichlet_bcs;
    std::vector<std::unique_ptr<BoundaryCondition>> _boundary_conditions;
    std::vector<std::unique_ptr<ParameterLib::ParameterBase>> const&
        _parameters;
};

}  // namespace ProcessLib
