# mTower Coding Standard

### 1.1 File Organization

File header. Every *.c, *.h, Makefile, or *.sh begins with a file header.
Thatfile header is enclosed with a block comment:
* The relative path to the file from the top-level directory.
* An optional, one-line description of the file contents.
* A blank line
* A copyright notice indented two additional spaces
* An optional, line identifying the author and contact information with the
  same indentation as the copyright notice.
* A blank line
* Standard (modified) BSD licensing information.

*Find example into appendix A.* 

**The following groupings should appear in all C source files in the following order:**
* Included Files
* Pre-processor Definitions
* Private Types (definitions)
* Private Function Prototypes (declarations)
* Private Data (definitions)
* Public Data (definitions)
* Private Functions (definitions)
* Public Functions (definitions)

**The following groupings should appear in all C header files in the following order:**
* Included Files
* Pre-processor Definitions
* Public Types (definitions)
* Public Data (declarations)
* Inline Functions (definitions)
* Public Function Prototypes (declarations)

**Header File Idempotence.** C header file must protect against multiple 
inclusion through the use of macros that "guard" against multiple definitions 
if the header file is included multiple times.

Each header file must contain the following pre-processor conditional logic
near the beginning of the header file: Between the file header and the
"Included Files" block comment. For example,

```c
#ifndef __INCLUDE_CRYPTO_AES_H
#define __INCLUDE_CRYPTO_AES_H

...

#endif /* __INCLUDE_CRYPTO_AES_H */
```

**Forming Guard Names.** Then pre-processor macro name used in the guard is
formed from the full, relative path to the header for from the top-level,
controlled directory. That pat is preceded by `__` and `_` replaces each
character that would otherwise be invalid in a macro name. So, for example,
`__INCLUDE_CRYPTO_AES_H` corresponds to the header file
`include/crypto/aes.h`

**Deoxygen Information.** mTower uses Deoxygen for documentation.

Sample File Formats. C source and header file templates are provided in an
Appendix to this document.

### 1.2 Lines

**Line Endings.** Files should be formatted with \n (Unix line endings). There
should be no extra whitespace at the end of the line. In addition, all text
files should end in a single newline (\n). This avoids the "No newline at end
of file" warning generated by certain tools.

**Line Width.** Text should not extend past column 80 in the typical C source
or header file.

### 1.3 Comments

**Indentation** Comments should, typically, be placed before the code section
to which they apply.

**Short, Single line comments.** Short comments must lie on a single line.
The comment delimiters must lie on the same line.

*Incorrect*

```c
/*
* This is a single line comment
*/
```
*Correct*

```c
/* This is a single line comment. */
```

**Multi-line comments.** If the comment is too long to fit on a single, it must
be broken into a multi-line comment. The comment must be begin on the first
line of the multi-line comment with the opening comment delimiter `/*`. The
following lines of the multi-line comment must be with an asterisk `*` aligned
in the same column as the asterisk in the preceding line. The closing comment
delimiter must lie on a separate line with the asterisk `*` aligned in the same
column as the asterisk in the preceding line.

*Incorrect*

```c
/*
    This is the first line of a multi-line comment.
    This is the second line of a multi-line comment.
    This is the third line of a multi-line comment. */

/* This is the first line of another multi-line comment.  */
/* This is the second line of another multi-line comment. */
/* This is the third line of another multi-line comment.  */
```

*Correct*

```c
/* This is the first line of a multi-line comment.
 * This is the second line of a multi-line comment.
 * This is the third line of a multi-line comment.
 */
```
**Comments to the Right of Data Definitions.** Comments to the right of a
declaration with an enumeration or structure, on the other hand, are encouraged,
provided that the comments are short and do not exceed the maximum line
width (usually 80 characters). Columnar alignment of comments is very
desireable (but often cannot be achieved without violating the line width).

*Incorrect*

```c
struct animals_s
{
    int dog; /* This is a dog */
    int cat; /* This is a cat */
    double monkey; /* This is a monkey */
    double oxen; /* This is an oxen */
    bool aardvark; /* This is an aardvark */
    bool macaque; /* This is a macaque */
};
```

*Correct*

```c
struct animals_s
{
  int    dog;      /* This is a dog. */
  int    cat;      /* This is a cat. */
  double monkey;   /* This is a monkey. */
  double oxen;     /* This is an oxen. */
  bool   aardvark; /* This is an aardvark. */
  bool   macaque;  /* This is a macaque. */
};
```

**C Style Comments.** C99/C11/C++ style comments (beginning wih //) should not
be used with mTower. mTower generally follows C89 and all code outside of
architecture specific directories must be compatible with C89.

*Incorrect*

```c
// This is a structure of animals
struct animals_s
{
  int    dog;      // This is a dog
  int    cat;      // This is a cat
  double monkey;   // This is a monkey
  double oxen;     // This is an oxen
  bool   aardvark; // This is an aardvark
  bool   macaque;  // This is a macaque
};
```

*Correct*

```c
/** This is a structure of animals. */
struct animals_s
{
  int    dog;      /**< This is a dog. */
  int    cat;      /**< This is a cat. */
  double monkey;   /**< This is a monkey. */
  double oxen;     /**< This is an oxen. */
  bool   aardvark; /**< This is an aardvark. */
  bool   macaque;  /**< This is a macaque. */
};
```
**"Commenting Out" Large Code Blocks.** Do not use C or C++ comments to disable
compilation of large blocks of code. Instead, use #if 0 to do that. Make sure
there is a comment before the #if 0 to explain why the code is not compiled

*Incorrect*

```c
void some_function(void)
{
    ... compiled code ...

    /*
    ... disabled code ..
     */

    ... compiled code ...
}

void some_function(void)
{
  ... compiled code ...

  //
  // ... disabled code ..
  //

  ... compiled code ...
}
```
*Correct*

```c
void some_function(void)
{
  ... compiled code ...

  /* The following code is disabled because it is no longer needed. */
#  if 0
   ... disabled code ..
#  endif

  ... compiled code ...
}
```
### 1.4 Braces

**Never Comments on Braces.** Do not put comments on the same line as braces.

The preferred way, as shown to us by the prophets Kernighan and Ritchie, is to
put the opening brace last on the line, and put the closing brace first, thusly:

*Incorrect*

```c
while (true)
    {
        if (valid)
          {
           ...
          } /* if valid */
          else
          {
           ...
          } /* not valid */
    } /* end forever */
```

*Correct*

```c
while (true) {
  if (valid) {
    /* if valid */
    ...
  } else {
  /* not valid */
  ...
  }
  /* end forever */
}

```
This applies to all non-function statement blocks (if, switch, for, while, do):

*Correct*

```c
switch (action) {
  case 1:
    return "add";
  default:
    return NULL;
}

do {
  body of do-loop
} while (condition);
```

However, there is one special case, namely functions: they have the opening
brace at the beginning of the next line, thus:

*Correct*

```c
int function(int x)
{
  body of function
}
```
### 1.5 Indentation
**Indentation Unit.** Indentation is in units of two spaces; Each indentation
level is **twos spaces** further to the right than the preceding identation
levels.

**Indentation of Pre-Processor Lines.** C Pre-processor commands following any
conditional computation are also indented following basically the indentation
same rules, differing in that the # always remains in column 1.

*Incorrect*

```c
#ifdef CONFIG_ABC
#define ABC_THING1 1
#define ABC_THING2 2
#define ABC_THING3 3
#endif

#ifdef CONFIG_ABC
  #define ABC_THING1 1
  #define ABC_THING2 2
  #define ABC_THING3 3
#endif
```

*Correct*

```c
#ifdef CONFIG_ABC
#  define ABC_THING1 1
#  define ABC_THING2 2
#  define ABC_THING3 3
#endif

#ifdef CONFIG_ABC
#  define ABC_THING1 1
#  define ABC_THING2 2
#  define ABC_THING3 3
#endif
```
**Exception.** Each header file includes idempotence definitions at the
beginning of the header file. This conditional compilation does not cause any
change to the indentation. 

*Correct*

```c
#ifndef __INCLUDE_SOMEHEADER_H
#define __INCLUDE_SOMEHEADER_H
...
#define THING1 1
#define THING2 2
#define THING3 3
...
#endif /* __INCLUDE_SOMEHEADER_H */
```

### 1.6 Parentheses

**Space after key words.** Do not put a left parenthesis `(` immediately after
any C keywords (for, switch, while, do, return, etc.). Put a space before the
left parenthesis in these cases.

**Otherwise, no space before left parentheses.** Otherwise, there should be no
space before the left parentheses No space betwen function name and argument
list. There should be no space between a function name and an argument list.

**Never space before the right parentheses.** There should never be space before
a right parenthesis `)`.

**No parentheses around returned values.** Returned values should never be
enclosed in parentheses unless the parentheses are required to force the correct
order of operations in a computed return value.

*Incorrect*

```c
int do_foobar ( void )
{
	int ret = 0;
	int i;

	for( i = 0; ( ( i < 5 ) || ( ret < 10 ) ); i++ ) {
	  ret = foobar ( i );
	}

	return ( ret );
}
```

*Correct*

```c
int do_foobar(void)
{
  int ret = 0;
  int i;

  for (i = 0; i < 5 || ret < 10; i++) {
    ret = foobar(i);
  }

  return ret;
}
```

## 2. Data and Type Definitions

### 2.1 One Definition/Declaration Per Line

*Incorrect*

```c
extern long time, money;
char **ach, *bch;
int i, j, k;
```

*Correct*

```c
extern long time;
extern long money;
FAR char **ach;
FAR char *bch;
int i;
int j;
int k;
```

### 2.2 Global Variables

**Short global variable names.** Names should be terse, but generally
descriptive of what the variable is for. Try to say something with the variable
name, but don't try to say too much. For example, the variable name of 
`filelen` is preferable to something like `lengthoffile`.

**Module name prefix.** If a global variable belongs in a module with a name of,
say `xyz`, then that module should be included as part of the prefix like:
`xyz_`.

**lowerCamelCase.** Use lowerCamelCase style.

**Minimal use of '_'.** Preferably there are no '_' separators within the name.
Long variable names might require some delimitation using '_'. Long variable
names, however, are discouraged.

**Use structures.** If possible, wrap all global variables within a structure
to minimize the liklihood of name collisions.

**Avoid global variables when possible.** Use of global variables, in general,
is discourage unless there is no other reasonable solution.


*Incorrect*

```c
extern int someint;
static int anotherint;
uint32_t dwA32BitInt;
uint32_t gAGlobalVariable;
```

*Correct*

```c
struct my_variables_s
{
	uint32_t a32bitint;
	uint32_t aglobal;
};

extern int g_someint;
static int g_anotherint;
struct my_variables_s g_myvariables;
```

### 2.3 Parameters and Local Variables


**Common naming standard.** Naming for function parameters and local variables
is the same.

**Short variable names.** Names should be terse, but generally descriptive of
what the variable is for. Try to say something with the variable name, but
don't try to say too much. For example, the variable name of len is preferable
to something like lengthofiobuffer.

**No special ornamentation.** There is no special ornamentation of the name to
indication that the variable is a local variable. The prefix p or pp may be
used on names of pointers (or pointer to pointers) if it helps to distinguish
the variable from some other local variable with a similar name. Even this
convention is discouraged when not necessary.

**lowerCamelCase.** Use lowerCamelCase style.

**Minimal use of single character variable names.** Short variable names are
preferred. However, single character variable names should be avoided.
Exceptions to this include i, j, and k which are reserved only for use as loop
indices (part of our Fortran legacy).

**Minimal use of '_'.** Preferably there are no '_' separators within the name.
Long variable names might require some delimitation using '_'. Long variable
names, however, are discouraged.

*Incorrect*

```c
uint32_t somefunction(int a, uint32_t dwBValue)
{
  uint32_t this_is_a_long_variable_name = 1;
  int i;

  for (i = 0; i < a; i++) {
    this_is_a_long_variable_name *= dwBValue--;
  }

  return this_is_a_long_variable_name;
}
```

*Correct*

```c
uint32_t somefunction(int limit, uint32_t value)
{
  uint32_t ret = 1;
  int i;

  for (i = 0; i < limit; i++) {
    ret *= value--;
  }

  return ret;
}
```
> NOTE: You will see the local variable named ret is frequently used in the 
> code base for the name of a local variable whose value will be returned or to
> received the returned value from a called function. 

### 2.4 Type Definitions

**Short type names.** Type names should be terse, but generally descriptive of 
what the type is for. Try to say something with the type name, but don't try to
say too much. For example, the type name of fhandle_t is preferable to 
something like `openfilehandle_t`.

**Type name suffix.** All typedef'ed names end with the suffix `_t`.

**Module name prefix** If a type belongs in a module with a name of, say `xyz`,
then that module should be included as a prefix to the type name like: `xyz_`.

**lowerCamelCase.** Use lowerCamelCase style.

**Minimal use of '_'**. Preferably there are few `_` separators within the type
name. Long type names might require some delimitation using `_`. Long type
names, however, are discouraged.

*Incorrect*

```c
typedef void *myhandle;
typedef int myInteger;
```

*Correct*

```c
typedef void *myhandle_t;
typedef int myinteger_t;
```

### 2.5 Structures

#### Structure Naming

**No un-named structures.** All structures must be named, even if they are part
of a type definition. That is, a structure name must follow the reserved word
struct in all structure definitions.

**Structured defined with structures discouraged.** Fields within a structure
may be another structure that is defined only with the scope of the containing
structure. This practice is acceptable, but discouraged.

**No un-named structure fields.** Structure may contain other structures as
fields. This this case, the structure field must be named.

**Short structure names.** Structure names should be terse, but generally 
descriptive of what the structure contains. Try to say something with the 
structure name, but don't try to say too much. For example, the structure name
of `xyz_info_s` is preferable to something like
`xyz_datainputstatusinformation_s`.

**Structure name suffix.** All structure names end with the `suffix _s`.

**Module name prefix** If a structure belongs to a module with a name of,
say `xyz`, then that module should be included as a prefix to the structure
name like: `xyz_`.

**lowerCamelCase.** Use lowerCamelCase style.

**Minimal use of '_'**. Preferably there are few `_` separators within the
structure name. Long variable names might require some delimitation using `_`.


#### Structure Field Naming


**Common variable naming.** Structure field naming is generally the same as for
local variables.

**One definition per line.** The one definition per line rule applies to 
structure fields, including bit field definitions.

**Each field should be commented.** Each structure field should be commented.

**Optional structure field prefix.** It make be helpful to add a two-letter
prefix to each field name so that is is clear which structure the field belongs
to.

**lowerCamelCase.** Use lowerCamelCase style.

**Minimal use of '_'.** Preferably there are few `_` separators within the
field name. Long variable names might require some delimitation using `_`.
Long variable names, however, are discouraged.


*Incorrect*

```c
typedef struct       /* Un-named structure */
{
	...
	int val1, val2, val3; /* Values 1-3 */
	...
} xzy_info_t;

struct xyz_information
{
	...
	uint8_t bita : 1,  /* Bit A */
	bitb : 1,  /* Bit B */
	bitc : 1;  /* Bit C */
	...
};

```

*Correct*

```c
struct xyz_info_s
{
  ...
  uint8_t bita : 1,  /* Bit A */
  uint8_t bitb : 1,  /* Bit B */
  uint8_t bitc : 1,  /* Bit C */
  ...
};
```

### 2.6 Enumerations

**Enumeration Naming.** Naming of enumerations follow the same naming rules as
for structure and union naming. The only difference is that the `suffix _e`
is used to identify an enumeration.

**Uppercase.** Enumeration values are always in upper case.

**Use of '_' encouraged.** Unlike other naming, use of the underscore character
`_` to break up enumeration names is encouraged.

**Prefix.** Each value in the enumeration should begin with an upper-case
prefix that identifies the value as a member of the enumeration. This prefix
should, ideally, derive from the name of the enumeration.

*Correct*

```c
enum xyz_state_e
{
  XYZ_STATE_UNINITIALIZED = 0,  /**< Uninitialized state. */
  XYZ_STATE_WAITING,            /**< Waiting for input state. */
  XYZ_STATE_BUSY,               /**< Busy processing input state. */
  XYZ_STATE_ERROR,              /**< Halted due to an error. */
  XYZ_STATE_TERMINATING,        /**< Terminating stated. */
  XYZ_STATE_TERMINATED          /**< Terminating stated. */
};
```

### 2.7 C Pre-processor Macros

**Uppercase.** Macro names are always in upper case.

**Macros that could be functions.** Lower-case macro names are also acceptable
if the macro is a substitute for a function name that might be used in some
other context. In that case, normal function naming applies.

**Use of '_' encouraged.** Unlike other naming, use of the underscore character
`_` to break up macro names is encouraged.

**Prefix.** Each related macro value should begin with an upper-case prefix
that identifies the value as part of a set of values.

**Single space after #define.** A single space character should separate the 
`#define` from the macro name. Tabs are never used.

**Line continuations.** Macro definitions may be continued on the next line by
terminating the line with the `\` character just before the newline character.
There should be a single space before the `\` character. Aligned `\` characters
on multiple line continuations are discouraged because they are a maintenance
problem.

**Parentheses** around macro argument expansions. Macros may have argument
lists. In the macros expansion, each argument should be enclosed in parentheses.

**Real statements.** If a macro functions as a statement, then the macro 
expansion should be wrapped in `do { ... } while (0)` to assume that the macros
 is, indeed, a statement.

**Magic numbers are prohibited in code.** Any numeric value is not intuitively
obvious, must be properly named and provided as either a pre-processor macro
or an enumeration value.

*Incorrect*

```c
#define max(a,b) a > b ? a : b

#define ADD(x,y) x + y

#ifdef HAVE_SOMEFUNCTION
int somefunction(struct somestruct_s* psomething);
#else
#define SOMEFUNCTION() (0)
#endif

#	define	IS_A_CAT(c)		((c) == A_CAT)

#define LONG_MACRO(a,b)                                   \
  {                                                    \
    int value;                                        \
    value = b-1;                                      \
    a = b*value;                                      \
  }

#define DO_ASSIGN(a,b) a = b
```

*Correct*

```c
#define MAX(a,b) (((a) > (b)) ? (a) : (b))

#define ADD(x,y) ((x) + (y))

#ifdef HAVE_SOMEFUNCTION
int somefunction(struct somestruct_s* psomething);
#else
#  define somefunction(p) (0)
#endif

# define IS_A_CAT(c)  ((c) == A_CAT)

#define LONG_MACRO(a,b) \
  { \
    int value; \
    value = (b)-1; \
    (a) = (b)*value; \
  }

#define DO_ASSIGN(a,b) do { (a) = (b); } while (0)
```

## 3.0 Functions

### 3.1 Function Headers

**Function headers.** Each function is preceded by a function header. The
function header is a block comment that provides information about the function.
The block comment consists of the following:

* The block comment begins with a line that consists of the opening C comment
  in column 1 (/*) followed by a series of asterisks extending to the length of
  the line (usually to column 80).
* The block comment ends with a line that consists of series of asterisks
  beginning at column 2 and extending to the near the end line
  (usually to column 79) followed by the closing C comment in (usually at 
  column 80 for a total length of 79 characters).
* Information about the function is included in lines between the first and
  final lines. Each of these begin with a space in column 1, an sterisk (*)
  in column 2, and a space in column 3.

**Function header preceded by one blank line.** Exactly one blank line precedes
each function header.

**Function header followed by one blank line.** Exactly one blank line is
placed after function header and before the function definition 

**Function header sections.** Within the function header, the following data
sections must be provided:

* Name - Description: followed by a description of the function beginning on
  the second line. Each line of the function description is indented by two
  additional spaces.
* Parameters: followed by a description of the of each input parameter
  beginning on the second line. Each input parameter begins on a separator line
  indented by two additional spaces. The description needs to include (1) the
  name of the input parameters, and (2) a short description of the input 
  parameter.
* Returned Value: followed by a description of the of returned value(s)
  beginning on the second line. The description of the returned value should
  identify all error values returned by the function.

Each of these data sections is separated by a single line like " * ".

Function header template. Refer to Appendix A for the template for a function
header.

### 3.2 Function Naming

**Short function names.** Function names should be terse, but generally
descriptive of what the function is for. Try to say something with the function
name, but don't try to say too much. For example, the variable name of 
`xyz_putvalue` is preferable to something like
`xyz_savethenewvalueinthebuffer`.

**lowerCamelCase.** Use lowerCamelCase style.

**Module prefix.** All functions in the same module, or sub-system, or within
the same file should have a name beginning with a common prefix separated from
the remainder of the function name with the underscore, `_`, character. 
For example, for a module called `xyz`, all of the functions should begin with
`xyz_`.

**Extended prefix.** Other larger functional grouping should have another level
in the naming convention. For example, if module `xyz` contains a set of
functions that manage a set of I/O buffers (IOB), then those functions all
should get naming beginning with a common prefix like `xyz_iob_`.

**Use of '_' discouraged.** Further use of the `_` separators is discouraged in
function naming. Long function names might require some additional elimitation
using `_`. Long function names, however, are also discouraged.

### 3.3 Function Body

**Single compound statement.** The function body consists of a single compound
statement.

**Braces in column 1** The opening and close braces of the compound statement
must be placed in column one.

**First definition or statement in column 3.** The first data definitions
or statements in the function body are idented by **two space**.

**Long functions are discouraged.** The length of a function should be limited
so that it would fit on a single page.

**Space after the function boady.** A one (and only one) blank line must follow
the closing right brace of the function body.

### Returned Values

**OS Internal Functions.** In general, OS internal functions return a type `int`
to indicate success or failure conditions. Non-negative values indicate success.
The return value of zero is the typical success return value, but other
successful return can be represented with other positive values. Errors are
always reported with negative values. These negative values must be a 
well-defined jerrno as defined in the file include/jerrno.h.

**Checking Return Values.** Callers of internal OS functions should always check
return values for an error. At a minimum, a debug statement should indicate
that an error has occurred. The calling logic intentionally ignores the
returned value, then the function return value should be explicitly cast to
(void) to indicate that the return value is intentionally ignored.
All calls to malloc or realloc must be checked for failures to allocate memory.

## 4.0 Statements

### 4.1 One Statement Per Line

**One statement per line.** There should never be more than one statement on a
line.

**No more than one assignment per statement.** Related to this, there should
never be multiple assignments in the same statement.

**Statements should never be on the same line as any keyword.** Statements
should never be on the same line as case selectors or any C keyword.

*Incorrect*

```c
if (var1 < var2) var1 = var2;

case 5: var1 = var2; break;

var1 = 5; var2 = 6; var3 = 7;

var1 = var2 = var3 = 0;
```

*Correct*

```c
if (var1 < var2) {
  var1 = var2;
}

case 5:
  var1 = var2;
  break;

var1 = 5;
var2 = 6;
var3 = 7;

var1 = 0;
var2 = 0;
var3 = 0;
```

### 4.2 Operators

88Spaces before and after binary operators.** All binary operators (operators
that come between two values), such as +, -, =, !=, ==, >, etc. should have a
space before and after the operator, for readability. As examples: 

*Incorrect*

```c
fok=bar;
if(a==b)
for(i=0;i>5;i++)
```

*Correct*

```c
fok = bar;
if (a == b)
for (i = 0; i > 5; i++)

```

### 4.3 if then else Statement

*Incorrect*

```c
if(var1 < var2) var1 = var2;

if (var1 < var2)
	{
		var1 = var2;
	}

if (var > 0)
	{
		var--;
	}
else
	{
		var = 0;
	}
```

*Correct*

```c
if	(var1 < var2)
  var1 = var2;

if(var > 0)
  var--;
else
  var = 0;

if (var1 > 0) {
  var1--;
} else {
  var1 = 0;
}
```

`<condition> ? <then> : <else>`

**Only if the expression is short.** Use of this form is only appropriate if
the entire sequence is short and fits neatly on the line.

**Multiple lines forbidden.** This form is forbidden if it extends to more than
one line. 

*Correct*

```c
int arg1 = arg2 > arg3 ? arg2 : arg3;
int arg1 = ((arg2 > arg3) ? arg2 : arg3);
```

### 4.4 switch Statement

*Correct*

```c
switch (...) {
  case 1:  /* Example of a comment following a case selector. */
    ...
  /* Example of a comment preceding a case selector. */
  case 2:
    /* Example of comment following the case selector. */
    int value;
    ...
    break;
  default:
    break;
}
```

### 4.5 while Statement

*Incorrect*

```c
while( notready() )
{
}
ready = true;

while (*ptr != '\0') ptr++;
```

*Correct*

```c
while (notready());

ready = true;

while (*ptr != '\0') {
  ptr++;
}
```
### 4.6 do while Statement

*Incorrect*

```c
do
	{
		ready = !notready();
	}
	while (!ready);
senddata();

do ptr++; while (*ptr != '\0');
```

*Correct*

```c
do {
  ready = !notready();
} while (!ready);

senddata();

do {
  ptr++;
} while (*ptr != '\0');
```
### 4.7 Use of goto

**Limited Usage of goto.** All use of the goto statement is prohibited except
for one usage: for handling error conditions in complex, nested logic. A simple
goto in those conditions can greatly improve the readability and complexity of
the code.

**Label Naming.** Labels must all lower case.

**Label Positioning.** Labels are never indented. Labels must always begin in
column 1.

*Correct*

```c
struct some_struct_s *ptr;
int fd;
int ret;
...

if (arg == NULL) {
  ret = -EINVAL;
  goto errout;
}
...
ptr = (struct some_struct_s *)malloc(sizeof(struct some_struct_s));
if (!ptr) {
  ret = -ENOMEM;
  goto errout_with_alloc;
}
...

errout_with_alloc:
  free(ptr);

errout:
  return ret;
```
## Appendix A

### A.1 C Source File Structure

```c
/**
 * @file        <Relative path to the file>
 * @brief       <Optional one line file description>
 *
 * @copyright   Copyright (c) 2018 Samsung Electronics Co., Ltd. All Rights Reserved.
 * @author      <Author's name> <Contact e-mail>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @todo
 */

/* Included Files. */
/* All header files are included here. */


/* Pre-processor Definitions. */
/* All C pre-processor macros are defined here. */


/* Private Types. */
/* Any types, enums, structures or unions used by the file are defined here. */


/* Private Function Prototypes. */
/* Prototypes of all static functions in the file are provided here. */


/* Private Data. */
/* All static data definitions appear here. */

/* Public Data. */
/* All data definitions with global scope appear here. */

/* Private Functions. */

/**
 * @brief         <Static function name> - Description of the operation
 *                of the static function.
 *
 * @param value   [in]/[out] A list of input parameters, one-per-line,
 *                appears here along with a description of each input
 *                parameter.
 *
 * @returns       Description of the value returned by this function
 *                (if any), including an enumeration of all possible
 *                error values.
 */

/* All static functions in the file are defined in this grouping. Each is
 * preceded by a function header similar to the above.
 */

/* Public Functions. */

/**
 * @brief         <Function name > - Description of the operation of
 *                the static function.
 *
 * @param value   [in]/[out] A list of input parameters, one-per-line,
 *                appears here along with a description of each input
 *                parameter.
 *
 * @returns       Description of the value returned by this function
 *                (if any), including an enumeration of all possible
 *                error values.
 */

/* All global functions in the file are defined here. */
```

### A.2 C Header File Structure

```c
/**
 * @file        <Relative path to the file>
 * @brief       <Optional one line file description>
 *
 * @copyright   Copyright (c) 2018 Samsung Electronics Co., Ltd. All Rights Reserved.
 * @author      <Author's name> <Contact e-mail>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @'todo
 */

#ifndef __JVMNMI_CRYPTO_CIPHER_H_
#define __JVMNMI_CRYPTO_CIPHER_H_

/* Included Files */
/* All header files are included here. */

/* Pre-processor Definitions */
/* All C pre-processor macros are defined here. */

/* Public Types */
/* Any types, enumerations, structures or unions are defined here. */

/* Public Data */
/* All data declarations with global scope appear here, preceded by 
 * the definition EXTERN.
 */

/* Inline Functions */

/**
 * @brief         <Inline function name> - Description of the operation
 *                of the static function.
 *
 * @param value   [in]/[out] A list of input parameters, one-per-line,
 *                appears here along with a description of each input
 *                parameter.
 *
 * @returns       Description of the value returned by this function
 *                (if any), including an enumeration of all possible
 *                error values.
 */

/* Any static inline functions may be defined in this grouping. 
 * Each is preceded by a function header similar to the above.
 */

/* Public Function Prototypes */

/**
 * @brief         <Global function name> - Description of the operation
 *                of the static function.
 *
 * @param value   [in]/[out] A list of input parameters, one-per-line,
 *                appears here along with a description of each input
 *                parameter.
 *
 * @returns       Description of the value returned by this function
 *                (if any), including an enumeration of all possible
 *                error values.
 */
 
/* All global functions in the file are prototyped here. The keyword
 * extern or the definition EXTERN are never used with function
 * prototypes.
 */

#endif /* __JVMNMI_CRYPTO_CIPHER_H_ */
```

