function [varargout] = logLikelihood_ConversionReaction(xi,D,options,approach)
% logLikelihood_ConversionReaction() computes the log-likelihood function for
% the conversion reaction
%
% USAGE:
% * [lLH] = logLikelihood_ConversionReaction(...)
% * [lLH,gradlLH] = logLikelihood_ConversionReaction(...)
% * [...] = logLikelihood_ConversionReaction(xi,D,options,approach)
%
% Parameters
%  xi: parameter for which log-likelihood is evaluated
%  D: data (see logLikelihoodHierarchical.m for the definition of the
%  data in the hierarchical case)
%  options.MS.HO:  A HOOptions object holding various options for the algorithm
%  approach: 'hierarchical' or 'standard' approach for the optimization
%
% Return values:
%   varargout:
%     lLH: Log-Likelihood, only the log-likelihood will be returned, no
%         sensitivity analysis is performed
%     gradlLH: Gradient of lLH, the log-likelihood and its gradient will be
%         returned

try
    options.ami.atol = 1e-12;
    options.ami.rtol = 1e-12;
    if nargout>1
        options.ami.sensi = 1;
    else
        options.ami.sensi = 0;
    end
    
    %% SIMULATION
    sol = simulate_ConversionReaction(D.t,xi(1:2),[],[],options.ami);
    switch approach
        case 'hierarchical'
            simulation(1).y = sol.y;
            if nargout > 1
                simulation(1).sy = sol.sy;
            end
            %% LOG-LIKELIHOOD, GRADIENT
            if nargout == 2
                [lLH, gradlLH] = logLikelihoodHierarchical(simulation,D,options.MS.HO);
            else
                lLH = logLikelihoodHierarchical(simulation,D,options.MS.HO);
            end
        case 'standard'
            % add scaling factor
            if nargout > 1
                sol.sy(:,:,3:4) = 0;
                ds = [0,0,0,10.^xi(4)*log(10)];
            end
            switch options.MS.HO.distribution
                case 'normal'
                    noise = 10.^(2*xi(3));
                    if nargout > 1
                        dnoise = [0,0,2*2.^(2*xi(3))*25.^xi(3)*log(10),0];
                    end
                case 'laplace'
                    noise = 10.^(xi(3));
                    if nargout > 1
                        dnoise = [0,0,10.^xi(3)*log(10),0];
                    end
                    
            end
            s = 10.^xi(4);
            y_sh = D.my-sol.y*s;
            switch options.MS.HO.distribution
                case 'normal'
                    temparg = 0.5*(bsxfun(@times,~isnan(D.my),log(2*pi*noise))+...
                        y_sh.^2./noise);
                case 'laplace'
                    temparg = bsxfun(@times,~isnan(D.my),log(2*noise))+...
                        abs(y_sh)./noise;
            end
            lLH = -sum(temparg);
            
            if nargout > 1
                dy_sh = bsxfun(@plus,-s*squeeze(sol.sy), -bsxfun(@times,ds,sol.y));
                
                switch options.MS.HO.distribution
                    case 'normal'
                        gradlLH = -sum(...
                            bsxfun(@plus,bsxfun(@times,bsxfun(@rdivide,y_sh,noise),dy_sh),...
                            bsxfun(@plus,0.5*bsxfun(@times,1/noise,dnoise),...
                            -0.5*bsxfun(@times,bsxfun(@rdivide,y_sh.^2,noise.^2),dnoise)))...
                            ,1)';
                    case 'laplace'
                        gradlLH = -sum(...
                            bsxfun(@plus,bsxfun(@times,bsxfun(@rdivide,sign(y_sh),noise),dy_sh),...
                            bsxfun(@plus,bsxfun(@times,1/noise,dnoise),...
                            -bsxfun(@times,bsxfun(@rdivide,abs(y_sh),noise.^2),dnoise)))...
                            ,1)';
                end
            end
            
    end
    assert(sol.status>=0)
catch error_thrown
    warning(['Evaluation of likelihood failed. ',error_thrown.message]);
    lLH = nan;
    gradlLH = nan(length(xi),1);
end

switch nargout
    case{0,1}
        varargout{1} = lLH;
    case 2
        varargout{1} = lLH;
        varargout{2} = gradlLH;
end
end

