function varargout = logLikelihood_histones_standard(xi,varargin)
% This function evaluates the likelihood function for the standard
% approach of optimization.
%
% USAGE:
% [...] = logLikelihood_histones_standard(xi,DA,simulateA,options) \n
% [logL] = logLikelihood_histones_standard(...) \n
% [logL, dlogL] = logLikelihood_histones_standard(...) \n
%
% Parameters:
%   xi: parameter values
%   varargin:
%   DA: data
%   simulateA: simulation function
%   options: struct
%
% Required fields of options.llh:
%    n_theta: number of parameters needed for the simulation
%    input: input for the simulation functions, needs to be the same for
%           both, could be adapted if the scenarios need different inputs
%    prior: with subfields mean and sigma2 (normally distributed prior)
%    ind_regularization: indices which parameters to penalize
%    lambda_regularization: regularization strength
%
% Return values:
%   logL: log-likelihood value
%   dlogL: gradient of log-likelihood function


%% Default options
options.llh.ind_regularization = [];
options.llh.lambda_regularization = 0;

%% Input assignemnt
DA = varargin{1};
simulateA = varargin{2};
if nargin == 4
    options = setdefault(varargin{3},options);
end
n_xi = length(xi);

try
    %% Initialization
    nderiv = nargout-1;
    logL = 0;
    if(nderiv>=1)
        dlogL = zeros(n_xi,1);
    end
    % AMICI options
    options.amiA = options.ami;
    if(nderiv>=1)
        options.amiA.sensi = 1;
    else
        options.amiA.sensi = 0;
    end
    %% simulation for A
    xiA = xi(1:options.llh.n_theta);
    solA = simulateA(DA(1).t,xiA,options.llh.input,options.amiA);
    
    if (solA.status<0)
        varargout{1} = NaN;
        if nderiv>=1
            varargout{2} = nan(n_xi,1);
        end
        return;
    end
    
    % Compute likelihood
    noise = 10.^(xi(end));
    y_sh = bsxfun(@minus,DA.my,solA.y);
    
    switch options.MS.HO.distribution
        case 'normal'
            temparg = 0.5*(bsxfun(@times,~isnan(DA.my),log(2*pi*noise))+...
                y_sh.^2./noise);
        case 'laplace'
            temparg = bsxfun(@times,~isnan(DA.my),log(2*noise))+...
                abs(y_sh)./noise;
    end
    logL = -nansum(nansum(nansum(temparg)));
    
    if nderiv >= 1
        solA.sy(:,:,end+1) = 0;
        dnoise = [zeros(1,numel(xi)-1),10.^xi(end)*log(10)];
        dy_sh_tmp = -squeeze(solA.sy);
        dy_sh = nan(size(y_sh,1),size(y_sh,2),size(y_sh,3),size(dy_sh_tmp,3));
        for r = 1:size(DA.my,3)
            dy_sh(:,:,r,:) = dy_sh_tmp;
        end
        dnoise_tmp = nan(size(dy_sh));
        for i1 = 1:size(dy_sh,1)
            for i2 = 1:size(dy_sh,2)
                for i3 = 1:size(dy_sh,3)
                    dnoise_tmp(i1,i2,i3,:) = dnoise;
                end
            end
        end
        dnoise = dnoise_tmp;
        
        switch options.MS.HO.distribution
            case 'normal'
                dlogL = -squeeze(nansum(nansum(nansum(...
                    bsxfun(@plus,bsxfun(@times,bsxfun(@rdivide,y_sh,noise),dy_sh),...
                    bsxfun(@plus,0.5*bsxfun(@times,1/noise,dnoise),...
                    -0.5*bsxfun(@times,bsxfun(@rdivide,y_sh.^2,noise.^2),dnoise)))...
                    ,1),2),3));
            case 'laplace'
                dlogL = -squeeze(nansum(nansum(nansum(...
                    bsxfun(@plus,bsxfun(@times,bsxfun(@rdivide,sign(y_sh),noise),dy_sh),...
                    bsxfun(@plus,bsxfun(@times,1/noise,dnoise),...
                    -bsxfun(@times,bsxfun(@rdivide,abs(y_sh),noise.^2),dnoise)))...
                    ,1),2),3));
        end
    end
    
    % prior
    logL = logL - 0.5*nansum((xi-options.llh.prior.mean).^2./ ...
        options.llh.prior.sigma2);
    if nargout >=2
        dlogL = nansum([dlogL,-(xi-options.llh.prior.mean)./ ...
            options.llh.prior.sigma2],2);
    end
    
    % L1 regularization
    if ~isempty(options.llh.ind_regularization)
        logL = logL - sum(options.llh.lambda_regularization*abs(xi(options.llh.ind_regularization)));
        if nargout >=2
            dlogL([options.llh.ind_regularization]) = dlogL([options.llh.ind_regularization]) ...
                - options.llh.lambda_regularization*sign(xi(options.llh.ind_regularization));
        end
    end
    
    %% Output assignment
    varargout{1} = logL;
    if nargout >= 2
        varargout{2} = dlogL;
        if sum(isnan(dlogL) > 0)
            varargout{1} = NaN;
            varargout{2} = nan(n_xi,1);
        end
        if sum(isinf(dlogL) > 0)
            varargout{1} = NaN;
            varargout{2} = nan(n_xi,1);
        end
    end
catch e
    disp(e.message)
    varargout{1} = NaN;
    if nderiv>=1
        varargout{2} = nan(n_xi,1);
    end
end


