function [fh,theoretical_quant,data] = myqqplot(varargin)
% Routine to create qq plots providing either the probability density
% function and numerically integrate it, or providing the empirical
% cumulative distribution function.
%
% USAGE:
% [fh,theoretical_quant,data] = myqqplot(data,F,'cdf')
% [fh,theoretical_quant,data] = myqqplot(data,f,'pdf')
% [fh,theoretical_quant,data] = myqqplot(data,f,'pdf','b')
%
% Parameters:
%   data: residual values at the maximum likelhood estimate
%   F: if 'cdf' is used: function handle for empirical cumulative
%   distribution function minus required percentile, with two inputs x
%   and the percentile. Uses then Matlab function fzero to numerically
%   invert the function.
%   f: if 'pdf' is used: function handle for probability densitivy
%   function with one input x for the value where the function should be
%   evaluated.
%   DA: data for scenario A
%   DB: data for scenario B
%   simulateA: simulation function for scenario A
%   simulateB: simulation function for scenario B
%   options: struct
%
data = varargin{1};
mode = varargin{3};
switch mode
    case 'cdf'
        F = varargin{2};
    case 'pdf'
        f = varargin{2};
end
if nargin > 3
    col = varargin{4};
else
    col = 'b';
end

data = sort(data(~isnan(data)));

% if only pdf is provided, numerically compute ecdf
if strcmp(mode,'pdf')
    % find good lower value
    lb = 0;
    lbs = linspace(-1e6,lb,20);
    while max(f(lbs)) > 1e-10
        lb = lb-1;
        lbs = linspace(-1e6,lb,20);
    end
    F = @(x,p) trapz(linspace(lb,x),f(linspace(lb,x))) - p;
end

for i = 1:length(data)
    % check at i-0.5 because there the it does not "jump"
    empirical_quant(i) = (i-0.5)./length(data);
    theoretical_quant(i) = fzero(@(x) F(x,empirical_quant(i)),data(i));
end

% Generate QQ-plot
fh = figure;
xgrid = linspace(min(min(theoretical_quant),data(1))-0.2,max(max((theoretical_quant),data(end)))+0.2);
plot(xgrid,xgrid,'k-'); hold on;
plot(theoretical_quant,data,'.','MarkerSize',8,'Color',col); hold on
xlim([xgrid(1),xgrid(end)]);
ylim([xgrid(1),xgrid(end)]);
xlabel('theoretical quantile')
ylabel('empirical quantile')
axis square
set(gca,'Position',[0.3,0.3,0.6,0.6]);

end