function fh = plot_histones(xi,varargin)
% Routine for the visualization of the histone methylation data
% and model fits.
% [...] = plot_histones(xi,DA,simulateA,options) \n
% [...] = plot_histones(xi,DA,DB,simulateA,simulateB,options)\n

% options and the defaults:
options.plot.samefigure = 0; % same figure for compared data sets
options.plot.errorbars = 0; % if mean and std. dev. of data should be plotted
options.plot.stddev = 0; % if mean and std. dev. of model should be plotted
options.plot.dataflag = 1; % if data should be plotted
options.plot.modelflag = 1; % if model should be plotted

options.plot.A.color.model = 'b';
options.plot.A.color.data = 'k';
options.plot.A.facealpha.data = 1;
options.plot.A.ls = '-';
options.plot.A.marker = '.';
options.plot.A.markersize = 8;

options.plot.B.color.model = 'g';
options.plot.B.color.data = 'r';
options.plot.B.facealpha.data = 1;
options.plot.B.ls = '--';
options.plot.B.marker = '.';
options.plot.B.markersize = 8;

options.plot.errorbarwidth = 0.1;

options.plot.tsim = [];
options.plot.legendflag = 1;
options.plot.plain = 1;
options.plot.fs = 6;
options.plot.lw = 1;
options.plot.model.lw = 1;
options.plot.title = false;
options.plot.percentage = 100;
options.plot.holdon = 0;
options.plot.pos_flag = false;
options.plot.generations = [1:3];
options.plot.observables = [1:15];

options.llh.input = 0;
options.llh.scale = 'log';
options.llh.inhibitor_light = false;

if nargin > 4
    DA = varargin{1};
    DB = varargin{2};
    simulateA = varargin{3};
    simulateB = varargin{4};
    if nargin == 6
        options = setdefault(varargin{5},options);
    end
    compareflag = 1;
else
    DA = varargin{1};
    simulateA = varargin{2};
    if nargin == 4
        options = setdefault(varargin{3},options);
    end
    compareflag = 0;
end

options.plot.samefigure = 0;

load_plot_settings

if isempty(simulateA) || isempty(xi)
    options.plot.modelflag = 0;
else
    n_xi = length(xi);
    n_theta = options.llh.n_theta;
end

%% SIMULATION
if options.plot.modelflag
    options.amiA = options.ami;
    options.ami.sensi = 0;
    if isempty(options.plot.tsim)
        options.plot.tsim = linspace(DA(1).t(1),DA(1).t(end));
    end
    tsim = options.plot.tsim;
    xiA = xi(1:options.llh.n_theta);
    solA = simulateA(tsim,xiA,options.llh.input,options.amiA);
    options.llh.save_analytical = true;
    
    if options.plot.stddev && ~compareflag
        ll = logLikelihood_histones_hierarchical(xi,DA,simulateA,options);
        load('analytical_results')
    elseif options.plot.stddev && ~compareflag
        ll = logLikelihood_histones_hierarchical(xi,DA,DB,simulateA,simulateB,options);
        load('analytical_results')
    end
    
    if compareflag
        xiB = xi(1:options.llh.n_theta);
        for i = 1:numel(options.llh.diffB_theta)
            xiB(options.llh.diffB_theta{i}) = xiB(options.llh.diffB_theta{i}) + ...
                xi(options.llh.diffB_xiind(i));
        end
        if options.llh.inhibitor_light
            % use xiA to preequilibrate
            solB = simulateB(tsim,xiA,xiB,options.llh.input,options.amiA);
        else
            solB = simulateB(tsim,xiB,options.llh.input,options.amiA);
        end
    end
    
end

n_obs = 15;
n_e = size(DA.my,3);

sx = round(sqrt(n_obs));
sy = ceil(n_obs/sx);

obs_names = {'00','01','02','03','10','11','12','13','20','21','22','23',...
    '30','31','32','33'};

%% PLOTTING
for imed = options.plot.generations
    if isfield(options.plot,'fh') && ~isempty(options.plot.fh)
        figure(options.plot.fh{imed});
        fh = options.plot.fh;
    else
        fh{imed} = figure;
    end
    col = options.plot.A.color.model;
    colB = options.plot.B.color.model;
    
    switch imed
        case 1
            %col = color.light;
            indices = options.plot.observables;
        case 2
            %col = color.medium;
            indices = options.plot.observables+15;
        case 3
            %col = color.heavy;
            indices = options.plot.observables+30;
    end
    for iobs = 1:length(options.plot.observables)
        if options.plot.modelflag
            if options.plot.pos_flag
                subplot('Position',options.plot.positions{iobs});
            else
                subplot(sx,sy,iobs);
            end
            if strcmp(options.plot.A.ls,'---')
                y = options.plot.percentage*...
                    (exp(solA.y(:,indices(iobs)))-options.llh.input)';
                yDashed = y;
                yDashed(5:5:length(yDashed)) = NaN;
                lmA = plot(options.plot.tsim,yDashed,...
                    'color',col,...
                    'LineWidth',options.plot.model.lw,...
                    'LineStyle','-');
            else
                lmA = plot(options.plot.tsim,options.plot.percentage*...
                    (exp(solA.y(:,indices(iobs)))-options.llh.input),...
                    'color',col,...
                    'LineWidth',options.plot.model.lw,...
                    'LineStyle',options.plot.A.ls);
            end
            hold on;
            if options.plot.stddev
                fill([tsim fliplr(tsim)],...
                    [options.plot.percentage*exp(solA.y(:,indices(iobs))'+...
                    sqrt(2*sigma2(1))-options.llh.input) ...
                    fliplr(options.plot.percentage*(exp(solA.y(:,indices(iobs))'...
                    -sqrt(2*sigma2(1)))-options.llh.input))],1,...
                    'facealpha',[0.3],...
                    'edgecolor','none',...
                    'facecolor',col); hold on;
            end
            xlim([options.plot.tsim(1)-1,options.plot.tsim(end)+1]);
            
            if compareflag && ismember(DB.generations,imed)
                if strcmp(options.plot.B.ls,'---')
                    y = options.plot.percentage*...
                        (exp(solB.y(:,iobs))-options.llh.input)';
                    yDashed = y;
                    yDashed(5:5:length(yDashed)) = NaN;
                    lmB = plot(options.plot.tsim,yDashed,...
                        'color',colB,...
                        'LineWidth',options.plot.model.lw,...
                        'LineStyle','-');
                else
                    lmB = plot(options.plot.tsim,options.plot.percentage*...
                        (exp(solB.y(:,iobs))-options.llh.input),...
                        'color',colB,...
                        'LineWidth',options.plot.model.lw,...
                        'LineStyle',options.plot.B.ls);
                end
                hold on;
                % indices hard coded
                if options.plot.stddev
                    fill([tsim fliplr(tsim)],...
                        [options.plot.percentage*exp(solB.y(:,iobs)'+...
                        sqrt(2*sigma2(1))-options.llh.input) ...
                        fliplr(options.plot.percentage*(exp(solB.y(:,iobs)'...
                        -sqrt(2*sigma2(1)))-options.llh.input))],1,...
                        'facealpha',[0.3],...
                        'edgecolor','none',...
                        'facecolor',col); hold on;
                end
            end
        else
            xlim([DA.t(1)-1,DA.t(end)+1]);
        end
        
        if options.plot.dataflag
            if iobs <= n_obs
                % plot data for first data set
                if options.plot.pos_flag
                    subplot('Position',options.plot.positions{iobs});
                else
                    subplot(sx,sy,iobs);
                end
                for idata = 1:n_e
                    switch options.llh.scale
                        case 'log'
                            Ys(idata,:) = options.plot.percentage*...
                                ((exp(DA.my(:,indices(iobs),idata))-options.llh.input));
                        case 'lin'
                            Ys(idata,:) = options.plot.percentage*DA.my(:,indices(iobs),idata);
                    end
                    if ~options.plot.errorbars
                        ldA = plot(DA.t, Ys(idata,:),options.plot.A.marker,...
                            'MarkerSize',options.plot.A.markersize,...
                            'color',options.plot.A.color.data); hold on;
                    end
                    % hard coded indices
                    if compareflag && idata <= size(DB.my,3) && ismember(imed,DB.generations)
                        switch options.llh.scale
                            case 'log'
                                YsB(idata,:) = options.plot.percentage*...
                                    ((exp(DB.my(:,iobs,idata))-options.llh.input));
                            case 'lin'
                                YsB(idata,:) = options.plot.percentage*DB.my(:,iobs,idata);
                        end
                        if ~options.plot.errorbars
                            ldB = plot(DB.t, YsB(idata,:),options.plot.B.marker,...
                                'MarkerSize',options.plot.B.markersize,...
                                'color',options.plot.B.color.data); hold on;
                        end
                    end
                end
                if options.plot.errorbars
                    nmYsA = nanmean(Ys);
                    nsYsA = nanstd(Ys);
                    for k = 1:numel(DA.t)
                        [ldA,~] = myerrorbar_histones(DA.t(k), nmYsA(k),nsYsA(k),...
                            options,'A'); hold on;
                    end
                    if compareflag && ismember(imed,DB.generations)
                        nmYsB = nanmean(YsB);
                        nsYsB = nanstd(YsB);
                        for k = 1:numel(DA.t)
                            [ldB,~] = myerrorbar_histones(DB.t(k), nmYsB(k),nsYsB(k),...
                                options,'B'); hold on;
                        end
                    end
                end
                
            end
        end
        % title and labels
        if iobs <= n_obs
            if strcmp(options.llh.scale,'log') & options.plot.modelflag
                ylimits{1} = get(gca,'ylim');
                set(gca,'ylim',[0,ylimits{1}(2)]);
            else
                ylimits{1} = get(gca,'ylim');
                set(gca,'ylim',[0,ylimits{1}(2)]);
            end
            if iobs == 12
                ylim([0,1]);
            end
            box off
            if iobs <12
                set(gca,'xtick',[DA.t],'xticklabel','');
            else
                % DA.t is shifted by 3
                if DA.t(1) == 0 % actually starting at -3
                    set(gca,'xtick',[DA.t],'xticklabel',{'','0','','8','16','24'});
                else  % starting at 0
                    set(gca,'xtick',[0,4,8,12,16,24]+3,...
                        'xticklabel',{'0','4','8','12','16','24'});
                end
            end
        end
        if options.plot.title
            title(obs_names{options.plot.observables(iobs)});
        end
        set(gca,'ticklength',[0.01,0.01],'FontSize',options.plot.fs)
    end
    
    % legend
    if options.plot.legendflag
        if options.plot.modelflag
            legend([lmA,ldA],['model'],['data'])
        else
            legend(ldA,['data'])
        end
    end
    
end