function varargout = logLikelihood_histones_hierarchical(xi,varargin)
% This function evaluates the likelihood function for the hierarchical
% parametrization of the histone models. It can be used either for one data
% set, or for more data sets for which different parameters are defined.
%
% USAGE:
% [...] = logLikelihood_histones_hierarchical(xi,DA,simulateA,options) \n
% [...] = logLikelihood_histones_hierarchical(xi,DA,DB,simulateA,simulateB,options)\n
% [logL] = logLikelihood_histones_hierarchical(...) \n
% [logL, dlogL] = logLikelihood_histones_hierarchical(...) \n
%
% Parameters:
%   xi: parameter values
%   varargin:
%   DA: data for scenario A
%   DB: data for scenario B
%   simulateA: simulation function for scenario A
%   simulateB: simulation function for scenario A
%   options: struct
%
% Required fields of options.llh:
%    save_analytical: (default) false, if analytical standard deviation
%                     should be saved
%    distribution: distribution assumption to be used, 'laplace-analytical'
%                   (more are implemented but no longer used)
%    input: input for the simulation functions, needs to be the same for
%           both, could be adapted if the scenarios need different inputs
%    prior: with subfields mean and sigma2 (normally distributed prior)
%    diffB_theta: indices of simulation parameters which are different for scenario B
%    diffB_xiind: indices of whole parameter vector which ae added to the
%                 simulation parameters for scenario B
%                 (theta_B(diffB_theta) = \theta_A(diffB_theta)
%                 + xi(diffB_xiind))
%   ind_regularization: indices which parameters to penalize with L1 norm
%   lambda_regularization: regularization strength of L1 norm
%   inhibitor_light: (default) false, if the parameters for scenario A are
%                  used to preequlibrate simulation for scenario B
%
% the following options are no longer used
%    fixsigma: (default) false, if sigma is not estimated but fixed, never
%               tested, use very carefully
%
% Return values:
%   logL: log-likelihood value
%   dlogL: gradient of log-likelihood function


%% Default options
options.llh.fixsigma = false;
options.llh.save_analytical = false;
options.llh.inhibitor_light = false;
options.llh.ind_regularization = [];
options.llh.lambda_regularization = 0;

%% Input assignment
if nargin > 4
    DA = varargin{1};
    DB = varargin{2};
    simulateA = varargin{3};
    simulateB = varargin{4};
    if nargin == 6
        options = setdefault(varargin{5},options);
    end
    compareflag = 1; % if two data sets and simulations are provided
else
    DA = varargin{1};
    simulateA = varargin{2};
    if nargin == 4
        options = setdefault(varargin{3},options);
    end
    compareflag = 0;
end
n_xi = length(xi);
%n_theta = options.llh.n_theta;

try
    %% Initialization
    nderiv = nargout-1;
    logL = 0;
    if(nderiv>=1)
        dlogL = zeros(n_xi,1);
    end
    
    % AMICI options  
    options.amiA = options.ami;
    if compareflag
        options.amiB = options.amiA;
    end
    if(nderiv>=1)
        options.amiA.sensi = 1;
        options.amiB.sensi = 1;
    else
        options.amiA.sensi = 0;
        options.amiB.sensi = 0;
    end
    %% simulation for scenario A
    xiA = xi(1:options.llh.n_theta);
    solA = simulateA(DA(1).t,xiA,options.llh.input,options.amiA);
    
    if options.llh.fixsigma
        solA.sigmay = DA(1).sigmay;
        solA.ssigmay = zeros(size(solA.ssigmay));
    end
    
    if (solA.status<0)
        varargout{1} = NaN;
        if nderiv>=1
            varargout{2} = nan(n_xi,1);
        end
        return;
    end
    %% simulation for scenario B
    if compareflag
        xiB = xi(1:options.llh.n_theta); % parameters for simulation of scenario B
        for i = 1:numel(options.llh.diffB_theta)
            xiB(options.llh.diffB_theta{i}) = xiB(options.llh.diffB_theta{i}) + ...
                xi(options.llh.diffB_xiind(i));
        end
        if options.llh.inhibitor_light
            % use xiA to preequilibrate, simulateB requires xiA as input
            solB = simulateB(DB(1).t,xiA,xiB,options.llh.input,options.amiB);
        else
            solB = simulateB(DB(1).t,xiB,options.llh.input,options.amiB);
        end
        if options.llh.fixsigma
            solB.sigmay = DB(1).sigmay;
            solB.ssigmay = zeros(size(solB.ssigmay));
        end
        if (solB.status<0)
            varargout{1} =NaN;
            if nderiv>=1
                varargout{2} = nan(n_xi,1);
            end
            return;
        end
    end
    
    % Compute likelihood
    if ~compareflag
        if nderiv < 1
            templogL = logLikelihoodHierarchical(solA,DA,options.MS.HO);
            logL = logL + templogL;
        else
            [templogL,tempdlogL] = logLikelihoodHierarchical(solA,DA,options.MS.HO);
            logL = logL + templogL;
            dlogL = dlogL + tempdlogL;
        end
    else % merge/concatenate simulations and data set
        solAB.y=[solA.y,solB.y];
        if nderiv > 0
            solAB.sy = zeros(size(solAB.y,1),size(solAB.y,2),n_xi);
            solAB.sy(:,1:size(solA.sy,2),1:size(solA.sy,3)) = solA.sy;
            if options.llh.inhibitor_light
                solAB.sy(:,size(solA.sy,2)+1:end,1:size(solA.sy,3)) = ...
                    solB.sy(:,:,1:length(xiA)) + solB.sy(:,:,length(xiA)+1:end);
                for i = 1:numel(options.llh.diffB_theta)
                    solAB.sy(:,size(solA.sy,2)+1:end,options.llh.diffB_xiind(i)) = ...
                        sum(solB.sy(:,:,options.llh.diffB_theta{i}),3);
                end
            else
                solAB.sy(:,size(solA.sy,2)+1:end,1:length(xiA)) = solB.sy;
                for i = 1:numel(options.llh.diffB_theta)
                    solAB.sy(:,size(solA.sy,2)+1:end,options.llh.diffB_xiind(i)) = ...
                        sum(solAB.sy(:,size(solA.sy,2)+1:end,...
                        options.llh.diffB_theta{i}),3);
                end
            end
        end
        % merged data set, required for hierarchical optimization
        DAB = DA;
        DAB.my = nan(size(DA.my,1),size(DA.my,2)+size(DB.my,2),max(size(DA.my,3),size(DB.my,3)));
        DAB.my(:,1:size(DA.my,2),1:size(DA.my,3)) = DA.my;
        DAB.my(:,size(DA.my,2)+1:size(DA.my,2)+size(DB.my,2),1:size(DB.my,3)) = DB.my;
        if nderiv <= 0
            templogL = logLikelihoodHierarchical(solAB,DAB,options.MS.HO);
            logL = logL + templogL;
        else
            [templogL,tempdlogL] = logLikelihoodHierarchical(solAB,DAB,options.MS.HO);
            logL = logL + templogL;
            dlogL = dlogL + tempdlogL;
        end
    end
    
    % prior
    logL = logL - 0.5*nansum((xi-options.llh.prior.mean).^2 ...
        ./options.llh.prior.sigma2);
    if nargout >=2
        dlogL = nansum([dlogL,-(xi-options.llh.prior.mean)...
            ./options.llh.prior.sigma2],2);
    end
 
    % L1 regularization 
    if ~isempty(options.llh.ind_regularization)
        logL = logL - sum(options.llh.lambda_regularization* ...
            abs(xi(options.llh.ind_regularization)));
        if nargout >=2
            dlogL([options.llh.ind_regularization]) = dlogL([options.llh.ind_regularization]) ...
                - options.llh.lambda_regularization*sign(xi(options.llh.ind_regularization));
        end
    end
    
    %% Output assignment
    varargout{1} = logL;
    if nargout >= 2
        varargout{2} = [dlogL];
        if sum(isnan(dlogL) > 0)
            varargout{1} = NaN;
            varargout{2} = nan(n_xi,1);
        end
        if sum(isinf(dlogL) > 0)
            varargout{1} = NaN;
            varargout{2} = nan(n_xi,1);
        end
    end
    
catch e
    disp(e.message)
    varargout{1} = NaN;
    if nderiv>=1
        varargout{2} = nan(n_xi,1);
    end
end


