function parameters = fitting_histones(varargin)
% Fitting routine for the histone models. Data is loaded, parameter and
% options are defined and then PESTO is used to maximize the likelihood
% function.

% ToDo:
% - remain stuff from proportions/regions to domains

opts_fitting.modelname = 'globalDemeth';
opts_fitting.experiment = 'CAL311_319';
opts_fitting.condition = 'WT';
opts_fitting.culturemedium = 'all';
opts_fitting.par0 = [];
% parameters fixed to 0 (in linear scale), used e.g., for removing the
% demethylation in the standard model:
opts_fitting.indRemove = [];
opts_fitting.approach = 'hierarchical';
opts_fitting.offset = 1e-1;
opts_fitting.nStarts = 100;
opts_fitting.distribution = 'laplace';
opts_fitting.resultsFile = [];
opts_fitting.domains = [];

if nargin >= 1
    opts_fitting = setdefault(varargin{1},opts_fitting);
end

%% Import data
if strcmp(opts_fitting.experiment,'CAL311312313_3141315316')
    load(['./data/CAL311312313_' opts_fitting.condition])
    D1 = D;
    clear D
    load(['./data/CAL314315316_' opts_fitting.condition])
    D2 = D;
    clear D
    D.observable_names = D2.observable_names;
    D.replicates = {'CAL311'  'CAL312'  'CAL313' 'CAL314'  'CAL315'  'CAL316'};
    D.modifications = D2.modifications;
    D.t = [-3 0 4 8 16 24];
    D.my = nan(6,45,6);
    D.my(2:6,:,1:3) = D1.my(1:5,:,:);
    D.my(1:5,:,4:6) = D2.my(1:5,:,:);
elseif strcmp(opts_fitting.experiment,'CAL311_319')
    load(['./data/CAL311312313_' opts_fitting.condition])
    D1 = D;
    clear D
    load(['./data/CAL314315316_' opts_fitting.condition])
    D2 = D;
    clear D
    load(['./data/CAL317318319_' opts_fitting.condition])
    D3 = D;
    clear D
    D.observable_names = D2.observable_names;
    D.replicates = {'CAL311'  'CAL312'  'CAL313' 'CAL314'  'CAL315'  'CAL316' ...
        'CAL317'  'CAL318'  'CAL319' };
    D.modifications = D2.modifications;
    D.t = [-3 0 4 8 16 24];
    D.my = nan(6,45,9);
    D.my(2:6,:,1:3) = D1.my(1:5,:,:);
    D.my(1:5,:,4:6) = D2.my(1:5,:,:);
    D.my(1:6,:,7:9) = D3.my(1:6,:,:);
    
    % for generation 2 only up to 16h
    D.my(6,16:30,:) = nan;
else
    load(['./data/' opts_fitting.experiment '_' opts_fitting.condition])
end

%% Define simulation function
indices = 1:45; % all generations are used
eval(['sim_light = @(t,xi,k,options)' ...
    ' simulate_histones_' opts_fitting.modelname ...
    '_light_preequ(t,xi,k,[],options);']);
eval(['sim_all = @(t,xi,k,options)' ...
    ' simulate_histones_' opts_fitting.modelname ...
    '_all(t,xi,k,[],options);']);
simulateA = @(t,xi,k,options) ...
    simulate_histones_all_withpreequ(t,xi,k,options,sim_light,sim_all);

DA = D;
options.llh.input = opts_fitting.offset;
DA.t = DA.t(DA.t<=24)+3;
DA.my = DA.my(1:length(DA.t),indices,:) + opts_fitting.offset;
DA.my = log(DA.my);
DA.my(isinf(DA.my))=NaN;
DA.obs_names = DA.modifications;
DA.name = [opts_fitting.experiment ' ' opts_fitting.condition ' ' ...
    opts_fitting.culturemedium];
n_data = sum(sum(sum(~isnan(DA.my))));

%% Define options and parameters
options.ami = amioption('sensi',0,'maxsteps',1e4);
options.MS = PestoOptions();
options.MS.localOptimizer = 'fmincon';
options.MS.localOptimizerOptions = optimset('GradObj','on',...
    'display','iter','TolFun',1e-10, 'TolX',1e-10, ...
    'MaxIter', 3000,'algorithm','interior-point');
options.MS.comp_type = 'sequential';
options.MS.mode = 'text';

switch opts_fitting.modelname
    case 'global'
        parameters.name = {'d','d27','d36',...
            'r00_01','r00_10','r01_02','r01_11','r02_03','r02_12',...
            'r03_13','r10_11','r10_20','r11_12','r11_21','r12_13','r12_22',...
            'r13_23','r20_21','r20_30',...
            'r21_22','r21_31','r22_23','r22_32','r30_31','r31_32'};
        parameters.number = length(parameters.name);
        parameters.max = [2;2*ones(2,1);... %kinetic
            2*ones(22,1) % scaling
            ]; % weight
        parameters.min = [-5;-5*ones(2,1);
            -10*ones(22,1);...
            ];
    case 'globalDemeth'
        parameters.name = {'d','d27_1','d27_2','d27_3','d36_1','d36_2','d36_3',...
            'r00_01','r00_10','r01_02','r01_11','r02_03','r02_12',...
            'r03_13','r10_11','r10_20','r11_12','r11_21','r12_13','r12_22',...
            'r13_23','r20_21','r20_30',...
            'r21_22','r21_31','r22_23','r22_32','r30_31','r31_32'};
        parameters.number = length(parameters.name);
        parameters.max = [5;5*ones(6,1);... %kinetic
            5*ones(22,1) % scaling
            ]; % weight
        parameters.min = [-5;-5*ones(6,1);
            -10*ones(22,1);...
            ];
    case {'15regions','15domains'}
        parameters.name = {'r00_01','r00_10','r01_02','r01_11','r02_03','r02_12',...
            'r03_13','r10_11','r10_20','r11_12','r11_21',...
            'r12_13','r12_22','r13_23','r20_21','r20_30','r21_22','r21_31',...
            'r22_23','r22_32','r30_31','r31_32','c',...
            'w00','w01','w02','w03','w10','w11','w12','w13','w20','w21','w22',...
            'w23','w30','w31','w32',};
        parameters.number = length(parameters.name);
        parameters.min = -10*ones(parameters.number,1);
        parameters.max = 7*ones(parameters.number,1);
        parameters.min(23) = -3;
        parameters.max(23) = 1;
        parameters.min(24:38) = -8;
        parameters.max(24:38) = 4;
    otherwise % reduced version of the domai model
        parameters.name = {'r00_01','r00_10','r01_02','r01_11','r02_03','r02_12',...
            'r03_13','r10_11','r10_20','r11_12','r11_21',...
            'r12_13','r12_22','r13_23','r20_21','r20_30','r21_22','r21_31',...
            'r22_23','r22_32','r30_31','r31_32','c'};
        count=1;
        for i = (length(parameters.name)+1):(length(parameters.name)+length(opts_fitting.domains))
            parameters.name{i} = ['w' opts_fitting.domains{count}];
            count = count+1;
        end
        parameters.number = length(parameters.name);
        parameters.min = -10*ones(parameters.number,1);
        parameters.max = 3*ones(parameters.number,1);
        parameters.min(23) = -3;
        parameters.min([14,19]) = -12;
        parameters.max(23) = 1;
        parameters.min(24:end) = -8;
        parameters.max(24:end) = 4;
end

if isempty(opts_fitting.resultsFile)
    str_foldername = '';
    for i = 1:length(opts_fitting.indRemove)
        str_foldername = strcat(str_foldername,['_' parameters.name{opts_fitting.indRemove(i)}]);
    end
    
    options.MS.foldername = ['./results/results_' opts_fitting.modelname '_' ...
        opts_fitting.experiment '_' opts_fitting.culturemedium '_' ...
        opts_fitting.condition '' str_foldername '_offset1em1'];
else
    options.MS.foldername = opts_fitting.resultsFile;
end

switch opts_fitting.approach
    case 'hierarchical'
        % Options for hierarchical optimization
        options.MS.HO.n_exp = 1;
        options.MS.HO.max_repl = size(DA.my,3);
        n_obs = size(DA.my,2);
        options.llh.n_states = n_obs;
        options.MS.HO.n_obs = n_obs;
        for i = 1:n_obs
            options.MS.HO.scaling{i} = 'absolute';
            options.MS.HO.noise{i} = 'single';
            options.MS.HO.scale{i} = 'lin';
            options.MS.HO.obsgroups_scaling{i} = i;
        end
        options.MS.HO.obsgroups_noise = {[1:n_obs]};
    case 'standard'
        parameters.name{end+1} = 'sigma';
        parameters.min(end+1) = -5;
        parameters.max(end+1) = 5;
end

options.llh.n_theta = parameters.number;
options.llh.prior.mean = nan(parameters.number,1);
options.llh.prior.sigma2 = nan(parameters.number,1);
options.MS.HO.distribution = opts_fitting.distribution;
options.MS.save = true;
options.MS.n_starts = opts_fitting.nStarts;

%% Removed/Fixed Parameters (in linear scale)
options.MS.fixedParameters= opts_fitting.indRemove' ;
options.MS.fixedParameterValues = zeros(numel(opts_fitting.indRemove),1);
options.ami.pscale = 2*ones(1,parameters.number);
options.ami.pscale(opts_fitting.indRemove) = 0;

switch opts_fitting.approach
    case 'standard'
        logL = @(xi,DA,simulateA,options) ...
            logLikelihood_histones_standard(xi,DA,simulateA,options);
    case 'hierarchical'
        logL = @(xi,DA,simulateA,options) ...
            logLikelihood_histones_hierarchical(xi,DA,simulateA,options);
end

%% Multi-start optimization
warning off
if ~isempty(opts_fitting.par0)
    parameters.guess = par0(1:end-1,:);
else
    parameters.guess = [getParameterGuesses(parameters,@(xi) ...
        logL(xi,DA,simulateA,options),...
        options.MS.n_starts, parameters.min, parameters.max,options.MS)];
end
parameters = getMultiStarts(parameters,@(xi) logL(xi,DA,simulateA,options),...
    options.MS);

switch opts_fitting.approach
    case 'hierarchical'
        parameters.MS.BIC = -2*parameters.MS.logPost +...
            log(n_data)*(parameters.number-length(options.MS.fixedParameters)+1);
    case 'standard'
        parameters.MS.BIC = -2*parameters.MS.logPost +...
            log(n_data)*(parameters.number-length(options.MS.fixedParameters));
end
save(options.MS.foldername,'opts_fitting','options','parameters',...
            'sim_light','sim_all','simulateA','DA')
