! Copyright (c) 2013,  Los Alamos National Security, LLC (LANS)
! and the University Corporation for Atmospheric Research (UCAR).
!
! Unless noted otherwise source code is licensed under the BSD license.
! Additional copyright and license information can be found in the LICENSE file
! distributed with this code, or at http://mpas-dev.github.com/license.html
!
!=================================================================================================================
 module mpas_atmphys_vars
 use mpas_kind_types
 
 implicit none
 public
 save


!mpas_atmphys_vars contains all local variables and arrays used in the physics parameterizations.
!Laura D. Fowler (send comments to laura@ucar.edu).
!2013-05-01.
!
! add-ons and modifications:
! --------------------------
! * added the variables sf_surface_physics,alswvisdir_p,alswvisdif_p,alswnirdir_p,alswnirdif_p,swvisdir_p,
!   swvisdif_p,swnirdir_p,and swnirdif_p to upgrade the RRTMG short wave radiation code to WRF version 3.4.1. 
!   see definition of each individual variables below.
!   Laura D. Fowler (laura@ucar.edu) / 2013-03-11.
! * removed call to the updated Kain-Fritsch convection scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2013-05-29.
! * added the arrays o3clim_p for implementation of monthly-varying climatological ozone in the
!   long wave and short wave RRTMG radiation codes.
!   Laura D. Fowler (laura@ucar.edu) / 2013-07-08.
! * corrected definition of local variable dx_p.
!   Laura D. Fowler (laura@ucar.edu) / 2013-08-23.
! * renamed local variable conv_deep_scheme to convection_scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2014-09-18.
! * added empty subroutine atmphys_vars_init that does not do anything, but needed for 
!   compiling MPAS with some compilers.
!   Laura D. Fowler (laura@ucar.edu) / 2015-01-12.
! * added local variables needed for the Thompson parameterization of cloud microphysics.
!   Laura D. Fowler (laura@ucar.edu) / 2016-03-28.
! * added local variables needed for the Grell-Freitas parameterization of deep and shallow convection.
! * Laura D. Fowler (laura@ucar.edu) / 2016-03-30.
! * added local arrays needed in the MYNN surface layer scheme and PBL scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2016-03-31.
! * added the logical ua_phys needed in the call to subroutine sfcdiags. ua_phys is set to false.
!   Laura D. Fowler (laura@ucar.edu) / 2016-05-13.
! * added the integers has_reqc,has_reqi,and has_reqs. when initialized to zero, the effective radii for cloud
!   water,cloud ice,and snow are calculated using the subroutines relcalc and ricalc in subroutines rrtmg_lwrad
!   and rrtmg_swrad. when initialized to 1, the effective radii are calculated in the Thompson cloud microphysics
!   scheme instead. has_reqc,has_reqi,and has_reqs are initialized depending on the logical config_microp_re.
!   Laura D. Fowler (laura@ucar.edu) / 2016-07-08.
! * added diagnostics of the effective radii for cloud water, cloud ice, and snow used in rrtmg_lwrad.
!   Laura D. Fowler (laura@ucar.edu) / 2016-07-08.
! * added the variables qvrad_p,qcrad_p,qirad_p, and qsrad_p which are the water vapor,cloud water,cloud ice,
!   and snow mixing ratios local to the calculation of the cloud fraction, and used in the radiation codes.
!   Laura D. Fowler (laura@ucar.edu) / 2016-07-08.
! * added the variables rqvften and rthften which are the forcing tendencies needed to run the "new" Tiedtke
!   parameterization of convection.
!   Laura D. Fowler (laura@ucar.edu) / 2016-09-20.
! * added local "_sea" arrays that are needed in the surface layer scheme and land surface scheme for handling
!   grid cells with fractional seaice when config_frac_seaice is set to true. also added local tsk_ice variable
!   needed in the land surface scheme for handling grid cells with fractional seaice when config_frac_seaice is
!   set to true.
!   Laura D. Fowler (laura@ucar.edu) / 2016-10-03.
! * added local variable regime_hold to save the original value of variable regime over seaice grid cells when
!   config_frac_seaice is set to true.
!   Laura D. Fowler (laura@ucar.edu) / 2016-10-21.
! * moved the declarations of arrays delta_p,wstar_p,uoce_p,and voce_p since they are now used in both modules
!   module_bl_ysu.F and module_bl_mynn.F.
!   Laura D. Fowler (laura@ucar.edu) / 2016-10-27.
! * added the variable opt_thcnd (option to treat thermal conductivity in NoahLSM). added additional options and
!   arrays to run the Noah LSM scheme from WRF version 3.9.0.
!   Laura D. Fowler (laura@ucar.edu) / 2017-01-27.
! * removed the initialization local variable gwdo_scheme. gwdo_scheme is no longer needed and can be replaced
!   with config_gwdo_scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2017-02-16.
! * removed the initialization local variable lsm_scheme. lsm_scheme is no longer needed and can be replaced
!   with config_lsm_scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2017-02-16.
! * removed the initialization local variable sfclayer_scheme. sfclayer_scheme is no longer needed and can be
!   replaced with config_sfclayer_scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2017-02-16.
! * removed the initialization local variable pbl_scheme. pbl_scheme is no longer needed and can be replaced
!   replaced with config_pbl_scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2017-02-16.
! * removed the initialization local variable radt_cld_scheme. radt_cld_scheme is no longer needed and can be
!   replaced replaced with config_radt_cld_scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2017-02-16.
! * removed the initialization local variable radt_lw_scheme. radt_lw_scheme is no longer needed and can be
!   replaced replaced with config_radt_lw_scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2017-02-16.
! * removed the initialization local variable radt_sw_scheme. radt_sw_scheme is no longer needed and can be
!   replaced replaced with config_radt_sw_scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2017-02-16.
! * removed the initialization local variable convection_scheme. convection_scheme is no longer needed and can be
!   replaced replaced with config_convection_scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2017-02-16.
! * removed the initialization local variable microp_scheme. microp_scheme is no longer needed and can be
!   replaced replaced with config_microp_scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2017-02-16.
! * add variables and arrays needed to the parameterization of seaice in the updated Noah land surface scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2017-02-19.
! * changed the option seaice_albedo_opt from 0 to 2 so that we can initialize the surface albedo over seaice
!   cells using the surface background albedo (see initialization of sfc_albedo_seaice in subroutine
!   landuse_init_forMPAS).
!   Laura D. Fowler (laura@ucar.edu) / 2017-03-02.
! * added local variables for the mass-weighted mean velocities for rain, cloud ice, snow, and graupel from the
!   Thompson cloud microphysics scheme.
!   Laura D. Fowler (laura@ucar.edu) / 2017-04-19.
! * added the local variables cosa_p and sina_p needed in call to subroutine gwdo after updating module_bl_gwdo.F
!   to that of WRF version 4.0.2
!   Laura D. Fowler (laura@ucar.edu) / 2019-01-30.


!=================================================================================================================
!wrf-variables:these variables are needed to keep calls to different physics parameterizations
!as in wrf model.
!=================================================================================================================

 logical:: l_radtlw                   !controls call to longwave radiation parameterization.
 logical:: l_radtsw                   !controls call to shortwave radiation parameterization.
 logical:: l_conv                     !controls call to convective parameterization.
 logical:: l_camlw                    !controls when to save local CAM LW abs and ems arrays.
 logical:: l_diags                    !controls when to calculate physics diagnostics.
 logical:: l_acrain                   !when .true., limit to accumulated rain is applied.
 logical:: l_acradt                   !when .true., limit to lw and sw radiation is applied.
 logical:: l_mp_tables                !when .true., read look-up tables for Thompson cloud microphysics scheme.

 integer,public:: ids,ide,jds,jde,kds,kde
 integer,public:: ims,ime,jms,jme,kms,kme
 integer,public:: its,ite,jts,jte,kts,kte
 integer,public:: iall
 integer,public:: n_microp

 integer,public:: num_months          !number of months                                         [-]

 real(kind=RKIND),public:: dt_dyn     !time-step for dynamics
 real(kind=RKIND),public:: dt_microp  !time-step for cloud microphysics parameterization.
 real(kind=RKIND),public:: dt_radtlw  !time-step for longwave radiation parameterization      [mns]
 real(kind=RKIND),public:: dt_radtsw  !time-step for shortwave radiation parameterization     [mns]
 
 real(kind=RKIND),public:: xice_threshold

 real(kind=RKIND),dimension(:,:),allocatable:: &
    area_p             !grid cell area                                                         [m2]

!... arrays related to surface:
 real(kind=RKIND),dimension(:,:),allocatable:: &
    ht_p,             &!
    psfc_p,           &!surface pressure                                                       [Pa]
    ptop_p             !model-top pressure                                                     [Pa]

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    fzm_p,            &!weight for interpolation to w points                                    [-]
    fzp_p              !weight for interpolation to w points                                    [-]

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
!... arrays related to u- and v-velocities interpolated to theta points:
    u_p,              &!u-velocity interpolated to theta points                               [m/s]
    v_p                !v-velocity interpolated to theta points                               [m/s]
    
!... arrays related to vertical sounding:
 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    zz_p,             &!
    pres_p,           &!pressure                                                               [Pa]
    pi_p,             &!(p_phy/p0)**(r_d/cp)                                                    [-]
    z_p,              &!height of layer                                                         [m]
    zmid_p,           &!height of middle of layer                                               [m]
    dz_p,             &!layer thickness                                                         [m]
    t_p,              &!temperature                                                             [K]
    th_p,             &!potential temperature                                                   [K]
    al_p,             &!inverse of air density                                              [m3/kg]
    rho_p,            &!air density                                                         [kg/m3]
    rh_p               !relative humidity                                                       [-]

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    qv_p,             &!water vapor mixing ratio                                            [kg/kg]
    qc_p,             &!cloud water mixing ratio                                            [kg/kg]
    qr_p,             &!rain mixing ratio                                                   [kg/kg]
    qi_p,             &!cloud ice mixing ratio                                              [kg/kg]
    qs_p,             &!snow mixing ratio                                                   [kg/kg]
    qg_p               !graupel mixing ratio                                                [kg/kg]

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    nc_p,             &!
    ni_p,             &!
    nr_p               !

!... arrays located at w (vertical velocity) points, or at interface between layers:
 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    w_p,              &!vertical velocity                                                     [m/s]
    pres2_p,          &!pressure                                                               [Pa]
    t2_p               !temperature                                                             [K]

!... arrays used for calculating the hydrostatic pressure and exner function:
 real(kind=RKIND),dimension(:,:),allocatable:: &
    psfc_hyd_p,       &!surface pressure                                                       [Pa]
    psfc_hydd_p        !"dry" surface pressure                                                 [Pa]
 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    pres_hyd_p,       &!pressure located at theta levels                                       [Pa]
    pres_hydd_p,      &!"dry" pressure located at theta levels                                 [Pa]
    pres2_hyd_p,      &!pressure located at w-velocity levels                                  [Pa]
    pres2_hydd_p,     &!"dry" pressure located at w-velocity levels                            [Pa]
    znu_hyd_p          !(pres_hyd_p / P0) needed in the Tiedtke convection scheme              [Pa]

!=================================================================================================================
!... variables related to ozone climatlogy:
!=================================================================================================================

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    o3clim_p           !climatological ozone volume mixing ratio                         [???]

!=================================================================================================================
!... variables and arrays related to parameterization of cloud microphysics:
!    warm_phase: logical that determines if we want to run warm-phase cloud microphysics only.
!          If set to false, cold-phase cloud microphysics is active. In MPAS, we always assume
!          that the ice phase is included (except for the Kessler scheme which includes water
!          clouds only.

!    f_qv,f_qc,f_qr,f_qi,f_qs,f_qg: These logicals were initially defined in WRF to determine
!          which kind of hydrometeors are present. Here, we assume that all six water species
!          are present, even if their mixing ratios and number concentrations are zero.

!=================================================================================================================

 logical,parameter:: &
    warm_rain=.false.  !warm-phase cloud microphysics only (used in WRF).

 logical,parameter:: &
    f_qv = .true.,    &!
    f_qc = .true.,    &!
    f_qr = .true.,    &!
    f_qi = .true.,    &!
    f_qs = .true.,    &!
    f_qg = .true.      !

 logical,parameter:: &
    f_qnc = .true.,   &!
    f_qni = .true.     !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    f_ice,            &!fraction of cloud ice (used in WRF only).
    f_rain             !fraction of rain (used in WRF only).

 real(kind=RKIND),dimension(:,:),allocatable:: &
    rainnc_p,         &!
    rainncv_p,        &!
    snownc_p,         &!
    snowncv_p,        &!
    graupelnc_p,      &!
    graupelncv_p,     &!
    sr_p

!... added for the thompson and wsm6 cloud microphysics:
 integer:: & 
    has_reqc,         &!
    has_reqi,         &!
    has_reqs

 real(kind=RKIND),dimension(:,:),allocatable:: &
    ntc_p,            &!
    muc_p              !
 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rainprod_p,       &!
    evapprod_p,       &!
    recloud_p,        &!
    reice_p,          &!
    resnow_p,         &!
    refl10cm_p         !

!=================================================================================================================
!... variables and arrays related to parameterization of convection:
!=================================================================================================================
 integer,public:: n_cu
 real(kind=RKIND),public:: dt_cu

 logical,dimension(:,:),allocatable:: &
    cu_act_flag
 real(kind=RKIND),dimension(:,:),allocatable::   &
    rainc_p,          &!
    raincv_p,         &!
    pratec_p           !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rthcuten_p,       &!
    rqvcuten_p,       &!
    rqccuten_p,       &!
    rqicuten_p         !

!... kain fritsch specific arrays:
 real(kind=RKIND),dimension(:,:),allocatable::   &
    cubot_p,          &!lowest convective level                                                 [-]
    cutop_p,          &!highest convective level                                                [-]
    nca_p              !counter for cloud relaxation time                                       [-]
 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    w0avg_p          !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rqrcuten_p,       &!
    rqscuten_p         !

!... tiedtke specific arrays:
 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    znu_p              !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rucuten_p,        &!
    rvcuten_p          !

!... grell-freitas specific parameters and arrays:
    integer, parameter:: ishallow  = 1 !shallow convection used with grell scheme.

 integer,dimension(:,:),allocatable:: &
    k22_shallow_p,    &!
    kbcon_shallow_p,  &!
    ktop_shallow_p,   &!
    kbot_shallow_p,   &!
    ktop_deep_p        !

 real(kind=RKIND),dimension(:,:),allocatable:: &
    xmb_total_p,      &!
    xmb_shallow_p      !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rthdynten_p,      &!
    qccu_p,           &!
    qicu_p             !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rthraten_p         !

!... grell and tiedkte specific arrays:
 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rqvdynten_p,      &!
    rqvdynblten_p,    &!
    rthdynblten_p      !

!... ntiedtke specific arrays:
 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rqvften_p,        &!
    rthften_p          !

!=================================================================================================================
!... variables and arrays related to parameterization of pbl:
!=================================================================================================================

 integer:: ysu_pblmix

 integer,dimension(:,:),allocatable:: &
    kpbl_p             !index of PBL top                                                                       [-]

 real(kind=RKIND),public:: dt_pbl

 real(kind=RKIND),dimension(:,:),allocatable:: &
    ctopo_p,          &!correction to topography                                                               [-]
    ctopo2_p,         &!correction to topography 2                                                             [-]
    hpbl_p,           &!PBL height                                                                             [m]
    delta_p,          &!
    wstar_p,          &!
    uoce_p,           &!
    voce_p             !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    exch_p             !exchange coefficient                                                                   [-]

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rublten_p,        &!
    rvblten_p,        &!
    rthblten_p,       &!
    rqvblten_p,       &!
    rqcblten_p,       &!
    rqiblten_p         !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    kzh_p,            &!
    kzm_p,            &!
    kzq_p              !

!... MYNN PBL scheme (module_bl_mynn.F):
 integer,parameter:: grav_settling = 0

 logical,parameter:: bl_mynn_tkeadvect = .false.!
 integer,parameter:: bl_mynn_tkebudget = 0      !
 integer,parameter:: bl_mynn_cloudpdf  = 0      !

 real(kind=RKIND),dimension(:,:),allocatable:: &
    vdfg_p             !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    dqke_p,           &!
    qbuoy_p,          &!
    qdiss_p,          &!
    qke_p,            &!
    qkeadv_p,         &!
    qshear_p,         &!
    qwt_p,            &!
    tkepbl_p           !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rniblten_p         !

!=================================================================================================================
!... variables and arrays related to parameterization of gravity wave drag over orography:
!=================================================================================================================

 real(kind=RKIND),dimension(:,:),allocatable:: &
    cosa_p,           &!cosine of map rotation                                                                 [-]
    sina_p             !sine of map rotation                                                                   [-]

 real(kind=RKIND),dimension(:,:),allocatable:: &
    var2d_p,          &!orographic variance                                                                   [m2]
    con_p,            &!orographic convexity                                                                  [m2]
    oa1_p,            &!orographic direction asymmetry function                                                [-]
    oa2_p,            &!orographic direction asymmetry function                                                [-]
    oa3_p,            &!orographic direction asymmetry function                                                [-]
    oa4_p,            &!orographic direction asymmetry function                                                [-]
    ol1_p,            &!orographic direction asymmetry function                                                [-]
    ol2_p,            &!orographic direction asymmetry function                                                [-]
    ol3_p,            &!orographic direction asymmetry function                                                [-]
    ol4_p              !orographic direction asymmetry function                                                [-]

 real(kind=RKIND),dimension(:,:),allocatable:: &
    dx_p               !mean distance between cell centers                                                     [m]

 real(kind=RKIND),dimension(:,:),allocatable:: &
    dusfcg_p,         &!vertically-integrated gwdo u-stress                                             [Pa m s-1]
    dvsfcg_p           !vertically-integrated gwdo v -stress                                            [Pa m s-1]

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    dtaux3d_p,        &!gravity wave drag over orography u-stress                                          [m s-1]
    dtauy3d_p          !gravity wave drag over orography u-stress                                          [m s-1]

!=================================================================================================================
!... variables and arrays related to parameterization of surface layer:
!=================================================================================================================

 real(kind=RKIND),dimension(:,:),allocatable:: &
    br_p,             &!bulk richardson number                                                                 [-]
    cd_p,             &!momentum exchange coeff at 10 meters                                                   [?]
    cda_p,            &!momentum exchange coeff at the lowest model level                                      [?]
    cpm_p,            &!
    chs_p,            &!
    chs2_p,           &!
    ck_p,             &!enthalpy exchange coeff at 10 meters                                                   [?]
    cka_p,            &!enthalpy exchange coeff at the lowest model level                                      [?]
    cqs2_p,           &!
    gz1oz0_p,         &!log of z1 over z0                                                                      [-]
    flhc_p,           &!exchange coefficient for heat                                                          [-]
    flqc_p,           &!exchange coefficient for moisture                                                      [-]
    hfx_p,            &!upward heat flux at the surface                                                     [W/m2]
    lh_p,             &!latent heat flux at the surface                                                     [W/m2]
    mavail_p,         &!surface moisture availability                                                          [-]
    mol_p,            &!T* in similarity theory                                                                [K]
    pblh_p,           &!PBL height                                                                             [m]
    psih_p,           &!similarity theory for heat                                                             [-]
    psim_p,           &!similarity theory for momentum                                                         [-]
    q2_p,             &!specific humidity at 2m                                                            [kg/kg]
    qfx_p,            &!upward moisture flux at the surface                                              [kg/m2/s]
    qgh_p,            &!
    qsfc_p,           &!specific humidity at lower boundary                                                [kg/kg]
    regime_p,         &!flag indicating PBL regime (stable_p,unstable_p,etc...)                                [-]
    rmol_p,           &!1 / Monin Ob length                                                                    [-]
    t2m_p,            &!temperature at 2m                                                                      [K]
    th2m_p,           &!potential temperature at 2m                                                            [K]
    u10_p,            &!u at 10 m                                                                            [m/s]
    ust_p,            &!u* in similarity theory                                                              [m/s]
    ustm_p,           &!u* in similarity theory       without vconv correction                               [m/s]
    v10_p,            &!v at 10 m                                                                            [m/s]
    wspd_p,           &!wind speed                                                                           [m/s]
    znt_p,            &!time-varying roughness length                                                          [m]
    zol_p              !

!... arrays only in monin_obukohv (module_sf_sfclay.F):
 real(kind=RKIND),dimension(:,:),allocatable:: &
    fh_p,             &!integrated stability function for heat                                                 [-]
    fm_p               !integrated stability function for momentum                                             [-]

!... arrays only in mynn surface layer scheme (module_sf_mynn.F):
 real(kind=RKIND),dimension(:,:),allocatable:: &
    ch_p,             &!surface exchange coeff for heat                                                      [m/s]
    qcg_p              !cloud water mixing ratio at the ground surface                                     [kg/kg]

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    cov_p,            &!liquid water-liquid water potential temperature covariance                       [K kg/kg]
    qsq_p,            &!liquid water variance                                                          [(kg/kg)^2]
    tsq_p,            &!liquid water potential temperature variance                                          [K^2]
    sh3d_p,           &!stability function for heat                                                            [-]
    elpbl_p            !length scale from PBL                                                                  [m]

!=================================================================================================================
!... variables and arrays related to parameterization of seaice:
!=================================================================================================================

 integer,parameter:: &
    seaice_albedo_opt    = 2           !option to set albedo over sea ice.
                                       !0 = seaice albedo is constant set in seaice_albedo_default.
                                       !1 = seaice albedo is f(Tair,Tskin,Tsnow), following Mill (2011).
                                       !2 = seaice albedo is read in from input variable albsi.
 integer,parameter:: &
    seaice_thickness_opt = 0           !option for treating seaice thickness.
                                       !0 = seaice thickness is constant set in seaice_thickness_default.
                                       !1 = seaice_thickness is read in from input variable icedepth.
 integer,parameter:: &
    seaice_snowdepth_opt = 0           !option for treating snow depth on sea ice.
                                       !0=snow depth is bounded by seaice_snowdepth_min and seaice_snowdepth_max.

 real(kind=RKIND),parameter:: &
    seaice_albedo_default     = 0.65 ,&!default value of seaice albedo for seaice_albedo_opt=0.
    seaice_thickness_default  = 3.0,  &!default value of seaice thickness for seaice_thickness_opt=0
    seaice_snowdepth_max      = 1.e10,&!maximum allowed accumulation of snow (m) on sea ice.
    seaice_snowdepth_min      = 0.001  !minimum snow depth (m) on sea ice.

 real(kind=RKIND),dimension(:,:),allocatable:: &
    albsi_p,          &!surface albedo over seaice                                                             [-]
    snowsi_p,         &!snow depth over seaice                                                                 [m]
    icedepth_p         !seaice thickness                                                                       [m]

!=================================================================================================================
!... variables and arrays related to parameterization of short-wave radiation:
!=================================================================================================================

 real(kind=RKIND):: &
    declin,           &!solar declination                                                       [-]
    solcon             !solar constant                                                      [W m-2]

 real(kind=RKIND),dimension(:,:),allocatable:: &
    coszr_p,          &!cosine of the solar zenith angle                                        [-]
    gsw_p,            &!net shortwave flux at surface                                       [W m-2]
    swcf_p,           &!shortwave cloud forcing at top-of-atmosphere                        [W m-2]
    swdnb_p,          &!all-sky downwelling shortwave flux at bottom-of-atmosphere          [J m-2]
    swdnbc_p,         &!clear-sky downwelling shortwave flux at bottom-of-atmosphere        [J m-2]
    swdnt_p,          &!all-sky downwelling shortwave flux at top-of-atmosphere             [J m-2]
    swdntc_p,         &!clear-sky downwelling shortwave flux at top-of-atmosphere           [J m-2]
    swupb_p,          &!all-sky upwelling shortwave flux at bottom-of-atmosphere            [J m-2]
    swupbc_p,         &!clear-sky upwelling shortwave flux at bottom-of-atmosphere          [J m-2]
    swupt_p,          &!all-sky upwelling shortwave flux at top-of-atmosphere               [J m-2]
    swuptc_p           !clear-sky upwelling shortwave flux at top-of-atmosphere             [J m-2]

 real(kind=RKIND),dimension(:,:),allocatable:: &
    swvisdir_p,       &!visible direct downward flux                                        [W m-2]
    swvisdif_p,       &!visible diffuse downward flux                                       [W m-2]
    swnirdir_p,       &!near-IR direct downward flux                                        [W m-2]
    swnirdif_p         !near-IR diffuse downward flux                                       [W m-2]

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    swdnflx_p,        &!
    swdnflxc_p,       &!
    swupflx_p,        &!
    swupflxc_p         !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rthratensw_p       !uncoupled theta tendency due to shortwave radiation                 [K s-1]

!=================================================================================================================
!... variables and arrays related to parameterization of long-wave radiation:
!=================================================================================================================

 integer,dimension(:,:),allocatable:: &
    nlrad_p            !number of layers added above the model top                              [-]
 real(kind=RKIND),dimension(:,:),allocatable:: &
    plrad_p            !pressure at model_top                                                  [Pa]

 real(kind=RKIND),dimension(:,:),allocatable:: &
    glw_p,            &!net longwave flux at surface                                        [W m-2]
    lwcf_p,           &!longwave cloud forcing at top-of-atmosphere                         [W m-2]
    lwdnb_p,          &!all-sky downwelling longwave flux at bottom-of-atmosphere           [J m-2]
    lwdnbc_p,         &!clear-sky downwelling longwave flux at bottom-of-atmosphere         [J m-2]
    lwdnt_p,          &!all-sky downwelling longwave flux at top-of-atmosphere              [J m-2]
    lwdntc_p,         &!clear-sky downwelling longwave flux at top-of-atmosphere            [J m-2]
    lwupb_p,          &!all-sky upwelling longwave flux at bottom-of-atmosphere             [J m-2]
    lwupbc_p,         &!clear-sky upwelling longwave flux at bottom-of-atmosphere           [J m-2]
    lwupt_p,          &!all-sky upwelling longwave flux at top-of-atmosphere                [J m-2]
    lwuptc_p,         &!clear-sky upwelling longwave flux at top-of-atmosphere              [J m-2]
    olrtoa_p           !outgoing longwave radiation at top-of-the-atmosphere                [W m-2]

  real(kind=RKIND),dimension(:,:,:),allocatable:: &
    lwdnflx_p,        &!
    lwdnflxc_p,       &!
    lwupflx_p,        &!
    lwupflxc_p         !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    rthratenlw_p,     &!uncoupled theta tendency due to longwave radiation                  [K s-1]
    rrecloud_p,       &!effective radius for cloud water calculated in rrtmg_lwrad             [mu]
    rreice_p,         &!effective radius for cloud ice calculated in rrmtg_lwrad               [mu]
    rresnow_p         !effective radius for snow calculated in rrtmg_lwrad                    [mu]

!=================================================================================================================
!... variables and arrays related to parameterization of long- and short-wave radiation needed
!    only by the "CAM" radiation codes:
!=================================================================================================================

 logical:: doabsems

 integer:: cam_abs_dim1
 integer:: cam_abs_dim2
 integer:: num_moist
 integer:: num_aerosols
 integer:: num_aerlevels
 integer:: num_oznlevels

 real(kind=RKIND),dimension(:),allocatable:: &
    pin_p,            &!pressure levels for ozone concentration                                [Pa]
    m_hybi_p           !hybrid levels for aerosols                                             [-]

 real(kind=RKIND),dimension(:,:),allocatable:: &
    m_psn_p,          &!
    m_psp_p            !

 real(kind=RKIND),dimension(:,:,:,:),allocatable:: &
    aerosolcn_p,      &!
    aerosolcp_p        !

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    emstot_p,         &!total emissivity                                                        [-]
    cemiss_p,         &!cloud emissivity for ISCCP                                              [-]
    taucldc_p,        &!cloud water optical depth for ISCCP                                     [-]
    taucldi_p          !cloud ice optical depth for ISCCP                                       [-]    

 real(kind=RKIND),dimension(:,:,:,:),allocatable:: &
    abstot_p,         &!total layer absorptivity                                                [-]
    absnxt_p,         &!total nearest layer absorptivity                                        [-]
    ozmixm_p           !ozone mixing ratio.

!=================================================================================================================
!.. variables and arrays related to cloudiness:
!=================================================================================================================

 integer,parameter:: &
    icloud= 1          !used in WRF only.

 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    cldfrac_p,        &!cloud fraction                                                                         [-]
    qvrad_p,          &!water vapor mixing ratio local to cloudiness and radiation                         [kg/kg]
    qcrad_p,          &!cloud liquid water mixing ratio local to cloudiness and radiation                  [kg/kg]
    qirad_p,          &!cloud ice mixing ratio local to cloudiness and radiation                           [kg/kg]
    qsrad_p            !snow mixing ratio local to cloudiness and radiation                                [kg/kg]

!=================================================================================================================
!.. variables and arrays related to land-surface parameterization:
!=================================================================================================================

 logical,parameter:: &
    ua_phys=.false.    !option to activate UA Noah changes: a different snow-cover physics in the land-surface
                       !scheme. That option is not currently implemented in MPAS.

 integer,parameter:: &
    opt_thcnd = 1      !option to treat thermal conductivity in NoahLSM (new option implemented in WRF 3.8.0).
                       != 1, original (default).
                       != 2, McCumber and Pielke for silt loam and sandy loam.

 integer,parameter:: &
    fasdas = 0         !for WRF surface data assimilation system (not used in MPAS).

 integer,public:: &
    sf_surface_physics !used to define the land surface scheme by a number instead of name. It
                       !is only needed in module_ra_rrtmg_sw.F to define the spectral surface
                       !albedos as functions of the land surface scheme.

 integer,public:: &
    num_soils          !number of soil layers                                                                  [-]
    
 integer,dimension(:,:),allocatable:: &
    isltyp_p,         &!dominant soil type category                                                            [-]
    ivgtyp_p           !dominant vegetation category                                                           [-]

 real(kind=RKIND),dimension(:),allocatable:: &
    dzs_p              !thickness of soil layers                                                               [m]
 real(kind=RKIND),dimension(:,:,:),allocatable:: &
    smcrel_p,         &!soil moisture threshold below which transpiration starts to stress                     [-]
    sh2o_p,           &!unfrozen soil moisture content                                       [volumetric fraction]
    smois_p,          &!soil moisture                                                        [volumetric fraction]
    tslb_p             !soil temperature                                                                       [K]

 real(kind=RKIND),dimension(:,:),allocatable:: &
    acsnom_p,         &!accumulated melted snow                                                           [kg m-2]
    acsnow_p,         &!accumulated snow                                                                  [kg m-2]
    canwat_p,         &!canopy water                                                                      [kg m-2]
    chklowq_p,        &!surface saturation flag                                                                [-]
    grdflx_p,         &!ground heat flux                                                                   [W m-2]
    lai_p,            &!leaf area index                                                                        [-]
    noahres_p,        &!residual of the noah land-surface scheme energy budget                             [W m-2]
    potevp_p,         &!potential evaporation                                                              [W m-2]
    qz0_p,            &!specific humidity at znt                                                         [kg kg-1]
    rainbl_p,         &!
    sfcrunoff_p,      &!surface runoff                                                                     [m s-1]
    shdmin_p,         &!minimum areal fractional coverage of annual green vegetation                           [-]
    shdmax_p,         &!maximum areal fractional coverage of annual green vegetation                           [-]
    smstav_p,         &!moisture availability                                                                  [-]
    smstot_p,         &!total moisture                                                                    [m3 m-3]
    snopcx_p,         &!snow phase change heat flux                                                        [W m-2]
    snotime_p,        &!
    snowc_p,          &!snow water equivalent                                                             [kg m-2]
    snowh_p,          &!physical snow depth                                                                    [m]
    swdown_p,         &!downward shortwave flux at the surface                                             [W m-2]
    udrunoff_p,       &!sub-surface runoff                                                                 [m s-1]
    tmn_p,            &!soil temperature at lower boundary                                                     [K]
    vegfra_p,         &!vegetation fraction                                                                    [-]
    z0_p               !background roughness length                                                            [m]

 real(kind=RKIND),dimension(:,:),allocatable:: &
    alswvisdir_p,     &!direct-beam surface albedo in visible spectrum                                         [-]
    alswvisdif_p,     &!diffuse-beam surface albedo in visible spectrum                                        [-]
    alswnirdir_p,     &!direct-beam surface albedo in near-IR spectrum                                         [-]
    alswnirdif_p       !diffuse-beam surface albedo in near-IR spectrum                                        [-]

!.. arrays needed to run UA Noah changes (different snow-cover physics):
 real(kind=RKIND),dimension(:,:),allocatable:: &
    flxsnow_p,        &!energy added to sensible heat flux when ua_phys=true                               [W m-2]
    fvbsnow_p,        &!fraction of vegetation with snow beneath when ua_phys=true                             [-]
    fbursnow_p,       &!fraction of canopy buried when ua_phys=true                                            [-]
    fgsnsnow_p         !fraction of ground snow cover when ua_phys=true                                        [-]

!.. arrays needed in the argument list in the call to the Noah LSM urban parameterization: note that these arrays
!.. are initialized to zero since we do not run an urban model:
 integer,dimension(:,:),allocatable:: &
    utype_urb_p        !urban type                                                                             [-]

 real(kind=RKIND),dimension(:,:),allocatable:: &
    frc_urb_p,        &!urban fraction                                                                         [-]
    ust_urb_p          !urban u* in similarity theory                                                        [m/s]

!.. arrays needed in the argument list in the call to the Noah LSM hydrology model: note that these arrays are
!.. initialized to zero since we do not run a hydrology model:
 real(kind=RKIND),dimension(:,:),allocatable:: &
    infxsrt_p,        &!timestep infiltration excess                                                          [mm]
    sfcheadrt_p,      &!surface water detph                                                                   [mm]
    soldrain_p         !soil column drainage                                                                  [mm]

!=================================================================================================================
!.. variables and arrays related to surface characteristics:
!=================================================================================================================

 real(kind=RKIND),dimension(:,:),allocatable:: &
    xlat_p,           &!longitude, west is negative                                                      [degrees]
    xlon_p             !latitude, south is negative                                                      [degrees]

 real(kind=RKIND),dimension(:,:),allocatable:: &
    sfc_albedo_p,     &!surface albedo                                                                         [-]
    sfc_albbck_p,     &!surface background albedo                                                              [-]
    sfc_emibck_p,     &!land surface background emissivity                                                     [-]
    sfc_emiss_p,      &!land surface emissivity                                                                [-]
    snoalb_p,         &!annual max snow albedo                                                                 [-]
    snow_p,           &!snow water equivalent                                                             [kg m-2]
    tsk_p,            &!surface-skin temperature                                                               [K]
    sst_p,            &!sea-surface temperature                                                                [K]
    xice_p,           &!ice mask                                                                               [-]
    xland_p            !land mask    (1 for land; 2 for water)                                                 [-]

!=================================================================================================================
!.. variables needed for the surface layer scheme and land surface scheme when config_frac_seaice
!   is set to true. the arrays below have the same definition as the corresponding "_p" arrays:
!=================================================================================================================

 real(kind=RKIND),dimension(:,:),allocatable:: br_sea,ch_sea,chs_sea,chs2_sea,cpm_sea,cqs2_sea,      &
                                flhc_sea,flqc_sea,gz1oz0_sea,hfx_sea,lh_sea,mavail_sea,mol_sea,      &
                                psih_sea,psim_sea,fh_sea,fm_sea,qfx_sea,qgh_sea,qsfc_sea,regime_sea, &
                                rmol_sea,ust_sea,wspd_sea,znt_sea,zol_sea,tsk_sea,xland_sea
 real(kind=RKIND),dimension(:,:),allocatable:: t2m_sea,th2m_sea,q2_sea,u10_sea,v10_sea
 real(kind=RKIND),dimension(:,:),allocatable:: cd_sea,cda_sea,ck_sea,cka_sea,ustm_sea

 real(kind=RKIND),dimension(:,:),allocatable:: regime_hold
 real(kind=RKIND),dimension(:,:),allocatable:: tsk_ice


 contains


!=================================================================================================================
 subroutine atmphys_vars_init()
!=================================================================================================================
!dummy subroutine that does not do anything.

 end subroutine atmphys_vars_init

!=================================================================================================================
 end module mpas_atmphys_vars
!=================================================================================================================
