/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain.labpal;

import ca.uqac.lif.labpal.Laboratory;
import ca.uqac.lif.labpal.LatexNamer;
import ca.uqac.lif.labpal.Region;
import ca.uqac.lif.labpal.TitleNamer;
import ca.uqac.lif.labpal.server.WebCallback;
import ca.uqac.lif.labpal.table.ExperimentTable;
import ca.uqac.lif.mtnp.table.TransformedTable;
import java.util.List;
import ca.uqac.lif.mtnp.table.ExpandAsColumns;

import static ca.uqac.lif.cep.supplychain.labpal.ParcelExperiment.NUM_HOPS;
import static ca.uqac.lif.cep.supplychain.labpal.ParcelExperiment.NUM_PARCELS;
import static ca.uqac.lif.cep.supplychain.labpal.ParcelExperiment.PROPERTY;
import static ca.uqac.lif.cep.supplychain.labpal.ParcelExperiment.REROUTING_PROB;
import static ca.uqac.lif.cep.supplychain.labpal.ParcelExperiment.THROUGHPUT;

import ca.uqac.lif.cep.supplychain.AverageShippingTime;
import ca.uqac.lif.cep.supplychain.DecreasingDistance;
import ca.uqac.lif.cep.supplychain.MaxReroutings;
import ca.uqac.lif.cep.supplychain.NewReroutings;
import ca.uqac.lif.cep.supplychain.ParcelLifecycle;
import ca.uqac.lif.cep.supplychain.ParcelsInTransit;
import ca.uqac.lif.cep.supplychain.TimeDistribution;
import ca.uqac.lif.json.JsonFalse;

/**
 * The main lab used to benchmark the processor chains.
 */
public class SupplyChainLab extends Laboratory
{
  /**
   * The step (in number of events) at which measurements are made in each experiment
   */
  public static int s_eventStep = 10000;

  /**
   * The maximum trace length to generate
   */
  public static int MAX_TRACE_LENGTH = 500001;

  /**
   * A nicknamer
   */
  public static transient LatexNamer s_nicknamer = new LatexNamer();

  /**
   * A title namer
   */
  public static transient TitleNamer s_titleNamer = new TitleNamer();

  /**
   * An experiment factory
   */
  public transient ParcelExperimentFactory m_factory = new ParcelExperimentFactory(this);

  @Override
  public void setup()
  {
    setTitle("Benchmark for supply chain monitoring properties");
    setDoi("10.5281/zenodo.3066013");
    setAuthor("Sylvain Hallé");

    // Define the basic parameters of the lab
    TriangleRegion main_region = new TriangleRegion();
    main_region.add(StreamExperiment.MULTITHREAD, JsonFalse.instance);
    main_region.add(NUM_HOPS, 10, 50, 100);
    main_region.add(NUM_PARCELS, 10, 25, 50, 100, 1000, 10000);
    main_region.add(REROUTING_PROB, 0.2);
    main_region.add(PROPERTY, DecreasingDistance.NAME, ParcelLifecycle.NAME, 
        MaxReroutings.NAME, NewReroutings.NAME, ParcelsInTransit.NAME,
        TimeDistribution.NAME, AverageShippingTime.NAME);

    // Instantiate the experiments
    {
      // For each property, plot against number of hops
      for (Region r_np : main_region.all(NUM_PARCELS))
      {
        ExperimentTable et = new ExperimentTable(PROPERTY, NUM_HOPS, THROUGHPUT);
        et.setShowInList(false);
        s_titleNamer.setTitle(et, r_np, "Throughput by number of hops (", ")");
        s_nicknamer.setNickname(et, r_np, "tThroughput", "");
        for (Region r : r_np.all(NUM_HOPS, PROPERTY))
        {
          ParcelExperiment exp = m_factory.get(r);
          if (exp == null)
            continue;
          et.add(exp);
        }
        TransformedTable tt = new TransformedTable(new ExpandAsColumns(PROPERTY, THROUGHPUT), et);
        s_titleNamer.setTitle(tt, r_np, "Throughput by number of hops (", ")");
        s_nicknamer.setNickname(tt, r_np, "tThroughput", "");
        add(et, tt);
      }
    }
    {
      // For each property, plot against number of slices
      for (Region r_np : main_region.all(NUM_HOPS))
      {
        ExperimentTable et = new ExperimentTable(PROPERTY, NUM_PARCELS, THROUGHPUT);
        et.setShowInList(false);
        for (Region r : r_np.all(NUM_PARCELS, PROPERTY))
        {
          ParcelExperiment exp = m_factory.get(r);
          if (exp == null)
            continue;
          et.add(exp);
        }
        TransformedTable tt = new TransformedTable(new ExpandAsColumns(PROPERTY, THROUGHPUT), et);
        s_titleNamer.setTitle(tt, r_np, "Throughput by number of parcels (", ")");
        s_nicknamer.setNickname(tt, r_np, "tThroughput", "");
        add(et, tt);
      }
    }

    // Macros
    add(new LabStats(this));
    add(new DataStats(this, m_factory));
  }
  
  /**
   * Initializes the lab
   * @param args Command line arguments
   */
  public static void main(String[] args)
  {
    // Nothing more to be done here
    initialize(args, SupplyChainLab.class);
  }
  
  @Override
  public void setupCallbacks(List<WebCallback> callbacks)
  {
    callbacks.add(new InnerFileCallback(this));
  }
  
  protected static class TriangleRegion extends Region
  {
    public TriangleRegion()
    {
      super();
    }
    
    public TriangleRegion(Region r)
    {
      super(r);
    }
    
    @Override
    public boolean isInRegion(Region r)
    {
      return r.getInt(NUM_HOPS) == 100 || r.getInt(NUM_PARCELS) == 10000;
    }
    
    @Override
    public TriangleRegion getRegion(Region r)
    {
      return new TriangleRegion(r);
    }
  }
}
