/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain.labpal;

import java.util.HashMap;
import java.util.Map;

import ca.uqac.lif.cep.Processor;
import ca.uqac.lif.cep.supplychain.Constants;
import ca.uqac.lif.cep.tuples.TupleMap;
import ca.uqac.lif.labpal.Random;

/**
 * A source of randomly-generated parcel events, used for testing and
 * benchmarking.
 */
public class RandomParcelSource extends RandomSource
{
  /**
   * The length of each slice
   */
  protected int m_sliceLength;

  /**
   * The number of slices to keep alive at any given time
   */
  protected int m_numSlices;

  /**
   * The index of the next slice for which to produce an event
   */
  protected int m_sliceIndex;
  
  /**
   * The probability that a parcel is rerouted
   * each time it is delivered
   */
  protected float m_reroutingProbability;

  /**
   * The ID of the lowest slice that is alive
   */
  protected int m_lowestSliceId;

  /**
   * The ID of the highest slice that is alive
   */
  protected int m_highestSliceId;
  
  /**
   * A map keeping the count of the number of events produced so far
   * for each "live" slice
   */
  protected Map<Integer,ParcelState> m_sliceStates;
  
  /**
   * The number of distinct values the "x" and "y" coordinates can have
   */
  protected int m_hopCoordinates = 10;
  
  /**
   * Creates a new random parcel source
   * @param r A random generator
   * @param num_events The number of events to produce
   * @param slice_length The length of each slice
   * @param num_slices The number of slices to keep alive at any given time
   * @param rerouting_probability The probability that a parcel is rerouted
   * each time it is delivered (must be between 0 and 1)
   */
  /*@ require rerouting_probability >= 0 && rerouting_probability <=1 @*/
  public RandomParcelSource(Random r, int num_events, int slice_length, int num_slices, 
      float rerouting_probability)
  {
    super(r, num_events);
    m_sliceLength = slice_length;
    m_numSlices = num_slices;
    m_lowestSliceId = 0;
    m_highestSliceId = 0;
    m_sliceStates = new HashMap<Integer,ParcelState>();
  }

  @Override
  protected TupleMap getEvent()
  {
    TupleMap tuple = new TupleMap();
    int state_size = m_sliceStates.size();
    if (state_size < m_numSlices)
    {
      // Start a new slice
      ParcelState parcel = new ParcelState(m_highestSliceId, m_random.nextInt(m_hopCoordinates), 
          m_random.nextInt(m_hopCoordinates), m_random.nextInt(m_hopCoordinates), m_random.nextInt(m_hopCoordinates), m_sliceLength);
      m_sliceStates.put(m_highestSliceId, parcel);
      tuple.put(Constants.S_TIMESTAMP, m_eventCount);
      tuple.put(Constants.S_SHIPMENT_ID, m_highestSliceId);
      tuple.put(Constants.S_LOCATION_X, parcel.m_lastX);
      tuple.put(Constants.S_LOCATION_Y, parcel.m_lastY);
      tuple.put(Constants.S_DESTINATION_X, parcel.m_destinationX);
      tuple.put(Constants.S_DESTINATION_Y, parcel.m_destinationY);
      tuple.put(Constants.S_TYPE, Constants.S_PICKUP);
      parcel.m_lastPickup = true;
      m_highestSliceId++;
    }
    else
    {
      assert state_size == m_numSlices;
      ParcelState parcel = m_sliceStates.get(m_sliceIndex);
      tuple.put(Constants.S_TIMESTAMP, m_eventCount);
      tuple.put(Constants.S_SHIPMENT_ID, m_sliceIndex);
      tuple.put(Constants.S_DESTINATION_X, parcel.m_destinationX);
      tuple.put(Constants.S_DESTINATION_Y, parcel.m_destinationY);
      if (parcel.m_lastPickup) // Last event was a pick-up, output a delivery
      {
        tuple.put(Constants.S_TYPE, Constants.S_DELIVERY);
        parcel.m_lastPickup = false;
        parcel.m_hopsLeft--;
      }
      else // Output a pickup
      {
        if (parcel.m_hopsLeft == 1)
        {
          parcel.m_lastX = parcel.m_destinationX;
          parcel.m_lastY = parcel.m_destinationY;
        }
        else
        {
          // Flip a (unfair) coin to decide if we reroute the parcel
          if (m_random.nextFloat() < m_reroutingProbability)
          {
            parcel.m_destinationX = m_random.nextInt(m_hopCoordinates);
            parcel.m_destinationY = m_random.nextInt(m_hopCoordinates);
          }
          parcel.m_lastX = m_random.nextInt(m_hopCoordinates);
          parcel.m_lastY = m_random.nextInt(m_hopCoordinates);
        }
      }
      tuple.put(Constants.S_LOCATION_X, parcel.m_lastX);
      tuple.put(Constants.S_LOCATION_Y, parcel.m_lastY);
      if (parcel.m_hopsLeft == 0)
      {
        // This parcel has finished its lifecycle
        m_sliceStates.remove(m_sliceIndex);
        m_lowestSliceId++;
        m_sliceIndex++;
        if (m_sliceIndex >= m_highestSliceId)
        {
          m_sliceIndex = m_lowestSliceId;
        }
      }
    }
    return tuple;
  }

  @Override
  public Processor duplicate(boolean with_state)
  {
    throw new UnsupportedOperationException("This source cannot be duplicated");
  }
  
  /**
   * A simple data structure used to represent the current state
   * of a parcel being simulated  
   */
  protected static class ParcelState
  {
    /**
     * The last X position of the parcel
     */
    public int m_lastX = 0;
    
    /**
     * The last Y position of the parcel
     */
    public int m_lastY = 0;
    
    /**
     * The X coordinate of the parcel's declared destination
     */
    public int m_destinationX = 0;
    
    /**
     * The Y coordinate of the parcel's declared destination
     */
    public int m_destinationY = 0;
    
    /**
     * The parcel's ID
     */
    public int m_shipmentId = 0;
    
    /**
     * The number of hops left before the parcel reaches its destination
     */
    public int m_hopsLeft = 0;
    
    /**
     * Whether the last event produced for this parcel was a <tt>Pickup</tt>
     */
    public boolean m_lastPickup = true;
    
    /**
     * Creates a new parcel.
     * @param shipment_id
     * @param start_x The last X position of the parcel
     * @param start_y The last Y position of the parcel
     * @param dest_x The X coordinate of the parcel's declared destination
     * @param dest_y The Y coordinate of the parcel's declared destination
     * @param num_hops The number of hops left before the parcel reaches its
     * destination
     */
    public ParcelState(int shipment_id, int start_x, int start_y, int dest_x, int dest_y, int num_hops)
    {
      super();
      m_shipmentId = shipment_id;
      m_lastX = start_x;
      m_lastY = start_y;
      m_destinationX = dest_x;
      m_destinationY = dest_y;
      m_hopsLeft = num_hops;
    }
    
    @Override
    public int hashCode()
    {
      return m_shipmentId;
    }
  }
}
