/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain.labpal;

import ca.uqac.lif.cep.Processor;
import ca.uqac.lif.cep.supplychain.AverageShippingTime;
import ca.uqac.lif.cep.supplychain.DecreasingDistance;
import ca.uqac.lif.cep.supplychain.MaxReroutings;
import ca.uqac.lif.cep.supplychain.NewReroutings;
import ca.uqac.lif.cep.supplychain.ParcelLifecycle;
import ca.uqac.lif.cep.supplychain.ParcelsInTransit;
import ca.uqac.lif.cep.supplychain.TimeDistribution;
import ca.uqac.lif.json.JsonFalse;
import ca.uqac.lif.json.JsonNumber;
import ca.uqac.lif.labpal.ExperimentFactory;
import ca.uqac.lif.labpal.Region;
import static ca.uqac.lif.cep.supplychain.labpal.ParcelExperiment.NUM_HOPS;
import static ca.uqac.lif.cep.supplychain.labpal.ParcelExperiment.NUM_PARCELS;
import static ca.uqac.lif.cep.supplychain.labpal.ParcelExperiment.PROPERTY;
import static ca.uqac.lif.cep.supplychain.labpal.ParcelExperiment.REROUTING_PROB;
import static ca.uqac.lif.cep.supplychain.labpal.StreamExperiment.MULTITHREAD;

/**
 * An {@link ExperimentFactory} that produces {@link ParcelExperiment}s.
 */
public class ParcelExperimentFactory extends ExperimentFactory<SupplyChainLab,ParcelExperiment>
{
  public ParcelExperimentFactory(SupplyChainLab lab)
  {
    super(lab, ParcelExperiment.class);
  }
  
  @Override
  protected ParcelExperiment createExperiment(Region r)
  {
    Processor p = getProcessor(r);
    if (p == null)
    {
      return null;
    }
    ParcelExperiment exp = new ParcelExperiment(m_lab.getRandom(), SupplyChainLab.MAX_TRACE_LENGTH, 
        r.getInt(NUM_PARCELS), r.getInt(NUM_HOPS), ((JsonNumber) r.get(REROUTING_PROB)).numberValue().floatValue(),
        r.getString(PROPERTY), p);
    exp.setInput(MULTITHREAD, JsonFalse.instance);
    exp.setEventStep(SupplyChainLab.s_eventStep);
    exp.setPropertyDescription(getPropertyDescription(r));
    exp.setImageUrl(getImageUrl(r));
    exp.setPredictedThroughput(guessThroughput(r));
    return exp;
  }
  
  protected Processor getProcessor(Region r)
  {
    String property_name = r.getString(PROPERTY);
    if (property_name == null)
    {
      return null;
    }
    if (property_name.compareTo(DecreasingDistance.NAME) == 0)
    {
      return new DecreasingDistance();
    }
    if (property_name.compareTo(ParcelLifecycle.NAME) == 0)
    {
      return new ParcelLifecycle();
    }
    if (property_name.compareTo(MaxReroutings.NAME) == 0)
    {
      return new MaxReroutings(5);
    }
    if (property_name.compareTo(NewReroutings.NAME) == 0)
    {
      return new NewReroutings();
    }
    if (property_name.compareTo(ParcelsInTransit.NAME) == 0)
    {
      return new ParcelsInTransit();
    }
    if (property_name.compareTo(TimeDistribution.NAME) == 0)
    {
      return new TimeDistribution();
    }
    if (property_name.compareTo(AverageShippingTime.NAME) == 0)
    {
      return new AverageShippingTime(2, 50, 50);
    }
    return null;
  }

  protected String getPropertyDescription(Region r)
  {
    String property_name = r.getString(PROPERTY);
    if (property_name == null)
    {
      return null;
    }
    if (property_name.compareTo(DecreasingDistance.NAME) == 0)
    {
      return "It checks that each parcel is getting closer to its destination when taking each hop.";
    }
    if (property_name.compareTo(ParcelLifecycle.NAME) == 0)
    {
      return "It checks that each parcel follows a predefined lifecycle specified by a Moore machine.";
    }
    if (property_name.compareTo(MaxReroutings.NAME) == 0)
    {
      return "It checks that each parcel is rerouted a maximum number of times.";
    }
    if (property_name.compareTo(NewReroutings.NAME) == 0)
    {
      return "It checks that each parcel, when rerouted, is directed towards a new destination each time.";
    }
    if (property_name.compareTo(ParcelsInTransit.NAME) == 0)
    {
      return "It counts the number of parcels in transit at any given moment.";
    }
    if (property_name.compareTo(TimeDistribution.NAME) == 0)
    {
      return "It computes the distribution of delivery time for all parcels.";
    }
    if (property_name.compareTo(AverageShippingTime.NAME) == 0)
    {
      return "It compares the average shipping time for each source-destination pair between two sliding windows of events.";
    }
    return null;
  }
  
  protected String getImageUrl(Region r)
  {
    String property_name = r.getString(PROPERTY);
    if (property_name == null)
    {
      return null;
    }
    if (property_name.compareTo(DecreasingDistance.NAME) == 0)
    {
      return "/resource/DecreasingDistance.png";
    }
    if (property_name.compareTo(ParcelLifecycle.NAME) == 0)
    {
      return "/resource/ParcelLifecycle.png";
    }
    if (property_name.compareTo(MaxReroutings.NAME) == 0)
    {
      return "/resource/MaxReroutings.png";
    }
    if (property_name.compareTo(NewReroutings.NAME) == 0)
    {
      return "/resource/Rerouting.png";
    }
    /*if (property_name.compareTo(ParcelsInTransit.NAME) == 0)
    {
      // No picture for this one
      return "/resource/ParcelsInTransit.png";
    }*/
    if (property_name.compareTo(TimeDistribution.NAME) == 0)
    {
      return "/resource/TimeDistro.png";
    }
    if (property_name.compareTo(AverageShippingTime.NAME) == 0)
    {
      return "/resource/HopAll.png";
    }
    return null;
  }
  
  /**
   * Estimates the throughput of an experiment.
   * These throughput values are rough estimates based on values
   * collected when running the experiments.
   * @param r The region representing an experiment
   * @return The estimated throughput
   */
  protected float guessThroughput(Region r)
  {
    String property_name = r.getString(PROPERTY);
    if (property_name == null)
    {
      return 0f;
    }
    if (property_name.compareTo(DecreasingDistance.NAME) == 0)
    {
      return 300000f;
    }
    if (property_name.compareTo(ParcelLifecycle.NAME) == 0)
    {
      return 450000f;
    }
    if (property_name.compareTo(MaxReroutings.NAME) == 0)
    {
      return 1000000f;
    }
    if (property_name.compareTo(NewReroutings.NAME) == 0)
    {
      return 750000f;
    }
    if (property_name.compareTo(ParcelsInTransit.NAME) == 0)
    {
      return 100000f;
    }
    if (property_name.compareTo(TimeDistribution.NAME) == 0)
    {
      return 1200000f;
    }
    if (property_name.compareTo(AverageShippingTime.NAME) == 0)
    {
      return 15000f;
    }
    return 0f;
  }
}
