/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain.labpal;

import ca.uqac.lif.cep.Processor;
import ca.uqac.lif.json.JsonFalse;
import ca.uqac.lif.labpal.Random;

/**
 * Experiment that runs a processor chain on a stream of parcel events.
 */
public class ParcelExperiment extends StreamExperiment
{
  /**
   * The number of parcels in transit at any given point in time
   */
  public static final transient String NUM_PARCELS = "Number of parcels";

  /**
   * The number of hops taken by any single parcel during its lifecycle
   */
  public static final transient String NUM_HOPS = "Slice length";

  /**
   * The probability that a parcel is rerouted each time it is delivered
   */
  public static final transient String REROUTING_PROB = "Re-routing probability";

  /**
   * Creates a new empty parcel experiment
   */
  public ParcelExperiment()
  {
    super();
    setDescription("Runs a processor chain on a stream of parcel events.");
    describe(NUM_PARCELS, "The number of parcels in transit at any given point in time");
    describe(REROUTING_PROB, "The probability that a parcel is rerouted each time it is delivered");
    describe(NUM_HOPS, "The number of hops taken by any single parcel during its lifecycle");
    addKeyToHide(MULTITHREAD);
  }

  public ParcelExperiment(Random r, int num_events, int num_parcels, int num_hops, float rerouting_prob, String property, Processor p)
  {
    this();
    setInput(NUM_PARCELS, num_parcels);
    setInput(NUM_HOPS, num_hops);
    setInput(PROPERTY, property);
    setInput(REROUTING_PROB, rerouting_prob);
    setInput(StreamExperiment.MULTITHREAD, JsonFalse.instance);
    setProcessor(p);
    setSource(new RandomParcelSource(r, num_events, num_parcels, num_hops, rerouting_prob));
  }
  
  @Override
  public String getDescription()
  {
    StringBuilder out = new StringBuilder();
    out.append("<p>Runs a processor chain on a stream of parcel events.");
    if (m_propertyDescription != null)
    {
      out.append(" ").append(m_propertyDescription);
    }
    if (m_imageUrl != null)
    {
      out.append("</p>\n<p>\nGraphically, this chain of processors is represented as in the following diagram:</p>\n");
      out.append("<p><img src=\"").append(m_imageUrl).append("\"/>\n");
    }
    out.append("</p>");
    return out.toString();
  }
}
