/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain.labpal;

import ca.uqac.lif.json.JsonElement;
import ca.uqac.lif.json.JsonNumber;
import ca.uqac.lif.labpal.Experiment;
import ca.uqac.lif.labpal.macro.MacroMap;
import java.util.Collection;
import java.util.Map;

/**
 * Computes various derived statistics on the lab's experimental
 * results of the lab.
 */
public class DataStats extends MacroMap
{
  /**
   * A factory used to access the experiments
   */
  protected transient ParcelExperimentFactory m_factory;
  
  /**
   * Instantiates the macro and defines its named data points
   * @param lab The lab from which to fetch the values
   */
  public DataStats(SupplyChainLab lab, ParcelExperimentFactory factory)
  {
    super(lab);
    m_factory = factory;
    add("updatespersec", "Number of updates per second that a parcel could produce to be still handled in realtime");
    add("tracelength", "Length of the trace (in events)");
    add("throughputTenThousand", "Throughput for 10,000 parcels at a time");
  }

  @Override
  public void computeValues(Map<String, JsonElement> map)
  {
    map.put("tracelength", new JsonNumber(SupplyChainLab.MAX_TRACE_LENGTH - 1));
    Collection<Experiment> exps = m_lab.getExperiments();
    float min_throughput = -1;
    @SuppressWarnings("unused")
    ParcelExperiment min_experiment = null;
    for (Experiment e : exps)
    {
      if (!(e instanceof ParcelExperiment))
        continue;
      ParcelExperiment pe = (ParcelExperiment) e;
      JsonNumber jn = (JsonNumber) pe.read(ParcelExperiment.THROUGHPUT);
      if (jn == null)
        continue;
      float throughput = jn.numberValue().floatValue();
      if (min_throughput == -1 || min_throughput > throughput)
      {
        min_throughput = throughput;
        min_experiment = pe;
      }
    }
    if (min_throughput > -1)
    {
      map.put("throughputTenThousand", new JsonNumber(min_throughput));
      map.put("updatespersec", new JsonNumber(min_throughput / 10000f));
    }
    else
    {
      map.put("throughputTenThousand", new JsonNumber(0));
      map.put("updatespersec", new JsonNumber(0));
    }
  }
}
