/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain;

import static ca.uqac.lif.cep.Connector.INPUT;
import static ca.uqac.lif.cep.Connector.OUTPUT;
import ca.uqac.lif.cep.GroupProcessor;
import ca.uqac.lif.cep.fsm.FunctionTransition;
import ca.uqac.lif.cep.fsm.MooreMachine;
import ca.uqac.lif.cep.fsm.TransitionOtherwise;
import ca.uqac.lif.cep.functions.Constant;
import ca.uqac.lif.cep.functions.ContextAssignment;
import ca.uqac.lif.cep.functions.ContextVariable;
import ca.uqac.lif.cep.functions.Cumulate;
import ca.uqac.lif.cep.functions.CumulativeFunction;
import ca.uqac.lif.cep.functions.FunctionTree;
import ca.uqac.lif.cep.functions.StreamVariable;
import ca.uqac.lif.cep.functions.TurnInto;
import ca.uqac.lif.cep.tmf.Slice;
import ca.uqac.lif.cep.tmf.SliceLast;
import ca.uqac.lif.cep.tuples.FetchAttribute;
import ca.uqac.lif.cep.util.Booleans;
import ca.uqac.lif.cep.util.Equals;
import ca.uqac.lif.cep.util.Numbers;

import static ca.uqac.lif.cep.supplychain.Constants.C_DELIVERY;
import static ca.uqac.lif.cep.supplychain.Constants.S_DESTINATION_X;
import static ca.uqac.lif.cep.supplychain.Constants.S_DESTINATION_Y;
import static ca.uqac.lif.cep.supplychain.Constants.S_LOCATION_X;
import static ca.uqac.lif.cep.supplychain.Constants.S_LOCATION_Y;
import static ca.uqac.lif.cep.supplychain.Constants.S_SHIPMENT_ID;
import static ca.uqac.lif.cep.supplychain.Constants.S_TIMESTAMP;
import static ca.uqac.lif.cep.supplychain.Constants.S_TYPE;
import static ca.uqac.lif.cep.supplychain.Constants.ZERO;

import ca.uqac.lif.cep.Connector;

/**
 * Processor chain that computes the distribution of delivery time for all parcels.
 * Graphically, this chain of processors is illustrated as follows:
 * <p>
 * <img src="{@docRoot}/doc-files/TimeDistro.png" alt="Processor chain">
 */
public class TimeDistribution extends GroupProcessor
{
  /**
   * A name given to this processor chain
   */
  public static final transient String NAME = "Time distribution";

  /**
   * Creates a new instance of the processor
   */
  public TimeDistribution()
  {
    super(1, 1);
    SliceLast s_id = new SliceLast(new FetchAttribute(S_SHIPMENT_ID), new TimeMachine());
    GroupProcessor counter = new GroupProcessor(1, 1);
    {
      TurnInto one = new TurnInto(1);
      Cumulate sum = new Cumulate(new CumulativeFunction<Number>(Numbers.addition));
      Connector.connect(one, sum);
      counter.addProcessors(one, sum);
      counter.associateInput(INPUT, one, INPUT);
      counter.associateOutput(OUTPUT, sum, OUTPUT);
    }
    Slice s_time = new Slice(new FunctionTree(Numbers.floor, 
        new FunctionTree(Numbers.division, StreamVariable.X, new Constant(100))), counter);
    Connector.connect(s_id, s_time);
    addProcessors(s_id, s_time);
    associateInput(INPUT, s_id, INPUT);
    associateOutput(OUTPUT, s_time, OUTPUT);
  }

  public static class TimeMachine extends MooreMachine
  {
    protected static ContextVariable t = new ContextVariable("t");

    protected static ContextVariable x = new ContextVariable("x");

    protected static ContextVariable y = new ContextVariable("y");

    protected static FetchAttribute dst_x = new FetchAttribute(S_DESTINATION_X);

    protected static FetchAttribute dst_y = new FetchAttribute(S_DESTINATION_Y);

    protected static FetchAttribute loc_x = new FetchAttribute(S_LOCATION_X);

    protected static FetchAttribute loc_y = new FetchAttribute(S_LOCATION_Y);

    protected static FetchAttribute ts = new FetchAttribute(S_TIMESTAMP);
    
    protected static FetchAttribute type = new FetchAttribute(S_TYPE);

    public TimeMachine()
    {
      super(1, 1);
      addSymbol(2, new ContextVariable("d"));
      addInitialAssignment(new ContextAssignment("d", ZERO));
      addTransition(0, new TransitionOtherwise(1, 
          new ContextAssignment("x", dst_x), 
          new ContextAssignment("y", dst_y), 
          new ContextAssignment("t", ts)));
      addTransition(1, new FunctionTransition(
          new FunctionTree(Booleans.and,
              new FunctionTree(Equals.instance, type, C_DELIVERY),
              new FunctionTree(Booleans.and,
                  new FunctionTree(Equals.instance, loc_x, x),
                  new FunctionTree(Equals.instance, loc_y, y)
                  )
              ), 2, new ContextAssignment("d", new FunctionTree(
                  Numbers.subtraction, ts, t))));
      addTransition(1, new TransitionOtherwise(1));
      addTransition(2, new TransitionOtherwise(2));
    }
  }
}
