/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain;

import static ca.uqac.lif.cep.Connector.INPUT;
import static ca.uqac.lif.cep.Connector.OUTPUT;
import static ca.uqac.lif.cep.supplychain.Constants.C_DELIVERY;
import static ca.uqac.lif.cep.supplychain.Constants.C_PICKUP;
import static ca.uqac.lif.cep.supplychain.Constants.ONE;
import static ca.uqac.lif.cep.supplychain.Constants.S_SHIPMENT_ID;
import static ca.uqac.lif.cep.supplychain.Constants.S_TYPE;
import static ca.uqac.lif.cep.supplychain.Constants.ZERO;

import ca.uqac.lif.cep.Connector;
import ca.uqac.lif.cep.GroupProcessor;
import ca.uqac.lif.cep.fsm.FunctionTransition;
import ca.uqac.lif.cep.fsm.MooreMachine;
import ca.uqac.lif.cep.functions.ApplyFunction;
import ca.uqac.lif.cep.functions.Cumulate;
import ca.uqac.lif.cep.functions.CumulativeFunction;
import ca.uqac.lif.cep.functions.FunctionTree;
import ca.uqac.lif.cep.functions.StreamVariable;
import ca.uqac.lif.cep.tmf.Slice;
import ca.uqac.lif.cep.tuples.FetchAttribute;
import ca.uqac.lif.cep.util.Bags.RunOn;
import ca.uqac.lif.cep.util.Equals;
import ca.uqac.lif.cep.util.Maps;
import ca.uqac.lif.cep.util.Numbers;

/**
 * Chain of processors that computes the number of parcels that are in
 * transit.
 */
public class ParcelsInTransit extends GroupProcessor
{
  /**
   * A name given to this processor chain
   */
  public static final transient String NAME = "Parcels in transit";
  
  public ParcelsInTransit()
  {
    super(1, 1);
    Slice slice = new Slice(new FetchAttribute(S_SHIPMENT_ID), new TransitMachine());
    ApplyFunction values = new ApplyFunction(Maps.values);
    Connector.connect(slice, values);
    RunOn add = new RunOn(new Cumulate(new CumulativeFunction<Number>(Numbers.addition)));
    Connector.connect(values, add);
    addProcessors(slice, values, add);
    associateInput(INPUT, slice, INPUT);
    associateOutput(OUTPUT, add, OUTPUT);
  }
  
  public static class TransitMachine extends MooreMachine
  {
    /**
     * A static reference to the function that fetches the attribute "type"
     * in a tuple
     */
    protected static final FetchAttribute s_getEvent = new FetchAttribute(S_TYPE);
    
    public TransitMachine()
    {
      super(1, 1);
      addSymbol(0, ZERO);
      addSymbol(1, ONE);
      addTransition(0, new FunctionTransition(
          new FunctionTree(Equals.instance, new FunctionTree(s_getEvent, StreamVariable.X), C_PICKUP), 1));
      addTransition(1, new FunctionTransition(
          new FunctionTree(Equals.instance, new FunctionTree(s_getEvent, StreamVariable.X), C_DELIVERY), 0));
    }
    
    @Override
    public TransitMachine duplicate(boolean with_state)
    {
      return new TransitMachine();
    }
  }  
}
