/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain;

import ca.uqac.lif.cep.GroupProcessor;
import ca.uqac.lif.cep.fsm.FunctionTransition;
import ca.uqac.lif.cep.fsm.MooreMachine;
import ca.uqac.lif.cep.fsm.TransitionOtherwise;
import ca.uqac.lif.cep.functions.ApplyFunction;
import ca.uqac.lif.cep.functions.ContextAssignment;
import ca.uqac.lif.cep.functions.ContextVariable;
import ca.uqac.lif.cep.functions.FunctionTree;
import ca.uqac.lif.cep.functions.StreamVariable;
import ca.uqac.lif.cep.ltl.Troolean;
import ca.uqac.lif.cep.tmf.Slice;
import ca.uqac.lif.cep.tuples.FetchAttribute;
import ca.uqac.lif.cep.util.Booleans;
import ca.uqac.lif.cep.util.Equals;
import ca.uqac.lif.cep.util.Maps;

import static ca.uqac.lif.cep.Connector.INPUT;
import static ca.uqac.lif.cep.Connector.OUTPUT;
import static ca.uqac.lif.cep.supplychain.Constants.C_DELIVERY;
import static ca.uqac.lif.cep.supplychain.Constants.C_PICKUP;
import static ca.uqac.lif.cep.supplychain.Constants.FALSE;
import static ca.uqac.lif.cep.supplychain.Constants.INCONCLUSIVE;
import static ca.uqac.lif.cep.supplychain.Constants.TRUE;
import static ca.uqac.lif.cep.supplychain.Constants.S_LOCATION_X;
import static ca.uqac.lif.cep.supplychain.Constants.S_LOCATION_Y;
import static ca.uqac.lif.cep.supplychain.Constants.S_TYPE;

import ca.uqac.lif.cep.Connector;

import static ca.uqac.lif.cep.supplychain.Constants.S_SHIPMENT_ID;

/**
 * Chain of processors that evaluates the lifecycle of each parcel in a
 * supply chain. The lifecycle is made of the following constraints:
 * <ul>
 * <li>The first event for a given parcel must be a <tt>pickup</tt></li>
 * <li>Events <tt>pickup</tt> and <tt>delivery</tt> must alternate</li>
 * <li>The x-y location of a <tt>pickup</tt> must match the x-y location
 * of the previous <tt>delivery</tt> (if any)</li>
 * </ul>
 * <p>
 * In a given stream, there may be <tt>pickup</tt> and <tt>delivery</tt>
 * events for multiple parcels; the previous constraints hold for each
 * parcel stream individually. A parcel is identified by the value of
 * attribute <tt>id</tt> in each event.
 * <p>
 * Graphically, this can be represented by the following chain of
 * processors:
 * <p>
 * <img src="{@docRoot}/doc-files/ParcelLifecycle.png" alt="Processor chain"> 
 * <p>
 * The processor returns <tt>false</tt> whenever one of the parcels
 * does not follow its lifecycle.
 */
public class ParcelLifecycle extends GroupProcessor
{
  /**
   * A name given to this processor chain
   */
  public static final transient String NAME = "Parcel Lifecycle";
  
	/**
	 * Creates a new parcel lifecycle processor
	 */
	public ParcelLifecycle()
	{
		super(1, 1);
		Slice slice = new Slice(new FetchAttribute(S_SHIPMENT_ID), new LifecycleMachine());
		ApplyFunction and = new ApplyFunction(new FunctionTree(
		    Troolean.AND_ARRAY_FUNCTION, Maps.values));
		Connector.connect(slice, and);
		addProcessors(slice, and);
		associateInput(INPUT, slice, INPUT);
		associateOutput(OUTPUT, and, OUTPUT);
	}
	
	/**
	 * A Moore machine representing the lifecycle of a single parcel. The
	 * transitions of the machine can be represented graphically as follows:
	 * <p>
	 * <img src="{@docRoot}/doc-files/LifecycleMachine.png" alt="Moore machine">
	 */
	protected static class LifecycleMachine extends MooreMachine
	{
	  /**
	   * A reference to the context variable "x"
	   */
		protected static final ContextVariable s_X = new ContextVariable("x");
		
		/**
		 * A reference to the context variable "y"
		 */
		protected static final ContextVariable s_Y = new ContextVariable("y");
		
		/**
		 * A static reference to the function that fetches the attribute "type"
		 * in a tuple
		 */
		protected static final FetchAttribute s_getEvent = new FetchAttribute(S_TYPE);
		
		/**
     * A static reference to the function that fetches the attribute "location x"
     * in a tuple
     */
		protected static final FetchAttribute s_getX = new FetchAttribute(S_LOCATION_X);
		
		/**
     * A static reference to the function that fetches the attribute "location y"
     * in a tuple
     */
		protected static final FetchAttribute s_getY = new FetchAttribute(S_LOCATION_Y);
		
		/**
		 * Creates a new instance of the Moore machine
		 */
		public LifecycleMachine()
		{
			super(1, 1);
			addSymbol(0, TRUE);
			addSymbol(1, INCONCLUSIVE);
			addSymbol(2, TRUE);
			addSymbol(3, FALSE);
			addTransition(0, new FunctionTransition(
							new FunctionTree(Equals.instance, new FunctionTree(s_getEvent, StreamVariable.X), C_PICKUP), 1));
			addTransition(0, new TransitionOtherwise(3));
			addTransition(1, new FunctionTransition(
					new FunctionTree(Equals.instance, new FunctionTree(s_getEvent, StreamVariable.X), C_DELIVERY), 2,
					new ContextAssignment("x", s_getX),
					new ContextAssignment("y", s_getY)));
			addTransition(1, new TransitionOtherwise(3));
			addTransition(2, new FunctionTransition(
					new FunctionTree(Booleans.and,
							new FunctionTree(Equals.instance, new FunctionTree(s_getEvent, StreamVariable.X), C_PICKUP),
							new FunctionTree(Booleans.and,
									new FunctionTree(Equals.instance, s_getX, s_X),
									new FunctionTree(Equals.instance, s_getY, s_Y))
							),
					1));
			addTransition(2, new TransitionOtherwise(3));
			addTransition(3, new TransitionOtherwise(3));
		}
		
		@Override
		public LifecycleMachine duplicate(boolean with_state)
		{
			return new LifecycleMachine();
		}
	}
}
