/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain;

import ca.uqac.lif.cep.Connector;
import static ca.uqac.lif.cep.Connector.BOTTOM;
import static ca.uqac.lif.cep.Connector.INPUT;
import static ca.uqac.lif.cep.Connector.OUTPUT;
import static ca.uqac.lif.cep.Connector.TOP;
import ca.uqac.lif.cep.GroupProcessor;
import ca.uqac.lif.cep.functions.ApplyFunction;
import ca.uqac.lif.cep.functions.Cumulate;
import ca.uqac.lif.cep.functions.CumulativeFunction;
import ca.uqac.lif.cep.functions.FunctionTree;
import ca.uqac.lif.cep.supplychain.util.ProjectTuple;
import ca.uqac.lif.cep.tmf.Fork;
import ca.uqac.lif.cep.tmf.Insert;
import ca.uqac.lif.cep.tmf.SliceLast;
import ca.uqac.lif.cep.tmf.Trim;
import ca.uqac.lif.cep.tuples.FetchAttribute;
import ca.uqac.lif.cep.tuples.Tuple;
import ca.uqac.lif.cep.util.Bags;
import ca.uqac.lif.cep.util.Booleans;
import ca.uqac.lif.cep.util.Equals;
import ca.uqac.lif.cep.util.Lists;
import ca.uqac.lif.cep.util.Sets;
import java.util.HashSet;

/**
 * Checks the property that each parcel, when rerouted, is directed towards
 * a new destination each time.
 * Graphically, this chain of processors is illustrated as follows:
 * <p>
 * <img src="{@docRoot}/doc-files/Rerouting.png" alt="Processor chain">
 */
public class NewReroutings extends GroupProcessor
{
  /**
   * A name given to this processor chain
   */
  public static final transient String NAME = "New reroutings";
  
  /**
   * Creates a new instance of the processor chain
   */
  public NewReroutings()
  {
    super(1, 1);
    SliceLast slice = new SliceLast(new FetchAttribute(Constants.S_SHIPMENT_ID), new ParcelRerouting());
    Lists.Unpack unpack = new Lists.Unpack();
    Connector.connect(slice, unpack);
    Cumulate and = new Cumulate(new CumulativeFunction<Boolean>(Booleans.and));
    Connector.connect(unpack, and);
    addProcessors(slice, unpack, and);
    associateInput(INPUT, slice, INPUT);
    associateOutput(OUTPUT, and, OUTPUT);
  }
  
  /**
   * The processor chain checking the reroutings for an individual parcel
   */
  public static class ParcelRerouting extends GroupProcessor
  {
    public ParcelRerouting()
    {
      super(1, 1);
      ApplyFunction dst_pair = new ApplyFunction(new ProjectTuple(
          new ProjectTuple.NameFunctionPair("x", new FetchAttribute(Constants.S_DESTINATION_X)),
          new ProjectTuple.NameFunctionPair("y", new FetchAttribute(Constants.S_DESTINATION_Y))));
      Fork f0 = new Fork(2);
      Connector.connect(dst_pair, f0);
      Fork f1 = new Fork(2);
      Connector.connect(f0, TOP, f1, INPUT);
      ApplyFunction equals = new ApplyFunction(new FunctionTree(Booleans.not, Equals.instance));
      Connector.connect(f1, TOP, equals, TOP);
      Trim trim = new Trim(1);
      Connector.connect(f1, BOTTOM, trim, INPUT);
      Connector.connect(trim, OUTPUT, equals, BOTTOM);
      Fork f2 = new Fork(2);
      Connector.connect(f0, BOTTOM, f2, INPUT);
      Sets.PutInto in_set = new Sets.PutInto();
      Connector.connect(f2, BOTTOM, in_set, INPUT);
      Insert ins = new Insert(1, new HashSet<Tuple>());
      Connector.connect(in_set, ins);
      ApplyFunction is_in_set = new ApplyFunction(Bags.isElement);
      Connector.connect(f2, TOP, is_in_set, TOP);
      Connector.connect(ins, OUTPUT, is_in_set, BOTTOM);
      ApplyFunction implies = new ApplyFunction(Booleans.implies);
      Connector.connect(equals, OUTPUT, implies, TOP);
      Connector.connect(is_in_set, OUTPUT, implies, BOTTOM);
      addProcessors(dst_pair, f0, f1, equals, trim, f2, in_set, ins, is_in_set, implies);
      associateInput(INPUT, dst_pair, INPUT);
      associateOutput(OUTPUT, implies, OUTPUT);
    }
  }
}
