/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain;

import ca.uqac.lif.cep.Connector;
import ca.uqac.lif.cep.GroupProcessor;
import ca.uqac.lif.cep.functions.ApplyFunction;
import ca.uqac.lif.cep.functions.Constant;
import ca.uqac.lif.cep.functions.Cumulate;
import ca.uqac.lif.cep.functions.CumulativeFunction;
import ca.uqac.lif.cep.functions.FunctionTree;
import ca.uqac.lif.cep.supplychain.util.ProjectTuple;
import ca.uqac.lif.cep.tmf.SliceLast;
import ca.uqac.lif.cep.tuples.FetchAttribute;
import ca.uqac.lif.cep.util.Bags;
import ca.uqac.lif.cep.util.Booleans;
import ca.uqac.lif.cep.util.Lists;
import ca.uqac.lif.cep.util.Numbers;
import ca.uqac.lif.cep.util.Sets;

/**
 * Checks that each parcel is rerouted a maximum number of times.
 * This property can be monitored by the following chain of processors:
 * <p>
 * <img src="{@docRoot}/doc-files/MaxReroutings.png" alt="Processor chain">
 */
public class MaxReroutings extends GroupProcessor
{
  /**
   * A name given to this processor chain
   */
  public static final transient String NAME = "Max reroutings";

  /**
   * The maximum number of reroutings allowed for a single parcel
   */
  protected int m_bound = 0;

  /**
   * Creates a new instance of the processor chain.
   * @param bound The maximum number of reroutings allowed for a single parcel
   */
  public MaxReroutings(int bound)
  {
    super(1, 1);
    m_bound = bound;
    SliceLast slice = new SliceLast(new FetchAttribute(Constants.S_SHIPMENT_ID), new ReroutingCounter());
    Lists.Unpack unpack = new Lists.Unpack();
    Connector.connect(slice, unpack);
    Cumulate and = new Cumulate(new CumulativeFunction<Boolean>(Booleans.and));
    Connector.connect(unpack, and);
    addProcessors(slice, unpack, and);
    associateInput(0, slice, 0);
    associateOutput(0, and, 0);
  }

  public class ReroutingCounter extends GroupProcessor
  {
    public ReroutingCounter()
    {
      super(1, 1);
      ApplyFunction dst_pair = new ApplyFunction(new ProjectTuple(
          new ProjectTuple.NameFunctionPair("x", new FetchAttribute(Constants.S_DESTINATION_X)),
          new ProjectTuple.NameFunctionPair("y", new FetchAttribute(Constants.S_DESTINATION_Y))));
      Sets.PutInto in_set = new Sets.PutInto();
      Connector.connect(dst_pair, in_set);
      ApplyFunction grt_k = new ApplyFunction(new FunctionTree(
          Numbers.isLessOrEqual,
          Bags.getSize,
          new Constant(m_bound)
          ));
      Connector.connect(in_set, grt_k);
      addProcessors(dst_pair, in_set, grt_k);
      associateInput(0, dst_pair, 0);
      associateOutput(0, grt_k, 0);
    }
  }

  public ReroutingCounter newReroutingCounter()
  {
    return new ReroutingCounter();
  }
}
