/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain;

import static ca.uqac.lif.cep.Connector.BOTTOM;
import static ca.uqac.lif.cep.Connector.INPUT;
import static ca.uqac.lif.cep.Connector.OUTPUT;
import static ca.uqac.lif.cep.Connector.TOP;
import static ca.uqac.lif.cep.supplychain.Constants.S_SHIPMENT_ID;

import ca.uqac.lif.cep.Connector;
import ca.uqac.lif.cep.GroupProcessor;
import ca.uqac.lif.cep.functions.ApplyFunction;
import ca.uqac.lif.cep.functions.FunctionTree;
import ca.uqac.lif.cep.ltl.Troolean;
import ca.uqac.lif.cep.ltl.TrooleanCast;
import ca.uqac.lif.cep.tmf.Fork;
import ca.uqac.lif.cep.tmf.Pad;
import ca.uqac.lif.cep.tmf.Slice;
import ca.uqac.lif.cep.tmf.Trim;
import ca.uqac.lif.cep.tuples.FetchAttribute;
import ca.uqac.lif.cep.util.Maps;
import ca.uqac.lif.cep.util.Numbers;

/**
 * Processor chain that checks that each parcel is getting closer to 
 * its destination when taking each hop.
 * Graphically, this chain of processors is illustrated as follows:
 * <p>
 * <img src="{@docRoot}/doc-files/DecreasingDistance.png" alt="Processor chain">
 */
public class DecreasingDistance extends GroupProcessor
{
  /**
   * A name given to this processor chain
   */
  public static final transient String NAME = "Decreasing distance";
  
  /**
   * Creates a new instance of the processor chain.
   */
  public DecreasingDistance()
  {
    super(1, 1);
    Slice slice = new Slice(new FetchAttribute(S_SHIPMENT_ID), new Pad(new DistanceDifference(), 1, Troolean.Value.TRUE));
    ApplyFunction and = new ApplyFunction(new FunctionTree(
        Troolean.AND_ARRAY_FUNCTION, Maps.values));
    Connector.connect(slice, and);
    addProcessors(slice, and);
    associateInput(INPUT, slice, INPUT);
    associateOutput(OUTPUT, and, OUTPUT);
  }
  
  public static class DistanceDifference extends GroupProcessor
  {
    public DistanceDifference()
    {
      super(1, 1);
      ApplyFunction distance = new ApplyFunction(DistanceToTarget.instance);
      Fork fork = new Fork(2);
      Connector.connect(distance, fork);
      Trim trim = new Trim(1);
      Connector.connect(fork, BOTTOM, trim, INPUT);
      ApplyFunction minus = new ApplyFunction(new FunctionTree(TrooleanCast.instance, Numbers.isGreaterOrEqual));
      Connector.connect(fork, TOP, minus, TOP);
      Connector.connect(trim, OUTPUT, minus, BOTTOM);
      addProcessors(distance, fork, trim, minus);
      associateInput(INPUT, distance, INPUT);
      associateOutput(OUTPUT, minus, OUTPUT);
    }
  }
}
