/*
    Processor chains for hyperconnected logistics
    Copyright (C) 2018-2019 Laboratoire d'informatique formelle

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ca.uqac.lif.cep.supplychain;

import ca.uqac.lif.cep.GroupProcessor;
import ca.uqac.lif.cep.fsm.MooreMachine;
import ca.uqac.lif.cep.fsm.TransitionOtherwise;
import ca.uqac.lif.cep.functions.ApplyFunction;
import ca.uqac.lif.cep.functions.Constant;
import ca.uqac.lif.cep.functions.ContextAssignment;
import ca.uqac.lif.cep.functions.ContextVariable;
import ca.uqac.lif.cep.functions.Cumulate;
import ca.uqac.lif.cep.functions.CumulativeFunction;
import ca.uqac.lif.cep.functions.FunctionTree;
import ca.uqac.lif.cep.functions.StreamVariable;
import ca.uqac.lif.cep.functions.TurnInto;
import ca.uqac.lif.cep.ltl.Troolean;
import ca.uqac.lif.cep.supplychain.util.ProjectTuple;

import static ca.uqac.lif.cep.supplychain.Constants.S_LOCATION_X;
import static ca.uqac.lif.cep.supplychain.Constants.S_LOCATION_Y;
import static ca.uqac.lif.cep.supplychain.Constants.S_SHIPMENT_ID;
import static ca.uqac.lif.cep.supplychain.Constants.S_TIMESTAMP;
import static ca.uqac.lif.cep.supplychain.Constants.ZERO;

import ca.uqac.lif.cep.Connector;
import static ca.uqac.lif.cep.Connector.BOTTOM;
import static ca.uqac.lif.cep.Connector.INPUT;
import static ca.uqac.lif.cep.Connector.OUTPUT;
import static ca.uqac.lif.cep.Connector.TOP;

import ca.uqac.lif.cep.tmf.Fork;
import ca.uqac.lif.cep.tmf.Slice;
import ca.uqac.lif.cep.tmf.SliceLast;
import ca.uqac.lif.cep.tmf.Trim;
import ca.uqac.lif.cep.tmf.Window;
import ca.uqac.lif.cep.tuples.FetchAttribute;
import ca.uqac.lif.cep.util.Lists;
import ca.uqac.lif.cep.util.Maps;
import ca.uqac.lif.cep.util.Numbers;

/**
 * Processor chain that computes the average shipping time over two sliding windows.
 * Graphically, this chain of processors is illustrated as follows:
 * <p>
 * <img src="{@docRoot}/doc-files/HopAll.png" alt="Processor chain">
 */
public class AverageShippingTime extends GroupProcessor
{
  /**
   * A name given to this processor chain
   */
  public static final transient String NAME = "Average shipping time";
  
  /**
   * The maximum ratio
   */
  protected int m_k;
  
  /**
   * The width of the "past" window
   */
  protected int m_n;
  
  /**
   * The width of the "present" window
   */
  protected int m_m;
  
  /**
   * A {@link Constant} containing the numerical value <tt>k</tt>
   */
  protected Constant m_constantK;
  
  /**
   * Creates a new instance of the processor chain
   * @param k The maximum ratio
   * @param m The width of the "past" window
   * @param n The width of the "present" window
   */
  public AverageShippingTime(int k, int m, int n)
  {
    super(1, 1);
    m_k = k;
    m_m = m;
    m_n = n;
    m_constantK = new Constant(k);
    SliceLast sl_id = new SliceLast(new FetchAttribute(S_SHIPMENT_ID), new HopMachine());
    Lists.Unpack unpack = new Lists.Unpack();
    Connector.connect(sl_id, unpack);
    Slice sl_hop = new Slice(new FetchAttribute("hop"), new TrendDistance());
    Connector.connect(unpack, sl_hop);
    ApplyFunction and = new ApplyFunction(new FunctionTree(
        Troolean.AND_ARRAY_FUNCTION, Maps.values));
    Connector.connect(sl_hop, and);
    addProcessors(sl_id, unpack, sl_hop, and);
    associateInput(INPUT, sl_id, INPUT);
    associateOutput(OUTPUT, and, OUTPUT);
  }
  
  /**
   * A Moore machine that computes the time spent for each parcel and each hop
   */
  public static class HopMachine extends MooreMachine
  {
    protected static FetchAttribute loc_x = new FetchAttribute(S_LOCATION_X);

    protected static FetchAttribute loc_y = new FetchAttribute(S_LOCATION_Y);
    
    protected static FetchAttribute ts = new FetchAttribute(S_TIMESTAMP);
    
    public HopMachine()
    {
      super(1, 1);
      addInitialAssignment(new ContextAssignment("d", ZERO));
      addTransition(0, new TransitionOtherwise(1, 
          new ContextAssignment("x", loc_x),
          new ContextAssignment("y", loc_y),
          new ContextAssignment("t", ts)));
      addTransition(1, new TransitionOtherwise(2, 
          new ContextAssignment("x'", loc_x),
          new ContextAssignment("y'", loc_y),
          new ContextAssignment("d", 
              new FunctionTree(Numbers.subtraction, ts, new ContextVariable("t")))));
      addTransition(2, new TransitionOtherwise(1,
          new ContextAssignment("x", loc_x),
          new ContextAssignment("y", loc_y),
          new ContextAssignment("t", ts)));
      addSymbol(2, new ProjectTuple(
          new ProjectTuple.NameFunctionPair("hop", 
              new ProjectTuple(
                  new ProjectTuple.NameFunctionPair("x", new ContextVariable("x")),
                  new ProjectTuple.NameFunctionPair("y", new ContextVariable("y")),
                  new ProjectTuple.NameFunctionPair("x'", new ContextVariable("x'")),
                  new ProjectTuple.NameFunctionPair("y'", new ContextVariable("y'")))),
          new ProjectTuple.NameFunctionPair("dur", new ContextVariable("d"))));
    }
  }
  
  public class TrendDistance extends GroupProcessor
  {
    public TrendDistance()
    {
      super(1, 1);
      ApplyFunction get_dur = new ApplyFunction(new FetchAttribute("dur"));
      Fork f0 = new Fork(2);
      Connector.connect(get_dur, f0);
      Trim trim = new Trim(m_n);
      Connector.connect(f0, TOP, trim, INPUT);
      Window win_top = new Window(new RunningAverage(), m_m);
      Connector.connect(trim, win_top);
      Window win_bot = new Window(new RunningAverage(), m_n);
      Connector.connect(f0, BOTTOM, win_bot, INPUT);
      ApplyFunction ratio = new ApplyFunction(new FunctionTree(
          Numbers.isLessOrEqual,
          new FunctionTree(Numbers.division, StreamVariable.X, StreamVariable.Y),
          m_constantK));
      Connector.connect(win_top, OUTPUT, ratio, TOP);
      Connector.connect(win_bot, OUTPUT, ratio, BOTTOM);
      addProcessors(get_dur, f0, trim, win_top, win_bot, ratio);
      associateInput(INPUT, get_dur, INPUT);
      associateOutput(OUTPUT, ratio, OUTPUT);
    }
    
    /**
     * Computes the running average of a stream of numbers
     */
    public class RunningAverage extends GroupProcessor
    {
      public RunningAverage()
      {
        super(1, 1);
        Fork f = new Fork(2);
        Cumulate sum = new Cumulate(new CumulativeFunction<Number>(Numbers.addition));
        Connector.connect(f, TOP, sum, INPUT);
        TurnInto one = new TurnInto(1);
        Connector.connect(f, BOTTOM, one, INPUT);
        Cumulate sum_one = new Cumulate(new CumulativeFunction<Number>(Numbers.addition));
        Connector.connect(one, sum_one);
        ApplyFunction div = new ApplyFunction(Numbers.division);
        Connector.connect(sum, OUTPUT, div, TOP);
        Connector.connect(sum_one, OUTPUT, div, BOTTOM);
        addProcessors(f, sum, one, sum_one, div);
        associateInput(INPUT, f, INPUT);
        associateOutput(OUTPUT, div, OUTPUT);
      }
      
      @Override
      public RunningAverage duplicate(boolean with_state)
      {
        return new RunningAverage();
      }
    }
  }
  
  
  @Override
  public AverageShippingTime duplicate(boolean with_state)
  {
    return new AverageShippingTime(m_k, m_m, m_n);
  }
  
  /**
   * Returns a new instance of the internal Moore machine. This method is
   * used only for testing and debugging.
   * @return The machine
   */
  public HopMachine newHopMachine()
  {
    return new HopMachine();
  }
}
