#!/usr/bin/env bash

## "PPHPC Java vs OpenCL-CPU Datasets"
## Copyright (c) 2017 Nuno Fachada
## MIT License

# ####################################### #
# Script used to create the Java datasets #
# ####################################### #

# Location of the Java PPHPC implementation
PPDIR="${HOME}/workspace/pphpc/java"

# Location of output files
OUTDIR="./java"

# Location of simulation configuration files
CFGDIR="${HOME}/workspace/pphpc-configs"

# Number of threads
THREADS=20

# Maximum memory to use for Java heap (megabytes)
JMAXMEM=61440

# Java command to run PPHPC
PPCMD="java -Xms%sm -Xmx${JMAXMEM}m -cp ${PPDIR}/bin:${PPDIR}/lib/* org.laseeb.pphpc.PredPrey"

# Create output directories if they don't yet exist
mkdir -p ${OUTDIR}/simout
mkdir -p ${OUTDIR}/times

# Start runs
for RUN in {1..10}
do
    # Determine seed for this run
    SEED=`echo ${RUN} | md5sum | cut -f 1 -d ' '`
    SEED=`echo ${SEED} | awk '{print toupper($0)}'`
    SEED=`echo "ibase=16; ${SEED}" | bc`

    # Cycle through model sizes
    for SIZE in 400 800 1600 3200 6400
    do

        # Cycle through parameter sets
        for VERS in 1 2
        do

            # Determine initial memory for Java heap
            let "JSMEM=512 * (${SIZE} * (${VERS} ** 2) / 200)"
            JSMEM=$(($JSMEM<$JMAXMEM?$JSMEM:$JMAXMEM))

            # Equal with reproducibility parallelization strategy (EX)
            OUTFILE="${OUTDIR}/simout/stats${SIZE}v${VERS}r${RUN}.txt"
            TIMEFILE="${OUTDIR}/times/time${SIZE}v${VERS}r${RUN}.txt"
            if [ ! -f ${OUTFILE} ] || [ ! -f ${TIMEFILE} ]; then
                echo "Run #${RUN} | ${SIZE}@${VERS} | Xms=${JSMEM}m | SEED=${SEED}"
                /usr/bin/time -o ${TIMEFILE} `printf "${PPCMD}" ${JSMEM}` -p ${CFGDIR}/config${SIZE}v${VERS}.txt -s ${OUTFILE} -r ${SEED} -n ${THREADS} -ps EX
            fi

        done
    done
done
