
/*

mangaDataDB schema version2

Stores metadata and data from the 3d reductions of the MaNGA DRP.

This schema builds a database where data arrays are stored as such.  The array size is NWAVE.

E.g., flux is stored as a single DOUBLE PRECISION [] .  This allows for easier creation, and loading,
and less complex tables.

Create Dec,2014 - B. Cherinka

*/

CREATE SCHEMA mangadatadb;

SET search_path TO mangadatadb;

CREATE TABLE mangadatadb.cube (pk serial PRIMARY KEY NOT NULL, plate INTEGER, mangaid TEXT, designid INTEGER,
					pipeline_info_pk INTEGER, wavelength_pk INTEGER, ifudesign_pk INTEGER, manga_target_pk integer,
					specres numeric[], xfocal double precision, yfocal double precision, ra DOUBLE PRECISION,
					dec DOUBLE PRECISION, cube_shape_pk INTEGER);

CREATE INDEX q3c_cube_idx ON mangadatadb.cube (functions.q3c_ang2ipix(ra, dec));

CREATE TABLE mangadatadb.cart (pk serial PRIMARY KEY NOT NULL, id INTEGER);

CREATE TABLE mangadatadb.cart_to_cube (pk serial PRIMARY KEY NOT NULL, cube_pk INTEGER, cart_pk INTEGER);

CREATE TABLE mangadatadb.rssfiber (pk serial PRIMARY KEY NOT NULL, flux numeric[],
				ivar double precision[], mask INTEGER[], xpos double precision[],
				ypos double precision[], exposure_no INTEGER, mjd INTEGER, exposure_pk INTEGER, cube_pk INTEGER, fibers_pk INTEGER);

CREATE TABLE mangadatadb.spaxel (pk serial PRIMARY KEY NOT NULL, flux numeric[], ivar double precision[],
				mask integer[], cube_pk INTEGER, x INTEGER, y INTEGER);

CREATE TABLE mangadatadb.wavelength (pk serial PRIMARY KEY NOT NULL, wavelength numeric[],
				bintype TEXT);

CREATE INDEX CONCURRENTLY wave_idx ON mangadatadb.wavelength USING GIN(wavelength);

CREATE TABLE mangadatadb.pipeline_info (pk serial PRIMARY KEY NOT NULL, pipeline_name_pk INTEGER,
	pipeline_stage_pk INTEGER, pipeline_version_pk INTEGER, pipeline_completion_status_pk INTEGER);

CREATE TABLE mangadatadb.pipeline_name (pk serial PRIMARY KEY NOT NULL, label TEXT);

CREATE TABLE mangadatadb.pipeline_stage (pk serial PRIMARY KEY NOT NULL, label TEXT);

CREATE TABLE mangadatadb.pipeline_completion_status (pk serial PRIMARY KEY NOT NULL, label TEXT);

CREATE TABLE mangadatadb.pipeline_version (pk serial PRIMARY KEY NOT NULL, version TEXT);

CREATE TABLE mangadatadb.fits_header_value (pk serial PRIMARY KEY NOT NULL,
	value TEXT, index INTEGER, comment TEXT, fits_header_keyword_pk INTEGER, cube_pk INTEGER);

CREATE TABLE mangadatadb.fits_header_keyword(pk serial PRIMARY KEY NOT NULL, label TEXT);

CREATE TABLE mangadatadb.ifudesign (pk serial PRIMARY KEY NOT NULL, name TEXT, nfiber INTEGER, nsky INTEGER,
	nblocks INTEGER, specid INTEGER, maxring INTEGER);

CREATE TABLE mangadatadb.ifu_to_block (pk serial PRIMARY KEY NOT NULL, ifudesign_pk INTEGER, slitblock_pk INTEGER);

CREATE TABLE mangadatadb.slitblock (pk serial PRIMARY KEY NOT NULL, blockid INTEGER, specblockid INTEGER, nfiber INTEGER);

CREATE TABLE mangadatadb.fibers (pk serial PRIMARY KEY NOT NULL, fiberid INTEGER, specfibid INTEGER, fnum INTEGER,
	ring INTEGER, dist_mm double precision,xpmm double precision, ypmm double precision, fiber_type_pk INTEGER,
	target_type_pk INTEGER, ifudesign_pk INTEGER);

CREATE TABLE mangadatadb.fiber_type (pk serial PRIMARY KEY NOT NULL, label TEXT);
CREATE TABLE mangadatadb.target_type (pk serial PRIMARY KEY NOT NULL, label TEXT);

CREATE TABLE mangadatadb.sample (pk serial PRIMARY KEY NOT NULL, manga_tileid INTEGER, ifu_ra double precision, ifu_dec double precision,
	target_ra double precision, target_dec double precision, iauname TEXT, ifudesignsize INTEGER, ifutargetsize INTEGER,
	ifudesignwrongsize INTEGER, field INTEGER, run INTEGER, nsa_redshift double precision, nsa_zdist double precision,
	nsa_absmag_F double precision,nsa_absmag_N double precision,nsa_absmag_u double precision,nsa_absmag_g double precision,
	nsa_absmag_r double precision,nsa_absmag_i double precision,nsa_absmag_z double precision,nsa_mstar double precision,
	nsa_vdisp double precision, nsa_inclination double precision, nsa_petro_th50 double precision,nsa_petroflux_F double precision,
	nsa_petroflux_N double precision,nsa_petroflux_u double precision,nsa_petroflux_g double precision,nsa_petroflux_r double precision,
	nsa_petroflux_i double precision,nsa_petroflux_z double precision,nsa_petroflux_ivar_F double precision,nsa_petroflux_ivar_N double precision,
	nsa_petroflux_ivar_u double precision,nsa_petroflux_ivar_g double precision,nsa_petroflux_ivar_r double precision,
	nsa_petroflux_ivar_i double precision,nsa_petroflux_ivar_z double precision,nsa_sersic_ba double precision, nsa_sersic_n double precision,
	nsa_sersic_phi double precision,nsa_sersic_th50 double precision, nsa_sersicflux_F double precision,nsa_sersicflux_N double precision,
	nsa_sersicflux_u double precision,nsa_sersicflux_g double precision,nsa_sersicflux_r double precision,nsa_sersicflux_i double precision,
	nsa_sersicflux_z double precision,nsa_sersicflux_ivar_F double precision,nsa_sersicflux_ivar_N double precision,
	nsa_sersicflux_ivar_u double precision,nsa_sersicflux_ivar_g double precision,nsa_sersicflux_ivar_r double precision,
	nsa_sersicflux_ivar_i double precision,nsa_sersicflux_ivar_z double precision, cube_pk INTEGER, nsa_version TEXT, nsa_id BIGINT,
	nsa_id100 BIGINT,nsa_ba double precision, nsa_phi double precision, nsa_mstar_el double precision, nsa_petro_th50_el double precision,
	nsa_petroflux_el_F double precision,nsa_petroflux_el_N double precision,nsa_petroflux_el_u double precision,
	nsa_petroflux_el_g double precision,nsa_petroflux_el_r double precision,nsa_petroflux_el_i double precision,
	nsa_petroflux_el_z double precision,nsa_petroflux_el_ivar_F double precision,nsa_petroflux_el_ivar_N double precision,
	nsa_petroflux_el_ivar_u double precision,nsa_petroflux_el_ivar_g double precision,nsa_petroflux_el_ivar_r double precision,
	nsa_petroflux_el_ivar_i double precision,nsa_petroflux_el_ivar_z double precision,nsa_absmag_el_F double precision,
	nsa_absmag_el_N double precision,nsa_absmag_el_u double precision,nsa_absmag_el_g double precision,nsa_absmag_el_r double precision,
	nsa_absmag_el_i double precision,nsa_absmag_el_z double precision,nsa_amivar_el_F double precision,nsa_amivar_el_N double precision,
	nsa_amivar_el_u double precision,nsa_amivar_el_g double precision,nsa_amivar_el_r double precision,nsa_amivar_el_i double precision,
	nsa_amivar_el_z double precision,nsa_extinction_F double precision,nsa_extinction_N double precision,nsa_extinction_u double precision,
	nsa_extinction_g double precision,nsa_extinction_r double precision,nsa_extinction_i double precision,nsa_extinction_z double precision);

CREATE TABLE mangadatadb.wcs (pk serial PRIMARY KEY NOT NULL, cube_pk INTEGER, ctype3 TEXT, crpix3 INTEGER, crval3 numeric, cd3_3 numeric,
	cunit3 TEXT, crpix1 numeric, crpix2 numeric, crval1 numeric, crval2 numeric, cd1_1 numeric, cd2_2 numeric, ctype1 TEXT, ctype2 TEXT,
	cunit1 TEXT, cunit2 TEXT, hduclass TEXT, hduclas1 TEXT, hduclas2 TEXT, errdata TEXT, qualdata TEXT, extname TEXT, naxis1 integer,
	naxis2 integer, naxis3 integer);

CREATE TABLE mangadatadb.cube_shape (pk serial PRIMARY KEY NOT NULL, size INTEGER, total INTEGER, indices integer[][]);

INSERT INTO mangadatadb.fiber_type VALUES (0, 'IFU'),(1,'SKY');
INSERT INTO mangadatadb.target_type VALUES (0, 'science'),(1,'sky'),(2,'standard');
INSERT INTO mangadatadb.cart VALUES (0, '1'),(1, '2'),(2, '3'),(3, '4'),(4, '5'),(5,'6');
INSERT INTO mangadatadb.pipeline_name VALUES (0,'DOS'), (1, 'DRP'), (2, 'DAP');
INSERT INTO mangadatadb.pipeline_version VALUES (0,'trunk'),(1,'v1_0_0'),(2,'v1_1_2'), (3, 'v1_3_3'), (4, 'v1_5_1'),
	(5, 'v1_5_4'), (6, '1.1.1'), (7, 'v1_2_0');
INSERT INTO mangadatadb.pipeline_stage VALUES (0,'DOS'),(1,'DRP-2d'), (2,'DRP-3d'), (3, 'DAP-Block1');
INSERT INTO mangadatadb.pipeline_completion_status VALUES (0,'Queued'), (1,'Transferred'),(2, 'Started'),
	(3, 'Running'),(4,'Done'), (5, 'Fault'), (6, 'Redo');

INSERT INTO mangadatadb.ifudesign VALUES (0,'701','  7',' 1',' 1','1','1'),(1,'702','  7',' 1',' 1','1','1'),
	(2,'703','  7',' 1',' 1','1','1'),(3,'704','  7',' 1',' 1','1','1'),(4,'705','  7',' 1',' 1','1','1'),
	(5,'706','  7',' 1',' 1','1','1'),(6,'707','  7',' 1',' 1','2','1'),(7,'708','  7',' 1',' 1','2','1'),
	(8,'709','  7',' 1',' 1','2','1'),(9,'710','  7',' 1',' 1','2','1'),(10,'711','  7',' 1',' 1','2','1'),
	(11,'712','  7',' 1',' 1','2','1'),(12,'1901',' 19',' 2',' 1','1','2'),(13,'1902',' 19',' 2',' 1','2','2'),
	(14,'3701',' 37',' 2',' 1','1','3'),(15,'3702',' 37',' 2',' 1','1','3'),(16,'3703',' 37',' 2',' 1','2','3'),
	(17,'3704',' 37',' 2',' 1','2','3'),(18,'6101',' 61',' 4',' 2','1','4'),(19,'6102',' 61',' 4',' 2','1','4'),
	(20,'6103',' 61',' 4',' 2','1','4'),(21,'6104',' 61',' 4',' 2','2','4'),(22,'9101',' 91',' 4',' 3','1','5'),
	(23,'9102',' 91',' 4',' 3','2','5'),(24,'12701','127',' 8',' 4','1','6'),(25,'12702','127',' 8',' 4','1','6'),
	(26,'12703','127',' 8',' 4','2','6'),(27,'12704','127',' 8',' 4','2','6'),(28,'12705','127',' 8',' 4','2','6');

INSERT INTO mangadatadb.slitblock VALUES (0,' 1',' 1','21'),(1,' 2',' 2','33'),(2,' 3',' 3','32'),(3,' 4',' 4','39'),
	(4,' 5',' 5','24'),(5,' 6',' 6','33'),(6,' 7',' 7','32'),(7,' 8',' 8','32'),(8,' 9',' 9','39'),(9,'10','10','32'),
	(10,'11','11','32'),(11,'12','12','32'),(12,'13','13','33'),(13,'14','14','32'),(14,'15','15','39'),
	(15,'16','16','24'),(16,'17','17','39'),(17,'18','18','32'),(18,'19','19','32'),(19,'20','20','32'),
	(20,'21','21','33'),(21,'22','22','32'),(22,'23',' 1','39'),(23,'24',' 2','39'),(24,'25',' 3','32'),
	(25,'26',' 4','32'),(26,'27',' 5','32'),(27,'28',' 6','24'),(28,'29',' 7','21'),(29,'30',' 8','39'),
	(30,'31',' 9','32'),(31,'32','10','32'),(32,'33','11','32'),(33,'34','12','33'),(34,'35','13','32'),
	(35,'36','14','32'),(36,'37','15','39'),(37,'38','16','24'),(38,'39','17','39'),(39,'40','18','32'),
	(40,'41','19','32'),(41,'42','20','32'),(42,'43','21','33'),(43,'44','22','32');

INSERT INTO mangadatadb.ifu_to_block VALUES (0,'0','15'),(1,'1','15'),(2,'2','15'),(3,'3','4'),(4,'4','4'),(5,'5','4'),
	(6,'6','37'),(7,'7','37'),(8,'8','37'),(9,'9','27'),(10,'10','27'),(11,'11','27'),(12,'12','0'),(13,'13','28'),
	(14,'14','14'),(15,'15','3'),(16,'16','36'),(17,'17','22'),(18,'18','20'),(19,'18','21'),(20,'19','12'),
	(21,'19','13'),(22,'20','1'),(23,'20','2'),(24,'21','42'),(25,'21','43'),(26,'22','5'),(27,'22','6'),
	(28,'22','7'),(29,'23','33'),(30,'23','34'),(31,'23','35'),(32,'24','16'),(33,'24','17'),(34,'24','18'),
	(35,'24','19'),(36,'25','8'),(37,'25','9'),(38,'25','10'),(39,'25','11'),(40,'26','38'),(41,'26','39'),
	(42,'26','40'),(43,'26','41'),(44,'27','29'),(45,'27','30'),(46,'27','31'),(47,'27','32'),(48,'28','23'),
	(49,'28','24'),(50,'28','25'),(51,'28','26');

INSERT INTO mangadatadb.fibers VALUES
(   0,'   1','   1',' -2','-1',' 11.98601064',' -5.12291310','10.83524500',' 1',' 1 ',Null),
(   1,'   2','   2',' 19',' 2','  0.31114069',' -0.15949406','-0.27363725',' 0',' 0 ','12 '),
(   2,'   3','   3',' 18',' 2','  0.27234791',' -0.24073565','-0.14037730',' 0',' 0 ','12 '),
(   3,'   4','   4',' 17',' 1','  0.15433905',' -0.15969517','-0.00533381',' 0',' 0 ','12 '),
(   4,'   5','   5',' 16',' 1','  0.15604679',' -0.08224928','-0.13914627',' 0',' 0 ','12 '),
(   5,'   6','   6',' 15',' 2','  0.27285441',' -0.00932738','-0.27617801',' 0',' 0 ','12 '),
(   6,'   7','   7',' 14',' 2','  0.30915071','  0.14792489','-0.27182055',' 0',' 0 ','12 '),
(   7,'   8','   8',' 13',' 1','  0.15606639','  0.06976123','-0.14014483',' 0',' 0 ','12 '),
(   8,'   9','   9',' 12',' 0','  0.00000000',' -0.00536884','-0.00335232',' 0',' 0 ','12 '),
(   9,'  10','  10',' 11',' 1','  0.15180674',' -0.07965839',' 0.12903480',' 0',' 0 ','12 '),
(  10,'  11','  11',' 10',' 1','  0.15393742','  0.07422238',' 0.12841250',' 0',' 0 ','12 '),
(  11,'  12','  12','  9',' 1','  0.15364354','  0.14826934','-0.00463648',' 0',' 0 ','12 '),
(  12,'  13','  13','  8',' 2','  0.26853527','  0.22644067','-0.13890873',' 0',' 0 ','12 '),
(  13,'  14','  14','  7',' 2','  0.30965869','  0.30428148','-0.00562878',' 0',' 0 ','12 '),
(  14,'  15','  15','  6',' 2','  0.26938275','  0.22885319',' 0.12971574',' 0',' 0 ','12 '),
(  15,'  16','  16','  5',' 2','  0.31020744','  0.15305023',' 0.26335361',' 0',' 0 ','12 '),
(  16,'  17','  17','  4',' 2','  0.26676624',' -0.00073343',' 0.26337365',' 0',' 0 ','12 '),
(  17,'  18','  18','  3',' 2','  0.30864541',' -0.15769666',' 0.26508431',' 0',' 0 ','12 '),
(  18,'  19','  19','  2',' 2','  0.26521757',' -0.23347547',' 0.13195372',' 0',' 0 ','12 '),
(  19,'  20','  20','  1',' 2','  0.31354085',' -0.31890639','-0.00479116',' 0',' 0 ','12 '),
(  20,'  21','  21',' -1','-1',' 16.29540663','-12.18691300','10.82024500',' 1',' 1 ',Null),
(  21,'  22','  22',' -4','-1',' 16.19967187',' 12.02369400','10.85122700',' 1',' 1 ',Null),
(  22,'  23','  23',' 61',' 4','  0.56324646',' -0.16242441','-0.54134260',' 0',' 0 ','20 '),
(  23,'  24','  24',' 60',' 4','  0.54087210',' -0.00847727','-0.54174200',' 0',' 0 ','20 '),
(  24,'  25','  25',' 59',' 4','  0.56057764','  0.14664273','-0.54089960',' 0',' 0 ','20 '),
(  25,'  26','  26',' 58',' 4','  0.61921318','  0.30241066','-0.53908330',' 0',' 0 ','20 '),
(  26,'  27','  27',' 57',' 4','  0.55957210','  0.38090333','-0.40723255',' 0',' 0 ','20 '),
(  27,'  28','  28',' 56',' 3','  0.46511451','  0.22462107','-0.40604353',' 0',' 0 ','20 '),
(  28,'  29','  29',' 55',' 3','  0.41247672','  0.07031864','-0.40665046',' 0',' 0 ','20 '),
(  29,'  30','  30',' 54',' 3','  0.41062322',' -0.08367775','-0.40367166',' 0',' 0 ','20 '),
(  30,'  31','  31',' 53',' 3','  0.46735735',' -0.24065318','-0.40379153',' 0',' 0 ','20 '),
(  31,'  32','  32',' 52',' 3','  0.41405042',' -0.31847816','-0.27000635',' 0',' 0 ','20 '),
(  32,'  33','  33',' 51',' 2','  0.30978985',' -0.15799125','-0.26958840',' 0',' 0 ','20 '),
(  33,'  34','  34',' 50',' 2','  0.27089763',' -0.00718224','-0.27176669',' 0',' 0 ','20 '),
(  34,'  35','  35',' 49',' 2','  0.31025194','  0.14671022','-0.27218140',' 0',' 0 ','20 '),
(  35,'  36','  36',' 48',' 3','  0.41097591','  0.30325864','-0.27403501',' 0',' 0 ','20 '),
(  36,'  37','  37',' 47',' 4','  0.53588046','  0.45791581','-0.27287228',' 0',' 0 ','20 '),
(  37,'  38','  38',' 46',' 4','  0.55969367','  0.53876698','-0.13823609',' 0',' 0 ','20 '),
(  38,'  39','  39',' 45',' 3','  0.40927777','  0.38184917','-0.13790447',' 0',' 0 ','20 '),
(  39,'  40','  40',' 44',' 2','  0.26965234','  0.22710259','-0.14013929',' 0',' 0 ','20 '),
(  40,'  41','  41',' 43',' 1','  0.15507699','  0.07401478','-0.13502317',' 0',' 0 ','20 '),
(  41,'  42','  42',' 42',' 1','  0.15565801',' -0.08312980','-0.13482382',' 0',' 0 ','20 '),
(  42,'  43','  43',' 41',' 2','  0.26823622',' -0.23694299','-0.13355910',' 0',' 0 ','20 '),
(  43,'  44','  44',' 40',' 3','  0.41252552',' -0.39397472','-0.13487192',' 0',' 0 ','20 '),
(  44,'  45','  45',' 39',' 3','  0.46543734',' -0.46924505',' 0.00133296',' 0',' 0 ','20 '),
(  45,'  46','  46',' 38',' 2','  0.31142848',' -0.31524071','-0.00018897',' 0',' 0 ','20 '),
(  46,'  47','  47',' 37',' 1','  0.15546302',' -0.15927266','-0.00191516',' 0',' 0 ','20 '),
(  47,'  48','  48',' 36',' 0','  0.00000000',' -0.00381302','-0.00089001',' 0',' 0 ','20 '),
(  48,'  49','  49',' 35',' 1','  0.15557198','  0.15175168','-0.00239478',' 0',' 0 ','20 '),
(  49,'  50','  50',' 34',' 2','  0.30950094','  0.30566905','-0.00430736',' 0',' 0 ','20 '),
(  50,'  51','  51',' 33',' 3','  0.46402475','  0.46020716','-0.00294967',' 0',' 0 ','20 '),
(  51,'  52','  52',' 32',' 4','  0.62149612','  0.61765772','-0.00650678',' 0',' 0 ','20 '),
(  52,'  53','  53',' 31',' 4','  0.56067342','  0.54186155',' 0.12792746',' 0',' 0 ','20 '),
(  53,'  54','  54',' -3','-1',' 11.91479765','  4.94569430','10.83722700',' 1',' 1 ',Null),
(  54,'  55','  55',' -2','-1',' 11.97093919',' -5.12730570','10.81822700',' 1',' 1 ',Null),
(  55,'  56','  56',' 30',' 4','  0.54113907','  0.46406540',' 0.26380472',' 0',' 0 ','20 '),
(  56,'  57','  57',' 29',' 4','  0.56774235','  0.39108657',' 0.40231819',' 0',' 0 ','20 '),
(  57,'  58','  58',' 28',' 4','  0.62829815','  0.31478289',' 0.53718828',' 0',' 0 ','20 '),
(  58,'  59','  59',' 27',' 4','  0.57149706','  0.16173702',' 0.54358051',' 0',' 0 ','20 '),
(  59,'  60','  60',' 26',' 3','  0.47099921','  0.23425506',' 0.40206526',' 0',' 0 ','20 '),
(  60,'  61','  61',' 25',' 3','  0.41643085','  0.30984941',' 0.26773745',' 0',' 0 ','20 '),
(  61,'  62','  62',' 24',' 3','  0.41462499','  0.38533110',' 0.13154274',' 0',' 0 ','20 '),
(  62,'  63','  63',' 23',' 2','  0.27238659','  0.23049577',' 0.13085922',' 0',' 0 ','20 '),
(  63,'  64','  64',' 22',' 2','  0.31597326','  0.15532553',' 0.26863486',' 0',' 0 ','20 '),
(  64,'  65','  65',' 21',' 3','  0.41843562','  0.08014616',' 0.40682712',' 0',' 0 ','20 '),
(  65,'  66','  66',' 20',' 4','  0.54651158','  0.00528243',' 0.54395110',' 0',' 0 ','20 '),
(  66,'  67','  67',' 19',' 4','  0.56203911',' -0.15152964',' 0.54073193',' 0',' 0 ','20 '),
(  67,'  68','  68',' 18',' 3','  0.41408390',' -0.07076034',' 0.40673309',' 0',' 0 ','20 '),
(  68,'  69','  69',' 17',' 2','  0.27062135','  0.00380470',' 0.26798215',' 0',' 0 ','20 '),
(  69,'  70','  70',' 16',' 1','  0.15900089','  0.07659908',' 0.13279394',' 0',' 0 ','20 '),
(  70,'  71','  71',' 15',' 1','  0.15295819',' -0.08024817',' 0.13188853',' 0',' 0 ','20 '),
(  71,'  72','  72',' 14',' 2','  0.30411262',' -0.15249756',' 0.26465784',' 0',' 0 ','20 '),
(  72,'  73','  73',' 13',' 3','  0.46057142',' -0.22462364',' 0.40352254',' 0',' 0 ','20 '),
(  73,'  74','  74',' 12',' 4','  0.61655513',' -0.30308510',' 0.53841792',' 0',' 0 ','20 '),
(  74,'  75','  75',' 11',' 4','  0.55398266',' -0.38161612',' 0.40574719',' 0',' 0 ','20 '),
(  75,'  76','  76',' 10',' 3','  0.40655879',' -0.30934597',' 0.26944414',' 0',' 0 ','20 '),
(  76,'  77','  77','  9',' 2','  0.26322568',' -0.23118903',' 0.13560813',' 0',' 0 ','20 '),
(  77,'  78','  78','  8',' 3','  0.40552782',' -0.38825562',' 0.13592468',' 0',' 0 ','20 '),
(  78,'  79','  79','  7',' 4','  0.53208128',' -0.46332172',' 0.27131587',' 0',' 0 ','20 '),
(  79,'  80','  80','  6',' 4','  0.55692192',' -0.54670178',' 0.13521725',' 0',' 0 ','20 '),
(  80,'  81','  81','  5',' 4','  0.61395964',' -0.62098457',' 0.00374429',' 0',' 0 ','20 '),
(  81,'  82','  82','  4',' 4','  0.55899911',' -0.55035996','-0.13395327',' 0',' 0 ','20 '),
(  82,'  83','  83','  3',' 4','  0.53838835',' -0.47496583','-0.26873360',' 0',' 0 ','20 '),
(  83,'  84','  84','  2',' 4','  0.56203640',' -0.39931058','-0.40493972',' 0',' 0 ','20 '),
(  84,'  85','  85','  1',' 4','  0.61819151',' -0.31667435','-0.53748860',' 0',' 0 ','20 '),
(  85,'  86','  86',' -1','-1',' 16.28575665','-12.18630600','10.80922700',' 1',' 1 ',Null),
(  86,'  87','  87',' -2','-1',' 11.97657890',' -5.12414090','10.82725300',' 1',' 1 ',Null),
(  87,'  88','  88',' 37',' 3','  0.46740160',' -0.24104571','-0.40415186',' 0',' 0 ','15 '),
(  88,'  89','  89',' 36',' 3','  0.41070021',' -0.31586432','-0.26944294',' 0',' 0 ','15 '),
(  89,'  90','  90',' 35',' 3','  0.41305664',' -0.39543292','-0.13680532',' 0',' 0 ','15 '),
(  90,'  91','  91',' 34',' 2','  0.30904490',' -0.31458449','-0.00200852',' 0',' 0 ','15 '),
(  91,'  92','  92',' 33',' 2','  0.27143821',' -0.24033105','-0.13662709',' 0',' 0 ','15 '),
(  92,'  93','  93',' 32',' 2','  0.31325438',' -0.16402736','-0.27062140',' 0',' 0 ','15 '),
(  93,'  94','  94',' 31',' 3','  0.41199649',' -0.08513313','-0.40465134',' 0',' 0 ','15 '),
(  94,'  95','  95',' 30',' 3','  0.41042329','  0.06871836','-0.40406437',' 0',' 0 ','15 '),
(  95,'  96','  96',' 29',' 2','  0.27143188',' -0.00975901','-0.27181463',' 0',' 0 ','15 '),
(  96,'  97','  97',' 28',' 1','  0.15666201',' -0.08389757','-0.13607556',' 0',' 0 ','15 '),
(  97,'  98','  98',' 27',' 1','  0.15742258',' -0.16292869','-0.00385548',' 0',' 0 ','15 '),
(  98,'  99','  99',' 26',' 2','  0.26906991',' -0.24012375',' 0.13138291',' 0',' 0 ','15 '),
(  99,' 100',' 100',' 25',' 2','  0.30758827',' -0.16239015',' 0.26417805',' 0',' 0 ','15 '),
( 100,' 101',' 101',' 24',' 1','  0.15230008',' -0.08418914',' 0.13000771',' 0',' 0 ','15 '),
( 101,' 102',' 102',' 23',' 0','  0.00251188',' -0.00705589','-0.00242118',' 0',' 0 ','15 '),
( 102,' 103',' 103',' 22',' 1','  0.15524111','  0.06932202','-0.13641154',' 0',' 0 ','15 '),
( 103,' 104',' 104',' 21',' 2','  0.30854241','  0.14412185','-0.27022772',' 0',' 0 ','15 '),
( 104,' 105',' 105',' 20',' 3','  0.46275771','  0.22323333','-0.40266636',' 0',' 0 ','15 '),
( 105,' 106',' 106',' 19',' 3','  0.40929303','  0.30287468','-0.26948590',' 0',' 0 ','15 '),
( 106,' 107',' 107',' 18',' 2','  0.26711272','  0.22417880','-0.13671214',' 0',' 0 ','15 '),
( 107,' 108',' 108',' 17',' 1','  0.15003484','  0.14446436','-0.00325005',' 0',' 0 ','15 '),
( 108,' 109',' 109',' 16',' 1','  0.15300517','  0.06999589',' 0.13264222',' 0',' 0 ','15 '),
( 109,' 110',' 110',' 15',' 2','  0.26726950',' -0.00852952',' 0.26683734',' 0',' 0 ','15 '),
( 110,' 111',' 111',' 14',' 2','  0.30393951','  0.14544548',' 0.26336756',' 0',' 0 ','15 '),
( 111,' 112',' 112',' 13',' 2','  0.26250200','  0.22213450',' 0.13023648',' 0',' 0 ','15 '),
( 112,' 113',' 113',' 12',' 2','  0.30625278','  0.30070157','-0.00255988',' 0',' 0 ','15 '),
( 113,' 114',' 114',' 11',' 3','  0.40581003','  0.37767769','-0.13391885',' 0',' 0 ','15 '),
( 114,' 115',' 115',' 10',' 3','  0.45849555','  0.45295110','-0.00124225',' 0',' 0 ','15 '),
( 115,' 116',' 116','  9',' 3','  0.40649709','  0.37856317',' 0.13263260',' 0',' 0 ','15 '),
( 116,' 117',' 117','  8',' 3','  0.40329675','  0.29921737',' 0.26372266',' 0',' 0 ','15 '),
( 117,' 118',' 118','  7',' 3','  0.45822430','  0.22233809',' 0.39712621',' 0',' 0 ','15 '),
( 118,' 119',' 119','  6',' 3','  0.40547005','  0.06852665',' 0.39823166',' 0',' 0 ','15 '),
( 119,' 120',' 120','  5',' 3','  0.40690603',' -0.08549709',' 0.39855817',' 0',' 0 ','15 '),
( 120,' 121',' 121','  4',' 3','  0.46204945',' -0.23907632',' 0.39827263',' 0',' 0 ','15 '),
( 121,' 122',' 122','  3',' 3','  0.40899646',' -0.31668328',' 0.26504688',' 0',' 0 ','15 '),
( 122,' 123',' 123','  2',' 3','  0.41103197',' -0.39396724',' 0.13402570',' 0',' 0 ','15 '),
( 123,' 124',' 124','  1',' 3','  0.46347867',' -0.46901729','-0.00258459',' 0',' 0 ','15 '),
( 124,' 125',' 125',' -1','-1',' 16.29715590','-12.19114100','10.82125300',' 1',' 1 ',Null),
( 125,' 126',' 126',' -3','-1',' 37.55753004',' 30.31096100','22.17220100',' 1',' 1 ',Null),
( 126,' 127',' 127',' -2','-1',' 29.99605096',' 20.22996100','22.14720100',' 1',' 1 ',Null),
( 127,' 128',' 128',' 21',' 1','  0.16092430','  0.15110766',' 0.00100941',' 0',' 2 ','5 '),
( 128,' 129',' 129',' 20',' 0','  0.14877586','  0.06957687',' 0.13444310',' 0',' 2 ','5 '),
( 129,' 130',' 130',' 19',' 0','  0.14519645',' -0.08693362',' 0.13142629',' 0',' 2 ','5 '),
( 130,' 131',' 131',' 18',' 0','  0.00982296',' -0.00554370','-0.00041548',' 0',' 2 ','5 '),
( 131,' 132',' 132',' 17',' 1','  0.16702709','  0.07391098','-0.13611528',' 0',' 2 ','5 '),
( 132,' 133',' 133',' 16',' 1','  0.16216636',' -0.08027921','-0.13746167',' 0',' 2 ','5 '),
( 133,' 134',' 134',' 15',' 1','  0.15275763',' -0.16176154','-0.00543219',' 0',' 2 ','5 '),
( 134,' 135',' 135',' 14',' 1','  0.15471959','  0.14507742',' 0.00890834',' 0',' 2 ','4 '),
( 135,' 136',' 136',' 13',' 1','  0.15483083','  0.06598005',' 0.14361899',' 0',' 2 ','4 '),
( 136,' 137',' 137',' 12',' 1','  0.15546226',' -0.08860586',' 0.14242662',' 0',' 2 ','4 '),
( 137,' 138',' 138',' 11',' 0','  0.00000000',' -0.00964166',' 0.00851184',' 0',' 2 ','4 '),
( 138,' 139',' 139',' 10',' 1','  0.15473323','  0.06772213','-0.12549269',' 0',' 2 ','4 '),
( 139,' 140',' 140','  9',' 1','  0.15672890',' -0.08883827','-0.12673551',' 0',' 2 ','4 '),
( 140,' 141',' 141','  8',' 1','  0.15857689',' -0.16821521',' 0.00748299',' 0',' 2 ','4 '),
( 141,' 142',' 142','  7',' 1','  0.15494100','  0.14515783',' 0.01513238',' 0',' 2 ','3 '),
( 142,' 143',' 143','  6',' 1','  0.15817017','  0.06556650',' 0.14765757',' 0',' 2 ','3 '),
( 143,' 144',' 144','  5',' 1','  0.15709785',' -0.08912929',' 0.14401627',' 0',' 2 ','3 '),
( 144,' 145',' 145','  4',' 0','  0.00342975',' -0.00915252',' 0.01190653',' 0',' 2 ','3 '),
( 145,' 146',' 146','  3',' 1','  0.15078703','  0.06868367','-0.12033640',' 0',' 2 ','3 '),
( 146,' 147',' 147','  2',' 1','  0.15237617',' -0.08525084','-0.12378233',' 0',' 2 ','3 '),
( 147,' 148',' 148','  1',' 1','  0.15566399',' -0.16529558',' 0.01028262',' 0',' 2 ','3 '),
( 148,' 149',' 149',' -1','-1',' 25.75225106',' 13.16396100','22.13620100',' 1',' 1 ',Null),
( 149,' 150',' 150',' -6','-1','  7.28896356',' -5.05435980','-5.24878480',' 1',' 1 ',Null),
( 150,' 151',' 151',' 91',' 5','  0.68510461','  0.61905473','-0.26145034',' 0',' 0 ','22 '),
( 151,' 152',' 152',' 90',' 5','  0.71792981','  0.69455217','-0.13030077',' 0',' 0 ','22 '),
( 152,' 153',' 153',' 89',' 4','  0.56772102','  0.54078149','-0.12908900',' 0',' 0 ','22 '),
( 153,' 154',' 154',' 88',' 4','  0.54493363','  0.46287025','-0.26039323',' 0',' 0 ','22 '),
( 154,' 155',' 155',' 87',' 3','  0.42169620','  0.38876998','-0.12768346',' 0',' 0 ','22 '),
( 155,' 156',' 156',' 86',' 2','  0.27768265','  0.23190142','-0.12593631',' 0',' 0 ','22 '),
( 156,' 157',' 157',' 85',' 3','  0.41899668','  0.30831497','-0.26153928',' 0',' 0 ','22 '),
( 157,' 158',' 158',' 84',' 4','  0.56716023','  0.38496309','-0.39594590',' 0',' 0 ','22 '),
( 158,' 159',' 159',' 83',' 4','  0.62524487','  0.30883480','-0.52643835',' 0',' 0 ','22 '),
( 159,' 160',' 160',' 82',' 3','  0.47039274','  0.23018685','-0.39304531',' 0',' 0 ','22 '),
( 160,' 161',' 161',' 81',' 2','  0.31638062','  0.15238900','-0.26012477',' 0',' 0 ','22 '),
( 161,' 162',' 162',' 80',' 1','  0.16231412','  0.07883326','-0.12449132',' 0',' 0 ','22 '),
( 162,' 163',' 163',' 79',' 0','  0.01201823','  0.00201053',' 0.00745410',' 0',' 0 ','22 '),
( 163,' 164',' 164',' 78',' 1','  0.16644418','  0.15726438',' 0.00889336',' 0',' 0 ','22 '),
( 164,' 165',' 165',' 77',' 2','  0.32022745','  0.31104754',' 0.00771829',' 0',' 0 ','22 '),
( 165,' 166',' 166',' 76',' 3','  0.47417820','  0.46500572',' 0.00755598',' 0',' 0 ','22 '),
( 166,' 167',' 167',' 75',' 4','  0.62738106','  0.61819433',' 0.00535475',' 0',' 0 ','22 '),
( 167,' 168',' 168',' 74',' 5','  0.78178001','  0.77257354',' 0.00272238',' 0',' 0 ','22 '),
( 168,' 169',' 169',' 73',' 5','  0.71741015','  0.69705436',' 0.13819656',' 0',' 0 ','22 '),
( 169,' 170',' 170',' 72',' 4','  0.56634966','  0.54244901',' 0.14031091',' 0',' 0 ','22 '),
( 170,' 171',' 171',' 71',' 3','  0.41801259','  0.38841859',' 0.14102562',' 0',' 0 ','22 '),
( 171,' 172',' 172',' 70',' 2','  0.27709968','  0.23429641',' 0.14425778',' 0',' 0 ','22 '),
( 172,' 173',' 173',' 69',' 2','  0.31581310','  0.15704757',' 0.28044972',' 0',' 0 ','22 '),
( 173,' 174',' 174',' 68',' 3','  0.41825079','  0.31051998',' 0.28161603',' 0',' 0 ','22 '),
( 174,' 175',' 175',' 67',' 4','  0.54222026','  0.46466166',' 0.27554063',' 0',' 0 ','22 '),
( 175,' 176',' 176',' 66',' 5','  0.68507609','  0.62300132',' 0.27593450',' 0',' 0 ','22 '),
( 176,' 177',' 177',' 65',' 5','  0.68360680','  0.54240329',' 0.41576844',' 0',' 0 ','22 '),
( 177,' 178',' 178',' 64',' 5','  0.71713597','  0.46428087',' 0.55055746',' 0',' 0 ','22 '),
( 178,' 179',' 179',' 63',' 5','  0.78114322','  0.38821299',' 0.68442819',' 0',' 0 ','22 '),
( 179,' 180',' 180',' 62',' 5','  0.71482715','  0.23400118',' 0.68410750',' 0',' 0 ','22 '),
( 180,' 181',' 181',' 61',' 5','  0.67580757','  0.07917302',' 0.68191733',' 0',' 0 ','22 '),
( 181,' 182',' 182',' -5','-1',' 13.21221838','-12.12236000','-5.26378480',' 1',' 1 ',Null),
( 182,' 183',' 183',' -4','-1',' 16.19668863',' 12.03664000','10.84721500',' 1',' 1 ',Null),
( 183,' 184',' 184',' 60',' 5','  0.68053607',' -0.07523134',' 0.68359243',' 0',' 0 ','22 '),
( 184,' 185',' 185',' 59',' 5','  0.71667298',' -0.23058611',' 0.68533238',' 0',' 0 ','22 '),
( 185,' 186',' 186',' 58',' 5','  0.78034526',' -0.38454343',' 0.68532929',' 0',' 0 ','22 '),
( 186,' 187',' 187',' 57',' 5','  0.71566129',' -0.46291549',' 0.55152660',' 0',' 0 ','22 '),
( 187,' 188',' 188',' 56',' 4','  0.62427758',' -0.30863431',' 0.54895413',' 0',' 0 ','22 '),
( 188,' 189',' 189',' 55',' 4','  0.56323860',' -0.15265589',' 0.54904068',' 0',' 0 ','22 '),
( 189,' 190',' 190',' 54',' 4','  0.54085406','  0.00179028',' 0.54830812',' 0',' 0 ','22 '),
( 190,' 191',' 191',' 53',' 4','  0.56296127','  0.15753249',' 0.54850707',' 0',' 0 ','22 '),
( 191,' 192',' 192',' 52',' 4','  0.62431305','  0.31208380',' 0.54932249',' 0',' 0 ','22 '),
( 192,' 193',' 193',' 51',' 4','  0.56375653','  0.39112095',' 0.41539345',' 0',' 0 ','22 '),
( 193,' 194',' 194',' 50',' 3','  0.47019799','  0.23473847',' 0.41601726',' 0',' 0 ','22 '),
( 194,' 195',' 195',' 49',' 3','  0.41219722','  0.07955179',' 0.41229222',' 0',' 0 ','22 '),
( 195,' 196',' 196',' 48',' 3','  0.41408313',' -0.07585369',' 0.41415055',' 0',' 0 ','22 '),
( 196,' 197',' 197',' 47',' 3','  0.46793410',' -0.23012489',' 0.41374891',' 0',' 0 ','22 '),
( 197,' 198',' 198',' 46',' 4','  0.56245996',' -0.38443427',' 0.41613689',' 0',' 0 ','22 '),
( 198,' 199',' 199',' 45',' 5','  0.67820340',' -0.53899851',' 0.41643952',' 0',' 0 ','22 '),
( 199,' 200',' 200',' 44',' 5','  0.67716182',' -0.61718781',' 0.28157369',' 0',' 0 ','22 '),
( 200,' 201',' 201',' 43',' 4','  0.53879147',' -0.46237630',' 0.28065937',' 0',' 0 ','22 '),
( 201,' 202',' 202',' 42',' 3','  0.41286257',' -0.30787161',' 0.28027009',' 0',' 0 ','22 '),
( 202,' 203',' 203',' 41',' 2','  0.31355027',' -0.15198640',' 0.28058176',' 0',' 0 ','22 '),
( 203,' 204',' 204',' 40',' 2','  0.27245240','  0.00149202',' 0.27990601',' 0',' 0 ','22 '),
( 204,' 205',' 205',' 39',' 1','  0.15748119','  0.08057225',' 0.14393994',' 0',' 0 ','22 '),
( 205,' 206',' 206',' 38',' 1','  0.15984600',' -0.07575001',' 0.14711098',' 0',' 0 ','22 '),
( 206,' 207',' 207',' 37',' 2','  0.26991056',' -0.23012325',' 0.14516982',' 0',' 0 ','22 '),
( 207,' 208',' 208',' 36',' 3','  0.41052222',' -0.38429342',' 0.14637046',' 0',' 0 ','22 '),
( 208,' 209',' 209',' 35',' 4','  0.55791575',' -0.53859075',' 0.14536799',' 0',' 0 ','22 '),
( 209,' 210',' 210',' 34',' 5','  0.70954202',' -0.69398252',' 0.14545246',' 0',' 0 ','22 '),
( 210,' 211',' 211',' 33',' 5','  0.77251411',' -0.77049660',' 0.01073723',' 0',' 0 ','22 '),
( 211,' 212',' 212',' 32',' 5','  0.71130287',' -0.69683023','-0.12511045',' 0',' 0 ','22 '),
( 212,' 213',' 213',' 31',' 5','  0.67510073',' -0.61796196','-0.25974451',' 0',' 0 ','22 '),
( 213,' 214',' 214',' -3','-1',' 11.90667656','  4.95464020','10.83521500',' 1',' 1 ',Null),
( 214,' 215',' 215',' -2','-1',' 11.96888344',' -5.11335980','10.82021500',' 1',' 1 ',Null),
( 215,' 216',' 216',' 30',' 5','  0.65543780',' -0.53996626','-0.39330950',' 0',' 0 ','22 '),
( 216,' 217',' 217',' 29',' 5','  0.68935146',' -0.46336895','-0.52652853',' 0',' 0 ','22 '),
( 217,' 218',' 218',' 28',' 5','  0.75385164',' -0.38527134','-0.66080076',' 0',' 0 ','22 '),
( 218,' 219',' 219',' 27',' 5','  0.69279432',' -0.23278951','-0.66279599',' 0',' 0 ','22 '),
( 219,' 220',' 220',' 26',' 4','  0.60090844',' -0.30911743','-0.52816494',' 0',' 0 ','22 '),
( 220,' 221',' 221',' 25',' 4','  0.53916169',' -0.38600158','-0.39373716',' 0',' 0 ','22 '),
( 221,' 222',' 222',' 24',' 4','  0.51825519',' -0.46421473','-0.25756550',' 0',' 0 ','22 '),
( 222,' 223',' 223',' 23',' 4','  0.54250605',' -0.54058002','-0.12343231',' 0',' 0 ','22 '),
( 223,' 224',' 224',' 22',' 4','  0.60553818',' -0.61606482',' 0.01234024',' 0',' 0 ','22 '),
( 224,' 225',' 225',' 21',' 3','  0.45127575',' -0.46175687',' 0.01097589',' 0',' 0 ','22 '),
( 225,' 226',' 226',' 20',' 2','  0.39274838',' -0.38540474','-0.12458251',' 0',' 0 ','22 '),
( 226,' 227',' 227',' 19',' 2','  0.39025047',' -0.30813026','-0.25931253',' 0',' 0 ','22 '),
( 227,' 228',' 228',' 18',' 3','  0.44563611',' -0.23177023','-0.39352825',' 0',' 0 ','22 '),
( 228,' 229',' 229',' 17',' 4','  0.54070396',' -0.15476782','-0.52771427',' 0',' 0 ','22 '),
( 229,' 230',' 230',' 16',' 5','  0.65978334',' -0.07741935','-0.66293686',' 0',' 0 ','22 '),
( 230,' 231',' 231',' 15',' 5','  0.66352548','  0.07679634','-0.66423863',' 0',' 0 ','22 '),
( 231,' 232',' 232',' 14',' 4','  0.52287352','  0.00071210','-0.52926970',' 0',' 0 ','22 '),
( 232,' 233',' 233',' 13',' 2','  0.39226560',' -0.07534146','-0.39344631',' 0',' 0 ','22 '),
( 233,' 234',' 234',' 12',' 2','  0.29091307',' -0.15334448','-0.26013201',' 0',' 0 ','22 '),
( 234,' 235',' 235',' 11',' 1','  0.24890189',' -0.22936044','-0.12565548',' 0',' 0 ','22 '),
( 235,' 236',' 236',' 10',' 2','  0.29709552',' -0.30742938',' 0.01047490',' 0',' 0 ','22 '),
( 236,' 237',' 237','  9',' 0','  0.14222957',' -0.15183596',' 0.01202113',' 0',' 0 ','22 '),
( 237,' 238',' 238','  8',' 0','  0.13515513',' -0.07447365','-0.12575085',' 0',' 0 ','22 '),
( 238,' 239',' 239','  7',' 2','  0.25470198','  0.00016573','-0.26098831',' 0',' 0 ','22 '),
( 239,' 240',' 240','  6',' 3','  0.40030177','  0.07882981','-0.39665710',' 0',' 0 ','22 '),
( 240,' 241',' 241','  5',' 4','  0.54871115','  0.15450745','-0.52973518',' 0',' 0 ','22 '),
( 241,' 242',' 242','  4',' 5','  0.69994021','  0.23072487','-0.66346487',' 0',' 0 ','22 '),
( 242,' 243',' 243','  3',' 5','  0.76740717','  0.38543330','-0.66371184',' 0',' 0 ','22 '),
( 243,' 244',' 244','  2',' 5','  0.70527589','  0.46298684','-0.52894147',' 0',' 0 ','22 '),
( 244,' 245',' 245','  1',' 5','  0.67281544','  0.53839275','-0.39516896',' 0',' 0 ','22 '),
( 245,' 246',' 246',' -1','-1',' 16.28137927','-12.18236000','10.80721500',' 1',' 1 ',Null),
( 246,' 247',' 247',' -8','-1',' 13.14387641',' 12.07419000','-5.19733520',' 1',' 1 ',Null),
( 247,' 248',' 248','127',' 6','  0.88266139',' -0.29908527','-0.82958171',' 0',' 0 ','25 '),
( 248,' 249',' 249','126',' 5','  0.79194849',' -0.38542230','-0.69064463',' 0',' 0 ','25 '),
( 249,' 250',' 250','125',' 5','  0.73171851',' -0.22746628','-0.69465526',' 0',' 0 ','25 '),
( 250,' 251',' 251','124',' 5','  0.69495059',' -0.07345715','-0.69061097',' 0',' 0 ','25 '),
( 251,' 252',' 252','123',' 6','  0.84235240',' -0.14527118','-0.82917201',' 0',' 0 ','25 '),
( 252,' 253',' 253','122',' 6','  0.82836074','  0.00658675','-0.82807616',' 0',' 0 ','25 '),
( 253,' 254',' 254','121',' 6','  0.83452049','  0.16270625','-0.81856050',' 0',' 0 ','25 '),
( 254,' 255',' 255','120',' 6','  0.87582193','  0.31658468','-0.81696688',' 0',' 0 ','25 '),
( 255,' 256',' 256','119',' 6','  0.93592301','  0.47306793','-0.80825382',' 0',' 0 ','25 '),
( 256,' 257',' 257','118',' 6','  0.86436185','  0.54395031','-0.67280096',' 0',' 0 ','25 '),
( 257,' 258',' 258','117',' 5','  0.77843884','  0.39002197','-0.67436257',' 0',' 0 ','25 '),
( 258,' 259',' 259','116',' 5','  0.72403433','  0.23885449','-0.68380360',' 0',' 0 ','25 '),
( 259,' 260',' 260','115',' 5','  0.69126974','  0.08261720','-0.68624095',' 0',' 0 ','25 '),
( 260,' 261',' 261','114',' 4','  0.55905527','  0.00221373','-0.55878515',' 0',' 0 ','25 '),
( 261,' 262',' 262','113',' 4','  0.57880510',' -0.15333760','-0.55740062',' 0',' 0 ','25 '),
( 262,' 263',' 263','112',' 4','  0.63608202',' -0.30871924','-0.55495586',' 0',' 0 ','25 '),
( 263,' 264',' 264','111',' 5','  0.73593023',' -0.47300747','-0.56213586',' 0',' 0 ','25 '),
( 264,' 265',' 265','110',' 5','  0.69942895',' -0.55293564','-0.42592647',' 0',' 0 ','25 '),
( 265,' 266',' 266','109',' 4','  0.58356450',' -0.39417755','-0.42853496',' 0',' 0 ','25 '),
( 266,' 267',' 267','108',' 3','  0.48641423',' -0.23948707','-0.42216957',' 0',' 0 ','25 '),
( 267,' 268',' 268','107',' 3','  0.43189762',' -0.07700258','-0.42440701',' 0',' 0 ','25 '),
( 268,' 269',' 269','106',' 2','  0.33142331',' -0.15438243','-0.29212964',' 0',' 0 ','25 '),
( 269,' 270',' 270','105',' 3','  0.43564242',' -0.32330283','-0.28989300',' 0',' 0 ','25 '),
( 270,' 271',' 271','104',' 4','  0.56313173',' -0.48010324','-0.29134263',' 0',' 0 ','25 '),
( 271,' 272',' 272','103',' 5','  0.69742998',' -0.63149803','-0.29220394',' 0',' 0 ','25 '),
( 272,' 273',' 273','102',' 5','  0.73256881',' -0.71301251','-0.16074083',' 0',' 0 ','25 '),
( 273,' 274',' 274','101',' 4','  0.58033792',' -0.55697864','-0.15700022',' 0',' 0 ','25 '),
( 274,' 275',' 275','100',' 3','  0.43033782',' -0.40002868','-0.15416603',' 0',' 0 ','25 '),
( 275,' 276',' 276',' 99',' 2','  0.29142582',' -0.23948422','-0.16339638',' 0',' 0 ','25 '),
( 276,' 277',' 277',' 98',' 1','  0.16954065',' -0.16789133','-0.00118392',' 0',' 0 ','25 '),
( 277,' 278',' 278',' 97',' 1','  0.16903580',' -0.08440855','-0.14522318',' 0',' 0 ','25 '),
( 278,' 279',' 279',' 96',' 2','  0.28505344','  0.00132285','-0.28478343',' 0',' 0 ','25 '),
( 279,' 280',' 280',' 95',' 3','  0.42414711','  0.07967099','-0.41663833',' 0',' 0 ','25 '),
( 280,' 281',' 281',' 94',' 4','  0.57582364','  0.15694692','-0.55421514',' 0',' 0 ','25 '),
( 281,' 282',' 282',' 93',' 4','  0.62654519','  0.31082866','-0.54467343',' 0',' 0 ','25 '),
( 282,' 283',' 283',' 92',' 5','  0.71255868','  0.46766225','-0.53877193',' 0',' 0 ','25 '),
( 283,' 284',' 284',' 91',' 6','  0.82052730','  0.62114189','-0.53776946',' 0',' 0 ','25 '),
( 284,' 285',' 285',' -7','-1','  7.21483296','  4.99719020','-5.20533520',' 1',' 1 ',Null),
( 285,' 286',' 286',' -6','-1','  7.29317972',' -5.05980980','-5.24233520',' 1',' 1 ',Null),
( 286,' 287',' 287',' 90',' 6','  0.80326059','  0.69727891','-0.40012885',' 0',' 0 ','25 '),
( 287,' 288',' 288',' 89',' 5','  0.67525238','  0.54394087','-0.39982415',' 0',' 0 ','25 '),
( 288,' 289',' 289',' 88',' 4','  0.56409338','  0.38688363','-0.40846003',' 0',' 0 ','25 '),
( 289,' 290',' 290',' 87',' 3','  0.47548813','  0.23290460','-0.41087118',' 0',' 0 ','25 '),
( 290,' 291',' 291',' 86',' 2','  0.32071771','  0.15559326','-0.27673358',' 0',' 0 ','25 '),
( 291,' 292',' 292',' 85',' 1','  0.15857302','  0.07020587','-0.13817365',' 0',' 0 ','25 '),
( 292,' 293',' 293',' 84',' 0','  0.01964446',' -0.01082053','-0.00652338',' 0',' 0 ','25 '),
( 293,' 294',' 294',' 83',' 1','  0.16063935',' -0.09281016',' 0.13402480',' 0',' 0 ','25 '),
( 294,' 295',' 295',' 82',' 0','  0.13939968','  0.06370181',' 0.13214378',' 0',' 0 ','25 '),
( 295,' 296',' 296',' 81',' 0','  0.13921597','  0.14326446','-0.00193939',' 0',' 0 ','25 '),
( 296,' 297',' 297',' 80',' 2','  0.26604180','  0.22633177','-0.14048263',' 0',' 0 ','25 '),
( 297,' 298',' 298',' 79',' 3','  0.41442291','  0.31008439','-0.27364923',' 0',' 0 ','25 '),
( 298,' 299',' 299',' 78',' 4','  0.53721198','  0.46424849','-0.27148661',' 0',' 0 ','25 '),
( 299,' 300',' 300',' 77',' 3','  0.40235292','  0.38109096','-0.13502138',' 0',' 0 ','25 '),
( 300,' 301',' 301',' 76',' 2','  0.29372757','  0.29794845',' 0.00027202',' 0',' 0 ','25 '),
( 301,' 302',' 302',' 75',' 1','  0.24895520','  0.21722098',' 0.13501661',' 0',' 0 ','25 '),
( 302,' 303',' 303',' 74',' 2','  0.39131824','  0.37427537',' 0.13344785',' 0',' 0 ','25 '),
( 303,' 304',' 304',' 73',' 3','  0.44942655','  0.45369696',' 0.00350069',' 0',' 0 ','25 '),
( 304,' 305',' 305',' 72',' 4','  0.54942294','  0.53524355','-0.13516720',' 0',' 0 ','25 '),
( 305,' 306',' 306',' 71',' 5','  0.67162763','  0.61805666','-0.26664391',' 0',' 0 ','25 '),
( 306,' 307',' 307',' 70',' 6','  0.81438496','  0.77192092','-0.26588679',' 0',' 0 ','25 '),
( 307,' 308',' 308',' 69',' 5','  0.70105678','  0.69156157','-0.13223558',' 0',' 0 ','25 '),
( 308,' 309',' 309',' 68',' 4','  0.60365448','  0.60793040',' 0.00461656',' 0',' 0 ','25 '),
( 309,' 310',' 310',' 67',' 5','  0.69151555','  0.68258563',' 0.14055093',' 0',' 0 ','25 '),
( 310,' 311',' 311',' 66',' 5','  0.75786399','  0.76214157',' 0.00584840',' 0',' 0 ','25 '),
( 311,' 312',' 312',' 65',' 6','  0.85468178','  0.84829641','-0.12854084',' 0',' 0 ','25 '),
( 312,' 313',' 313',' 64',' 6','  0.91223348','  0.91650769',' 0.00853635',' 0',' 0 ','25 '),
( 313,' 314',' 314',' 63',' 6','  0.84363806','  0.83674150',' 0.14289861',' 0',' 0 ','25 '),
( 314,' 315',' 315',' 62',' 6','  0.80107876','  0.75837037',' 0.27636148',' 0',' 0 ','25 '),
( 315,' 316',' 316',' 61',' 5','  0.78611629','  0.67810409',' 0.41093544',' 0',' 0 ','25 '),
( 316,' 317',' 317',' -5','-1',' 13.22646013','-12.12981000','-5.25733520',' 1',' 1 ',Null),
( 317,' 318',' 318',' -4','-1',' 16.19194212',' 12.01119000','10.86366500',' 1',' 1 ',Null),
( 318,' 319',' 319',' 60',' 6','  0.80447607','  0.59650849',' 0.54473710',' 0',' 0 ','25 '),
( 319,' 320',' 320',' 59',' 5','  0.66107310','  0.52420934',' 0.40868209',' 0',' 0 ','25 '),
( 320,' 321',' 321',' 58',' 5','  0.65896982','  0.60439110',' 0.27331415',' 0',' 0 ','25 '),
( 321,' 322',' 322',' 57',' 4','  0.54134942','  0.52874354',' 0.13647127',' 0',' 0 ','25 '),
( 322,' 323',' 323',' 56',' 4','  0.51974364','  0.44883721',' 0.26994292',' 0',' 0 ','25 '),
( 323,' 324',' 324',' 55',' 2','  0.39524204','  0.29383913',' 0.26927805',' 0',' 0 ','25 '),
( 324,' 325',' 325',' 54',' 2','  0.29949249','  0.13751067',' 0.26805765',' 0',' 0 ','25 '),
( 325,' 326',' 326',' 53',' 2','  0.26733545',' -0.01652954',' 0.26593192',' 0',' 0 ','25 '),
( 326,' 327',' 327',' 52',' 2','  0.32046482',' -0.17417550',' 0.26514935',' 0',' 0 ','25 '),
( 327,' 328',' 328',' 51',' 3','  0.41346584',' -0.09983978',' 0.39941488',' 0',' 0 ','25 '),
( 328,' 329',' 329',' 50',' 3','  0.40449245','  0.05773816',' 0.40050824',' 0',' 0 ','25 '),
( 329,' 330',' 330',' 49',' 3','  0.45547746','  0.21419942',' 0.40406814',' 0',' 0 ','25 '),
( 330,' 331',' 331',' 48',' 4','  0.54592910','  0.36821874',' 0.40704879',' 0',' 0 ','25 '),
( 331,' 332',' 332',' 47',' 5','  0.69549640','  0.44264039',' 0.54002296',' 0',' 0 ','25 '),
( 332,' 333',' 333',' 46',' 4','  0.60909924','  0.28793197',' 0.53887458',' 0',' 0 ','25 '),
( 333,' 334',' 334',' 45',' 4','  0.55310806','  0.13394257',' 0.53733879',' 0',' 0 ','25 '),
( 334,' 335',' 335',' 44',' 4','  0.53594949',' -0.02236919',' 0.53471554',' 0',' 0 ','25 '),
( 335,' 336',' 336',' 43',' 4','  0.56573330',' -0.17709381',' 0.53508698',' 0',' 0 ','25 '),
( 336,' 337',' 337',' 42',' 5','  0.71944773',' -0.25784342',' 0.66917445',' 0',' 0 ','25 '),
( 337,' 338',' 338',' 41',' 5','  0.67890735',' -0.10100581',' 0.67004376',' 0',' 0 ','25 '),
( 338,' 339',' 339',' 40',' 5','  0.67293767','  0.05581093',' 0.67048423',' 0',' 0 ','25 '),
( 339,' 340',' 340',' 39',' 5','  0.70061845','  0.20826887',' 0.66995497',' 0',' 0 ','25 '),
( 340,' 341',' 341',' 38',' 5','  0.76426830','  0.36390235',' 0.67423123',' 0',' 0 ','25 '),
( 341,' 342',' 342',' 37',' 6','  0.85086253','  0.51883126',' 0.67767167',' 0',' 0 ','25 '),
( 342,' 343',' 343',' 36',' 6','  0.91739551','  0.43853248',' 0.80797066',' 0',' 0 ','25 '),
( 343,' 344',' 344',' 35',' 6','  0.85428628','  0.28457622',' 0.80671592',' 0',' 0 ','25 '),
( 344,' 345',' 345',' 34',' 6','  0.81561269','  0.13087489',' 0.80530990',' 0',' 0 ','25 '),
( 345,' 346',' 346',' 33',' 6','  0.80318642',' -0.02608184',' 0.80205034',' 0',' 0 ','25 '),
( 346,' 347',' 347',' 32',' 6','  0.82501593',' -0.18265430',' 0.80285315',' 0',' 0 ','25 '),
( 347,' 348',' 348',' 31',' 6','  0.87015425',' -0.33926791',' 0.79861321',' 0',' 0 ','25 '),
( 348,' 349',' 349',' -3','-1',' 11.92696429','  4.94419020','10.85566500',' 1',' 1 ',Null),
( 349,' 350',' 350',' -2','-1',' 11.97474531',' -5.12680980','10.82566500',' 1',' 1 ',Null),
( 350,' 351',' 351',' 30',' 6','  0.93182274',' -0.49333336',' 0.79699131',' 0',' 0 ','25 '),
( 351,' 352',' 352',' 29',' 6','  0.86662681',' -0.57039163',' 0.66351766',' 0',' 0 ','25 '),
( 352,' 353',' 353',' 28',' 5','  0.77549662',' -0.41359976',' 0.66252775',' 0',' 0 ','25 '),
( 353,' 354',' 354',' 27',' 4','  0.62125539',' -0.33327870',' 0.53082901',' 0',' 0 ','25 '),
( 354,' 355',' 355',' 26',' 5','  0.71337827',' -0.49039141',' 0.53037254',' 0',' 0 ','25 '),
( 355,' 356',' 356',' 25',' 6','  0.82596587',' -0.64732090',' 0.53105678',' 0',' 0 ','25 '),
( 356,' 357',' 357',' 24',' 6','  0.80742396',' -0.72437936',' 0.38783220',' 0',' 0 ','25 '),
( 357,' 358',' 358',' 23',' 5','  0.67915940',' -0.56795324',' 0.39481392',' 0',' 0 ','25 '),
( 358,' 359',' 359',' 22',' 4','  0.56266514',' -0.41006890',' 0.39947551',' 0',' 0 ','25 '),
( 359,' 360',' 360',' 21',' 3','  0.46668390',' -0.25429758',' 0.39795526',' 0',' 0 ','25 '),
( 360,' 361',' 361',' 20',' 3','  0.41107442',' -0.33090276',' 0.26269592',' 0',' 0 ','25 '),
( 361,' 362',' 362',' 19',' 4','  0.54148567',' -0.48990411',' 0.26224108',' 0',' 0 ','25 '),
( 362,' 363',' 363',' 18',' 5','  0.67435002',' -0.64363251',' 0.24905744',' 0',' 0 ','25 '),
( 363,' 364',' 364',' 17',' 6','  0.81916012',' -0.79796764',' 0.24836783',' 0',' 0 ','25 '),
( 364,' 365',' 365',' 16',' 6','  0.86279109',' -0.87407036',' 0.11008402',' 0',' 0 ','25 '),
( 365,' 366',' 366',' 15',' 5','  0.70805559',' -0.71696463',' 0.11403488',' 0',' 0 ','25 '),
( 366,' 367',' 367',' 14',' 4','  0.55871918',' -0.56353760',' 0.12000747',' 0',' 0 ','25 '),
( 367,' 368',' 368',' 13',' 3','  0.41405360',' -0.41034761',' 0.13015746',' 0',' 0 ','25 '),
( 368,' 369',' 369',' 12',' 2','  0.26752864',' -0.24755863',' 0.13397011',' 0',' 0 ','25 '),
( 369,' 370',' 370',' 11',' 2','  0.31039107',' -0.32935734','-0.00061575',' 0',' 0 ','25 '),
( 370,' 371',' 371',' 10',' 3','  0.46527499',' -0.48405313','-0.01937685',' 0',' 0 ','25 '),
( 371,' 372',' 372','  9',' 4','  0.61871140',' -0.63755422','-0.01893391',' 0',' 0 ','25 '),
( 372,' 373',' 373','  8',' 5','  0.77238870',' -0.79112366','-0.02523235',' 0',' 0 ','25 '),
( 373,' 374',' 374','  7',' 6','  0.92678799',' -0.94543611','-0.03053965',' 0',' 0 ','25 '),
( 374,' 375',' 375','  6',' 6','  0.86379527',' -0.86809510','-0.16374021',' 0',' 0 ','25 '),
( 375,' 376',' 376','  5',' 6','  0.82203780',' -0.78787710','-0.29588001',' 0',' 0 ','25 '),
( 376,' 377',' 377','  4',' 6','  0.80912610',' -0.70715691','-0.43065346',' 0',' 0 ','25 '),
( 377,' 378',' 378','  3',' 6','  0.82324984',' -0.62503719','-0.56225513',' 0',' 0 ','25 '),
( 378,' 379',' 379','  2',' 6','  0.87074201',' -0.54202802','-0.70122317',' 0',' 0 ','25 '),
( 379,' 380',' 380','  1',' 6','  0.93556608',' -0.45828516','-0.83108950',' 0',' 0 ','25 '),
( 380,' 381',' 381',' -1','-1',' 16.28805767','-12.18881000','10.82066500',' 1',' 1 ',Null),
( 381,' 382',' 382',' -4','-1',' 16.20093845',' 12.02158800','10.85851500',' 1',' 1 ',Null),
( 382,' 383',' 383',' 61',' 4','  0.56887808',' -0.16048282','-0.53995310',' 0',' 0 ','19 '),
( 383,' 384',' 384',' 60',' 4','  0.55252229',' -0.00749647','-0.54397970',' 0',' 0 ','19 '),
( 384,' 385',' 385',' 59',' 4','  0.57208853','  0.14437283','-0.54245240',' 0',' 0 ','19 '),
( 385,' 386',' 386',' 58',' 4','  0.63246215','  0.30393242','-0.54076398',' 0',' 0 ','19 '),
( 386,' 387',' 387',' 57',' 4','  0.57390089','  0.38365685','-0.40948759',' 0',' 0 ','19 '),
( 387,' 388',' 388',' 56',' 3','  0.47821642','  0.23029285','-0.40518042',' 0',' 0 ','19 '),
( 388,' 389',' 389',' 55',' 3','  0.42378434','  0.07580163','-0.40656036',' 0',' 0 ','19 '),
( 389,' 390',' 390',' 54',' 3','  0.42189336',' -0.07995108','-0.40744024',' 0',' 0 ','19 '),
( 390,' 391',' 391',' 53',' 3','  0.47232314',' -0.23436033','-0.40685514',' 0',' 0 ','19 '),
( 391,' 392',' 392',' 52',' 3','  0.41179536',' -0.31117017','-0.27182567',' 0',' 0 ','19 '),
( 392,' 393',' 393',' 51',' 2','  0.31966943',' -0.15891115','-0.27409446',' 0',' 0 ','19 '),
( 393,' 394',' 394',' 50',' 2','  0.28225164',' -0.00256179','-0.27362622',' 0',' 0 ','19 '),
( 394,' 395',' 395',' 49',' 2','  0.32051651','  0.15208121','-0.26823606',' 0',' 0 ','19 '),
( 395,' 396',' 396',' 48',' 3','  0.42697944','  0.31084375','-0.27369439',' 0',' 0 ','19 '),
( 396,' 397',' 397',' 47',' 4','  0.55340018','  0.46546720','-0.27537122',' 0',' 0 ','19 '),
( 397,' 398',' 398',' 46',' 4','  0.57110284','  0.54200355','-0.13958518',' 0',' 0 ','19 '),
( 398,' 399',' 399',' 45',' 3','  0.42479217','  0.38799248','-0.14113999',' 0',' 0 ','19 '),
( 399,' 400',' 400',' 44',' 2','  0.28451859','  0.23371725','-0.13900458',' 0',' 0 ','19 '),
( 400,' 401',' 401',' 43',' 1','  0.17037028','  0.07481630','-0.13947265',' 0',' 0 ','19 '),
( 401,' 402',' 402',' 42',' 1','  0.16540981',' -0.08177819','-0.14027098',' 0',' 0 ','19 '),
( 402,' 403',' 403',' 41',' 2','  0.27002219',' -0.23525528','-0.13968994',' 0',' 0 ','19 '),
( 403,' 404',' 404',' 40',' 3','  0.40564284',' -0.38762926','-0.13845308',' 0',' 0 ','19 '),
( 404,' 405',' 405',' 39',' 3','  0.45411519',' -0.46354253','-0.00226357',' 0',' 0 ','19 '),
( 405,' 406',' 406',' 38',' 2','  0.29953779',' -0.30880148','-0.00468752',' 0',' 0 ','19 '),
( 406,' 407',' 407',' 37',' 0','  0.14652861',' -0.15550394','-0.00449134',' 0',' 0 ','19 '),
( 407,' 408',' 408',' 36',' 0','  0.01392088','  0.00164309',' 0.00026983',' 0',' 0 ','19 '),
( 408,' 409',' 409',' 35',' 1','  0.16742638','  0.15750289','-0.00255072',' 0',' 0 ','19 '),
( 409,' 410',' 410',' 34',' 2','  0.31973984','  0.30988935','-0.00518486',' 0',' 0 ','19 '),
( 410,' 411',' 411',' 33',' 3','  0.47550222','  0.46578214','-0.00395792',' 0',' 0 ','19 '),
( 411,' 412',' 412',' 32',' 4','  0.62939674','  0.61968017','-0.00568448',' 0',' 0 ','19 '),
( 412,' 413',' 413',' 31',' 4','  0.56664014','  0.54413030',' 0.12900748',' 0',' 0 ','19 '),
( 413,' 414',' 414',' -3','-1',' 11.92894035','  4.95458800','10.85551500',' 1',' 1 ',Null),
( 414,' 415',' 415',' -2','-1',' 11.97160961',' -5.11941200','10.82651500',' 1',' 1 ',Null),
( 415,' 416',' 416',' 30',' 4','  0.53606786','  0.46775934',' 0.26380897',' 0',' 0 ','19 '),
( 416,' 417',' 417',' 29',' 4','  0.55371563','  0.38880856',' 0.39718303',' 0',' 0 ','19 '),
( 417,' 418',' 418',' 28',' 4','  0.61434534','  0.31464353',' 0.53107241',' 0',' 0 ','19 '),
( 418,' 419',' 419',' 27',' 4','  0.55602497','  0.16036779',' 0.53617567',' 0',' 0 ','19 '),
( 419,' 420',' 420',' 26',' 3','  0.45932480','  0.23485141',' 0.39816417',' 0',' 0 ','19 '),
( 420,' 421',' 421',' 25',' 3','  0.41108711','  0.31558945',' 0.26610682',' 0',' 0 ','19 '),
( 421,' 422',' 422',' 24',' 3','  0.41082825','  0.38981975',' 0.13010321',' 0',' 0 ','19 '),
( 422,' 423',' 423',' 23',' 2','  0.26527398','  0.23288419',' 0.12890313',' 0',' 0 ','19 '),
( 423,' 424',' 424',' 22',' 2','  0.30564932','  0.15893926',' 0.26447460',' 0',' 0 ','19 '),
( 424,' 425',' 425',' 21',' 3','  0.40733557','  0.08305772',' 0.40267036',' 0',' 0 ','19 '),
( 425,' 426',' 426',' 20',' 4','  0.53123859','  0.00631239',' 0.53533474',' 0',' 0 ','19 '),
( 426,' 427',' 427',' 19',' 4','  0.55291110',' -0.14933002',' 0.53685536',' 0',' 0 ','19 '),
( 427,' 428',' 428',' 18',' 3','  0.40309323',' -0.07315455',' 0.40077312',' 0',' 0 ','19 '),
( 428,' 429',' 429',' 17',' 2','  0.26361141','  0.00502249',' 0.26768699',' 0',' 0 ','19 '),
( 429,' 430',' 430',' 16',' 0','  0.14844250','  0.07657841',' 0.13056913',' 0',' 0 ','19 '),
( 430,' 431',' 431',' 15',' 0','  0.14623882',' -0.07504812',' 0.13038691',' 0',' 0 ','19 '),
( 431,' 432',' 432',' 14',' 2','  0.30217278',' -0.14878485',' 0.26784320',' 0',' 0 ','19 '),
( 432,' 433',' 433',' 13',' 3','  0.45583758',' -0.23032710',' 0.39823065',' 0',' 0 ','19 '),
( 433,' 434',' 434',' 12',' 4','  0.60924171',' -0.30428707',' 0.53266498',' 0',' 0 ','19 '),
( 434,' 435',' 435',' 11',' 4','  0.55236784',' -0.38451163',' 0.40189469',' 0',' 0 ','19 '),
( 435,' 436',' 436',' 10',' 3','  0.40233664',' -0.30826491',' 0.26415137',' 0',' 0 ','19 '),
( 436,' 437',' 437','  9',' 2','  0.26070492',' -0.22968288',' 0.12974337',' 0',' 0 ','19 '),
( 437,' 438',' 438','  8',' 3','  0.40357252',' -0.38357367',' 0.13330005',' 0',' 0 ','19 '),
( 438,' 439',' 439','  7',' 4','  0.53126567',' -0.46174820',' 0.26902895',' 0',' 0 ','19 '),
( 439,' 440',' 440','  6',' 4','  0.55412020',' -0.54012974',' 0.13310798',' 0',' 0 ','19 '),
( 440,' 441',' 441','  5',' 4','  0.61670604',' -0.61791356','-0.00041648',' 0',' 0 ','19 '),
( 441,' 442',' 442','  4',' 4','  0.56082064',' -0.54396073','-0.13711998',' 0',' 0 ','19 '),
( 442,' 443',' 443','  3',' 4','  0.54295081',' -0.47016481','-0.26951166',' 0',' 0 ','19 '),
( 443,' 444',' 444','  2',' 4','  0.56527328',' -0.39087601','-0.40536904',' 0',' 0 ','19 '),
( 444,' 445',' 445','  1',' 4','  0.62724340',' -0.31429361','-0.53937772',' 0',' 0 ','19 '),
( 445,' 446',' 446',' -1','-1',' 16.28726305','-12.17741200','10.82151500',' 1',' 1 ',Null),
( 446,' 447',' 447',' -2','-1',' 11.97610627',' -5.12164090','10.81934100',' 1',' 1 ',Null),
( 447,' 448',' 448',' 37',' 3','  0.44377104',' -0.22817112','-0.39401992',' 0',' 0 ','14 '),
( 448,' 449',' 449',' 36',' 2','  0.39086043',' -0.30655867','-0.26013572',' 0',' 0 ','14 '),
( 449,' 450',' 450',' 35',' 2','  0.39502676',' -0.38327430','-0.12861799',' 0',' 0 ','14 '),
( 450,' 451',' 451',' 34',' 2','  0.29756924',' -0.30380113',' 0.00597535',' 0',' 0 ','14 '),
( 451,' 452',' 452',' 33',' 2','  0.25177939',' -0.22893347','-0.12772215',' 0',' 0 ','14 '),
( 452,' 453',' 453',' 32',' 2','  0.29064206',' -0.15149560','-0.26147009',' 0',' 0 ','14 '),
( 453,' 454',' 454',' 31',' 2','  0.39446366',' -0.07424521','-0.39812791',' 0',' 0 ','14 '),
( 454,' 455',' 455',' 30',' 2','  0.39832084','  0.07951957','-0.39839356',' 0',' 0 ','14 '),
( 455,' 456',' 456',' 29',' 2','  0.25511989','  0.00247831','-0.26445870',' 0',' 0 ','14 '),
( 456,' 457',' 457',' 28',' 0','  0.13595245',' -0.07625905','-0.12627281',' 0',' 0 ','14 '),
( 457,' 458',' 458',' 27',' 0','  0.14386915',' -0.14968008',' 0.00587167',' 0',' 0 ','14 '),
( 458,' 459',' 459',' 26',' 2','  0.26503586',' -0.22496323',' 0.14075362',' 0',' 0 ','14 '),
( 459,' 460',' 460',' 25',' 2','  0.31886277',' -0.14788947',' 0.27636643',' 0',' 0 ','14 '),
( 460,' 461',' 461',' 24',' 1','  0.16382529',' -0.07075183',' 0.14125551',' 0',' 0 ','14 '),
( 461,' 462',' 462',' 23',' 0','  0.01899364','  0.00427761',' 0.00604447',' 0',' 0 ','14 '),
( 462,' 463',' 463',' 22',' 0','  0.14975018','  0.08114228','-0.13082897',' 0',' 0 ','14 '),
( 463,' 464',' 464',' 21',' 2','  0.30186164','  0.15634272','-0.26358591',' 0',' 0 ','14 '),
( 464,' 465',' 465',' 20',' 3','  0.45517107','  0.23318740','-0.39636863',' 0',' 0 ','14 '),
( 465,' 466',' 466',' 19',' 3','  0.40947749','  0.31266937','-0.26585442',' 0',' 0 ','14 '),
( 466,' 467',' 467',' 18',' 2','  0.27087539','  0.23566243','-0.13060325',' 0',' 0 ','14 '),
( 467,' 468',' 468',' 17',' 1','  0.16230559','  0.15505500',' 0.00462509',' 0',' 0 ','14 '),
( 468,' 469',' 469',' 16',' 1','  0.17264046','  0.08263772',' 0.13826567',' 0',' 0 ','14 '),
( 469,' 470',' 470',' 15',' 2','  0.28451022','  0.00580024',' 0.27473673',' 0',' 0 ','14 '),
( 470,' 471',' 471',' 14',' 2','  0.32753725','  0.16004017',' 0.27245610',' 0',' 0 ','14 '),
( 471,' 472',' 472',' 13',' 2','  0.28593971','  0.23692981',' 0.14029114',' 0',' 0 ','14 '),
( 472,' 473',' 473',' 12',' 2','  0.31804083','  0.31118693',' 0.00230582',' 0',' 0 ','14 '),
( 473,' 474',' 474',' 11',' 3','  0.41473375','  0.38954431','-0.13216504',' 0',' 0 ','14 '),
( 474,' 475',' 475',' 10',' 3','  0.47210139','  0.46513360',' 0.00823099',' 0',' 0 ','14 '),
( 475,' 476',' 476','  9',' 3','  0.42463658','  0.39068309',' 0.14034778',' 0',' 0 ','14 '),
( 476,' 477',' 477','  8',' 3','  0.42819790','  0.31374978',' 0.27458862',' 0',' 0 ','14 '),
( 477,' 478',' 478','  7',' 3','  0.47949386','  0.23318769',' 0.40570870',' 0',' 0 ','14 '),
( 478,' 479',' 479','  6',' 3','  0.42767782','  0.08252456',' 0.40877933',' 0',' 0 ','14 '),
( 479,' 480',' 480','  5',' 3','  0.42223640',' -0.07183709',' 0.40767007',' 0',' 0 ','14 '),
( 480,' 481',' 481','  4',' 3','  0.47348298',' -0.22838805',' 0.40884221',' 0',' 0 ','14 '),
( 481,' 482',' 482','  3',' 3','  0.41165366',' -0.30506278',' 0.27404623',' 0',' 0 ','14 '),
( 482,' 483',' 483','  2',' 3','  0.40199712',' -0.37911534',' 0.14169302',' 0',' 0 ','14 '),
( 483,' 484',' 484','  1',' 3','  0.44920667',' -0.45558164',' 0.00577034',' 0',' 0 ','14 '),
( 484,' 485',' 485',' -1','-1',' 16.29641733','-12.19464100','10.80834100',' 1',' 1 ',Null),
( 485,' 486',' 486',' -3','-1',' 37.19824439',' 29.87476500','22.17078100',' 1',' 1 ',Null),
( 486,' 487',' 487',' -2','-1',' 29.70673021',' 19.80376500','22.14978100',' 1',' 1 ',Null),
( 487,' 488',' 488',' 21',' 1','  0.16040362','  0.16141056','-0.00102561',' 0',' 2 ','2 '),
( 488,' 489',' 489',' 20',' 1','  0.15068315','  0.08120214',' 0.13367454',' 0',' 2 ','2 '),
( 489,' 490',' 490',' 19',' 0','  0.14733356',' -0.07312835',' 0.13324056',' 0',' 2 ','2 '),
( 490,' 491',' 491',' 18',' 0','  0.00759348','  0.00501764','-0.00053376',' 0',' 2 ','2 '),
( 491,' 492',' 492',' 17',' 1','  0.16434238','  0.08485289','-0.13542837',' 0',' 2 ','2 '),
( 492,' 493',' 493',' 16',' 1','  0.15890020',' -0.07136642','-0.13537497',' 0',' 2 ','2 '),
( 493,' 494',' 494',' 15',' 1','  0.15095792',' -0.14959487','-0.00179619',' 0',' 2 ','2 '),
( 494,' 495',' 495',' 14',' 0','  0.13590065','  0.13698401',' 0.00140837',' 0',' 2 ','1 '),
( 495,' 496',' 496',' 13',' 0','  0.13812804','  0.05525512',' 0.13310280',' 0',' 2 ','1 '),
( 496,' 497',' 497',' 12',' 1','  0.15766272',' -0.10100368',' 0.12609031',' 0',' 2 ','1 '),
( 497,' 498',' 498',' 11',' 0','  0.02300171',' -0.01899821','-0.00506840',' 0',' 2 ','1 '),
( 498,' 499',' 499',' 10',' 1','  0.15475997','  0.06354893','-0.13562020',' 0',' 2 ','1 '),
( 499,' 500',' 500','  9',' 1','  0.17431284',' -0.09256124','-0.14096548',' 0',' 2 ','1 '),
( 500,' 501',' 501','  8',' 1','  0.17523296',' -0.17337820','-0.00956769',' 0',' 2 ','1 '),
( 501,' 502',' 502','  7',' 0','  0.14448851','  0.14545775',' 0.01345305',' 0',' 2 ','0 '),
( 502,' 503',' 503','  6',' 1','  0.15394912','  0.06502838',' 0.14608365',' 0',' 2 ','0 '),
( 503,' 504',' 504','  5',' 1','  0.16414136',' -0.09094143',' 0.14187326',' 0',' 2 ','0 '),
( 504,' 505',' 505','  4',' 0','  0.01101190',' -0.00955584',' 0.00853875',' 0',' 2 ','0 '),
( 505,' 506',' 506','  3',' 0','  0.14775015','  0.07252791','-0.12336362',' 0',' 2 ','0 '),
( 506,' 507',' 507','  2',' 1','  0.15773605',' -0.08400097','-0.12676317',' 0',' 2 ','0 '),
( 507,' 508',' 508','  1',' 1','  0.16641667',' -0.16525439',' 0.00540788',' 0',' 2 ','0 '),
( 508,' 509',' 509',' -1','-1',' 25.54319802',' 12.74376500','22.14378100',' 1',' 1 ',Null),
( 509,' 510',' 510',' -8','-1',' 13.15449568',' 12.07619800','-5.20505380',' 1',' 1 ',Null),
( 510,' 511',' 511','127',' 6','  0.86174611',' -0.31799926','-0.79661132',' 0',' 0 ','24 '),
( 511,' 512',' 512','126',' 5','  0.77227741',' -0.39604425','-0.65916412',' 0',' 0 ','24 '),
( 512,' 513',' 513','125',' 5','  0.71015466',' -0.24213628','-0.66320086',' 0',' 0 ','24 '),
( 513,' 514',' 514','124',' 5','  0.67637836',' -0.08968006','-0.66545211',' 0',' 0 ','24 '),
( 514,' 515',' 515','123',' 6','  0.82240918',' -0.16359951','-0.80118692',' 0',' 0 ','24 '),
( 515,' 516',' 516','122',' 6','  0.80730630',' -0.00975662','-0.80199799',' 0',' 0 ','24 '),
( 516,' 517',' 517','121',' 6','  0.81920301','  0.14406849','-0.80072250',' 0',' 0 ','24 '),
( 517,' 518',' 518','120',' 6','  0.86109843','  0.30110759','-0.80055126',' 0',' 0 ','24 '),
( 518,' 519',' 519','119',' 6','  0.92646057','  0.45507765','-0.80034077',' 0',' 0 ','24 '),
( 519,' 520',' 520','118',' 6','  0.86255877','  0.53483369','-0.66952599',' 0',' 0 ','24 '),
( 520,' 521',' 521','117',' 5','  0.77238019','  0.37742968','-0.66724153',' 0',' 0 ','24 '),
( 521,' 522',' 522','116',' 5','  0.70900269','  0.22068707','-0.66770640',' 0',' 0 ','24 '),
( 522,' 523',' 523','115',' 5','  0.67589475','  0.06398334','-0.66734899',' 0',' 0 ','24 '),
( 523,' 524',' 524','114',' 4','  0.54018114',' -0.00911338','-0.53486472',' 0',' 0 ','24 '),
( 524,' 525',' 525','113',' 4','  0.56040205',' -0.16550625','-0.53087442',' 0',' 0 ','24 '),
( 525,' 526',' 526','112',' 4','  0.61804190',' -0.32017378','-0.52483251',' 0',' 0 ','24 '),
( 526,' 527',' 527','111',' 5','  0.70711202',' -0.47417326','-0.52147416',' 0',' 0 ','24 '),
( 527,' 528',' 528','110',' 5','  0.67328525',' -0.55176839','-0.38400536',' 0',' 0 ','24 '),
( 528,' 529',' 529','109',' 4','  0.55601043',' -0.39512883','-0.38834205',' 0',' 0 ','24 '),
( 529,' 530',' 530','108',' 3','  0.46509383',' -0.24084122','-0.39406424',' 0',' 0 ','24 '),
( 530,' 531',' 531','107',' 3','  0.41042359',' -0.08730840','-0.39627569',' 0',' 0 ','24 '),
( 531,' 532',' 532','106',' 2','  0.30934609',' -0.16464670','-0.25812656',' 0',' 0 ','24 '),
( 532,' 533',' 533','105',' 3','  0.41088078',' -0.32176306','-0.25327622',' 0',' 0 ','24 '),
( 533,' 534',' 534','104',' 4','  0.53562636',' -0.47238336','-0.25170447',' 0',' 0 ','24 '),
( 534,' 535',' 535','103',' 5','  0.67226174',' -0.62482089','-0.24881972',' 0',' 0 ','24 '),
( 535,' 536',' 536','102',' 5','  0.70679206',' -0.69941988','-0.11202241',' 0',' 0 ','24 '),
( 536,' 537',' 537','101',' 4','  0.55573111',' -0.54516690','-0.11419301',' 0',' 0 ','24 '),
( 537,' 538',' 538','100',' 3','  0.40788673',' -0.39138108','-0.11755755',' 0',' 0 ','24 '),
( 538,' 539',' 539',' 99',' 2','  0.26688383',' -0.23774018','-0.12064404',' 0',' 0 ','24 '),
( 539,' 540',' 540',' 98',' 1','  0.15388325',' -0.15621864',' 0.01064108',' 0',' 0 ','24 '),
( 540,' 541',' 541',' 97',' 1','  0.15785048',' -0.08647750','-0.12833864',' 0',' 0 ','24 '),
( 541,' 542',' 542',' 96',' 2','  0.27123369',' -0.00793871','-0.26590266',' 0',' 0 ','24 '),
( 542,' 543',' 543',' 95',' 3','  0.40963803','  0.06509469','-0.39875943',' 0',' 0 ','24 '),
( 543,' 544',' 544',' 94',' 4','  0.55731389','  0.14190969','-0.53302330',' 0',' 0 ','24 '),
( 544,' 545',' 545',' 93',' 4','  0.61662463','  0.29711736','-0.53370354',' 0',' 0 ','24 '),
( 545,' 546',' 546',' 92',' 5','  0.71139843','  0.45854964','-0.53656111',' 0',' 0 ','24 '),
( 546,' 547',' 547',' 91',' 6','  0.81877126','  0.61074294','-0.53731740',' 0',' 0 ','24 '),
( 547,' 548',' 548',' -7','-1','  7.24383734','  5.00819750','-5.22605380',' 1',' 1 ',Null),
( 548,' 549',' 549',' -6','-1','  7.28872792',' -5.06480250','-5.25005380',' 1',' 1 ',Null),
( 549,' 550',' 550',' 90',' 6','  0.81897091','  0.69186813','-0.40465524',' 0',' 0 ','24 '),
( 550,' 551',' 551',' 89',' 5','  0.68947869','  0.53746800','-0.40399720',' 0',' 0 ','24 '),
( 551,' 552',' 552',' 88',' 4','  0.57071992','  0.38070546','-0.40318530',' 0',' 0 ','24 '),
( 552,' 553',' 553',' 87',' 3','  0.47213212','  0.22213549','-0.40061249',' 0',' 0 ','24 '),
( 553,' 554',' 554',' 86',' 2','  0.32051219','  0.14656538','-0.26916084',' 0',' 0 ','24 '),
( 554,' 555',' 555',' 85',' 1','  0.16353127','  0.07142871','-0.13112745',' 0',' 0 ','24 '),
( 555,' 556',' 556',' 84',' 0','  0.01517504',' -0.00122442',' 0.00414961',' 0',' 0 ','24 '),
( 556,' 557',' 557',' 83',' 0','  0.14615104',' -0.07309044',' 0.14158221',' 0',' 0 ','24 '),
( 557,' 558',' 558',' 82',' 1','  0.16361469','  0.08080118',' 0.13879667',' 0',' 0 ','24 '),
( 558,' 559',' 559',' 81',' 1','  0.16899282','  0.15275325',' 0.00095315',' 0',' 0 ','24 '),
( 559,' 560',' 560',' 80',' 2','  0.28120464','  0.22734446','-0.13370884',' 0',' 0 ','24 '),
( 560,' 561',' 561',' 79',' 3','  0.42272871','  0.30351514','-0.26964511',' 0',' 0 ','24 '),
( 561,' 562',' 562',' 78',' 4','  0.54855522','  0.45695685','-0.27068166',' 0',' 0 ','24 '),
( 562,' 563',' 563',' 77',' 3','  0.42315984','  0.38191806','-0.13661145',' 0',' 0 ','24 '),
( 563,' 564',' 564',' 76',' 2','  0.32340953','  0.30719341','-0.00039658',' 0',' 0 ','24 '),
( 564,' 565',' 565',' 75',' 2','  0.28165376','  0.23380197',' 0.13684064',' 0',' 0 ','24 '),
( 565,' 566',' 566',' 74',' 3','  0.42451217','  0.38781252',' 0.13752276',' 0',' 0 ','24 '),
( 566,' 567',' 567',' 73',' 3','  0.47986105','  0.46368584',' 0.00053433',' 0',' 0 ','24 '),
( 567,' 568',' 568',' 72',' 4','  0.57233475','  0.53849526','-0.13427815',' 0',' 0 ','24 '),
( 568,' 569',' 569',' 71',' 5','  0.69059345','  0.61621677','-0.27059995',' 0',' 0 ','24 '),
( 569,' 570',' 570',' 70',' 6','  0.83556828','  0.77069701','-0.27421522',' 0',' 0 ','24 '),
( 570,' 571',' 571',' 69',' 5','  0.72351792','  0.69260062','-0.13853441',' 0',' 0 ','24 '),
( 571,' 572',' 572',' 68',' 4','  0.63170444','  0.61551058','-0.00186639',' 0',' 0 ','24 '),
( 572,' 573',' 573',' 67',' 5','  0.72681780','  0.69979592',' 0.13233301',' 0',' 0 ','24 '),
( 573,' 574',' 574',' 66',' 5','  0.78804538','  0.77181615','-0.00540304',' 0',' 0 ','24 '),
( 574,' 575',' 575',' 65',' 6','  0.87763225','  0.84912239','-0.13989216',' 0',' 0 ','24 '),
( 575,' 576',' 576',' 64',' 6','  0.94212794','  0.92590568','-0.00606724',' 0',' 0 ','24 '),
( 576,' 577',' 577',' 63',' 6','  0.87768173','  0.85271935',' 0.13117593',' 0',' 0 ','24 '),
( 577,' 578',' 578',' 62',' 6','  0.83803757','  0.78048989',' 0.26718093',' 0',' 0 ','24 '),
( 578,' 579',' 579',' 61',' 6','  0.82255226','  0.70558957',' 0.40158875',' 0',' 0 ','24 '),
( 579,' 580',' 580',' -5','-1',' 13.20385042','-12.12180200','-5.26505380',' 1',' 1 ',Null),
( 580,' 581',' 581',' -4','-1',' 16.20230710',' 12.01919800','10.84794600',' 1',' 1 ',Null),
( 581,' 582',' 582',' 60',' 6','  0.83983691','  0.62985202',' 0.53782841',' 0',' 0 ','24 '),
( 582,' 583',' 583',' 59',' 5','  0.69211749','  0.54920370',' 0.40189836',' 0',' 0 ','24 '),
( 583,' 584',' 584',' 58',' 5','  0.69123041','  0.62343881',' 0.27184498',' 0',' 0 ','24 '),
( 584,' 585',' 585',' 57',' 4','  0.57191845','  0.54539882',' 0.13367258',' 0',' 0 ','24 '),
( 585,' 586',' 586',' 56',' 4','  0.55162504','  0.46777325',' 0.27022527',' 0',' 0 ','24 '),
( 586,' 587',' 587',' 55',' 3','  0.43023012','  0.31808029',' 0.27216607',' 0',' 0 ','24 '),
( 587,' 588',' 588',' 54',' 2','  0.32840832','  0.16211051',' 0.27287551',' 0',' 0 ','24 '),
( 588,' 589',' 589',' 53',' 2','  0.28307888','  0.00293511',' 0.27528535',' 0',' 0 ','24 '),
( 589,' 590',' 590',' 52',' 2','  0.31825638',' -0.14954585',' 0.27789798',' 0',' 0 ','24 '),
( 590,' 591',' 591',' 51',' 3','  0.42440598',' -0.07280208',' 0.41200402',' 0',' 0 ','24 '),
( 591,' 592',' 592',' 50',' 3','  0.42535329','  0.08421279',' 0.40750715',' 0',' 0 ','24 '),
( 592,' 593',' 593',' 49',' 3','  0.48275230','  0.23840607',' 0.40709971',' 0',' 0 ','24 '),
( 593,' 594',' 594',' 48',' 4','  0.57364077','  0.39261467',' 0.40223344',' 0',' 0 ','24 '),
( 594,' 595',' 595',' 47',' 5','  0.72879705','  0.47343026',' 0.53888399',' 0',' 0 ','24 '),
( 595,' 596',' 596',' 46',' 4','  0.63627439','  0.31710955',' 0.53891322',' 0',' 0 ','24 '),
( 596,' 597',' 597',' 45',' 4','  0.57689381','  0.16023376',' 0.54402144',' 0',' 0 ','24 '),
( 597,' 598',' 598',' 44',' 4','  0.55301765','  0.00827900',' 0.54520378',' 0',' 0 ','24 '),
( 598,' 599',' 599',' 43',' 4','  0.57337430',' -0.14585604',' 0.54920542',' 0',' 0 ','24 '),
( 599,' 600',' 600',' 42',' 5','  0.72450259',' -0.22288358',' 0.68460361',' 0',' 0 ','24 '),
( 600,' 601',' 601',' 41',' 5','  0.68760258',' -0.06910624',' 0.67740395',' 0',' 0 ','24 '),
( 601,' 602',' 602',' 40',' 5','  0.69346354','  0.08665361',' 0.67931152',' 0',' 0 ','24 '),
( 602,' 603',' 603',' 39',' 5','  0.72583156','  0.24137415',' 0.67380148',' 0',' 0 ','24 '),
( 603,' 604',' 604',' 38',' 5','  0.79104962','  0.39492861',' 0.67271117',' 0',' 0 ','24 '),
( 604,' 605',' 605',' 37',' 6','  0.87931544','  0.55257815',' 0.66923633',' 0',' 0 ','24 '),
( 605,' 606',' 606',' 36',' 6','  0.94773754','  0.47719929',' 0.80608515',' 0',' 0 ','24 '),
( 606,' 607',' 607',' 35',' 6','  0.87791684','  0.32025263',' 0.80636423',' 0',' 0 ','24 '),
( 607,' 608',' 608',' 34',' 6','  0.83241259','  0.16398199',' 0.80673136',' 0',' 0 ','24 '),
( 608,' 609',' 609',' 33',' 6','  0.82121052','  0.00998494',' 0.81344564',' 0',' 0 ','24 '),
( 609,' 610',' 610',' 32',' 6','  0.83461938',' -0.14399264',' 0.81604906',' 0',' 0 ','24 '),
( 610,' 611',' 611',' 31',' 6','  0.87562446',' -0.29874088',' 0.81868144',' 0',' 0 ','24 '),
( 611,' 612',' 612',' -3','-1',' 11.92644474','  4.94819750','10.83994600',' 1',' 1 ',Null),
( 612,' 613',' 613',' -2','-1',' 11.96645972',' -5.12080250','10.81294600',' 1',' 1 ',Null),
( 613,' 614',' 614',' 30',' 6','  0.93913633',' -0.45234865',' 0.81993841',' 0',' 0 ','24 '),
( 614,' 615',' 615',' 29',' 6','  0.87324906',' -0.53152945',' 0.68816522',' 0',' 0 ','24 '),
( 615,' 616',' 616',' 28',' 5','  0.78271058',' -0.37488729',' 0.68403389',' 0',' 0 ','24 '),
( 616,' 617',' 617',' 27',' 4','  0.63098349',' -0.30246138',' 0.55069190',' 0',' 0 ','24 '),
( 617,' 618',' 618',' 26',' 5','  0.72152191',' -0.45637625',' 0.55378755',' 0',' 0 ','24 '),
( 618,' 619',' 619',' 25',' 6','  0.83343128',' -0.61393709',' 0.55656357',' 0',' 0 ','24 '),
( 619,' 620',' 620',' 24',' 6','  0.81842451',' -0.69263489',' 0.42514368',' 0',' 0 ','24 '),
( 620,' 621',' 621',' 23',' 5','  0.68640125',' -0.53543103',' 0.42120827',' 0',' 0 ','24 '),
( 621,' 622',' 622',' 22',' 4','  0.57009999',' -0.38158831',' 0.41784822',' 0',' 0 ','24 '),
( 622,' 623',' 623',' 21',' 3','  0.47369713',' -0.22657086',' 0.41291418',' 0',' 0 ','24 '),
( 623,' 624',' 624',' 20',' 3','  0.42033535',' -0.30615062',' 0.28103533',' 0',' 0 ','24 '),
( 624,' 625',' 625',' 19',' 4','  0.54824908',' -0.46072772',' 0.28651532',' 0',' 0 ','24 '),
( 625,' 626',' 626',' 18',' 5','  0.68802841',' -0.61684442',' 0.29051934',' 0',' 0 ','24 '),
( 626,' 627',' 627',' 17',' 6','  0.83136930',' -0.77114307',' 0.29288819',' 0',' 0 ','24 '),
( 627,' 628',' 628',' 16',' 6','  0.87220693',' -0.85047695',' 0.15902063',' 0',' 0 ','24 '),
( 628,' 629',' 629',' 15',' 5','  0.71992176',' -0.69540501',' 0.15751531',' 0',' 0 ','24 '),
( 629,' 630',' 630',' 14',' 4','  0.56962066',' -0.54125266',' 0.15448929',' 0',' 0 ','24 '),
( 630,' 631',' 631',' 13',' 3','  0.42160654',' -0.38712160',' 0.14993725',' 0',' 0 ','24 '),
( 631,' 632',' 632',' 12',' 2','  0.27860275',' -0.23124075',' 0.14492277',' 0',' 0 ','24 '),
( 632,' 633',' 633',' 11',' 2','  0.32017190',' -0.31255826',' 0.01535143',' 0',' 0 ','24 '),
( 633,' 634',' 634',' 10',' 3','  0.47383707',' -0.46622946',' 0.01830303',' 0',' 0 ','24 '),
( 634,' 635',' 635','  9',' 4','  0.62606444',' -0.61847861',' 0.02019818',' 0',' 0 ','24 '),
( 635,' 636',' 636','  8',' 5','  0.78161709',' -0.77404922',' 0.02179167',' 0',' 0 ','24 '),
( 636,' 637',' 637','  7',' 6','  0.93384994',' -0.92632924',' 0.02170434',' 0',' 0 ','24 '),
( 637,' 638',' 638','  6',' 6','  0.87159316',' -0.85664206','-0.11433451',' 0',' 0 ','24 '),
( 638,' 639',' 639','  5',' 6','  0.82484717',' -0.77869384','-0.24923269',' 0',' 0 ','24 '),
( 639,' 640',' 640','  4',' 6','  0.81264899',' -0.70862885','-0.38355271',' 0',' 0 ','24 '),
( 640,' 641',' 641','  3',' 6','  0.82527022',' -0.63340745','-0.51920004',' 0',' 0 ','24 '),
( 641,' 642',' 642','  2',' 6','  0.86256896',' -0.55026338','-0.65717787',' 0',' 0 ','24 '),
( 642,' 643',' 643','  1',' 6','  0.92679453',' -0.47232378','-0.79207345',' 0',' 0 ','24 '),
( 643,' 644',' 644',' -1','-1',' 16.28869171','-12.18180200','10.80594600',' 1',' 1 ',Null),
( 644,' 645',' 645',' -4','-1',' 16.19490663',' 12.02573400','10.85583700',' 1',' 1 ',Null),
( 645,' 646',' 646',' 61',' 4','  0.57877048',' -0.15622040','-0.55435057',' 0',' 0 ','18 '),
( 646,' 647',' 647',' 60',' 4','  0.55226113',' -0.00278116','-0.55130926',' 0',' 0 ','18 '),
( 647,' 648',' 648',' 59',' 4','  0.56639141','  0.15174074','-0.54678391',' 0',' 0 ','18 '),
( 648,' 649',' 649',' 58',' 4','  0.62326761','  0.30768081','-0.54520684',' 0',' 0 ','18 '),
( 649,' 650',' 650',' 57',' 4','  0.55619502','  0.38324161','-0.40897566',' 0',' 0 ','18 '),
( 650,' 651',' 651',' 56',' 3','  0.46828342','  0.22876271','-0.41170576',' 0',' 0 ','18 '),
( 651,' 652',' 652',' 55',' 3','  0.42084225','  0.07452375','-0.41456560',' 0',' 0 ','18 '),
( 652,' 653',' 653',' 54',' 3','  0.43035634',' -0.07984368','-0.42059488',' 0',' 0 ','18 '),
( 653,' 654',' 654',' 53',' 3','  0.48504106',' -0.23338123','-0.42029475',' 0',' 0 ','18 '),
( 654,' 655',' 655',' 52',' 3','  0.42969511',' -0.31124341','-0.28760758',' 0',' 0 ','18 '),
( 655,' 656',' 656',' 51',' 2','  0.33079661',' -0.15723952','-0.28615281',' 0',' 0 ','18 '),
( 656,' 657',' 657',' 50',' 2','  0.28167940',' -0.00347434','-0.28061486',' 0',' 0 ','18 '),
( 657,' 658',' 658',' 49',' 2','  0.31412803','  0.15075698','-0.27855909',' 0',' 0 ','18 '),
( 658,' 659',' 659',' 48',' 3','  0.40571545','  0.30306569','-0.27678022',' 0',' 0 ','18 '),
( 659,' 660',' 660',' 47',' 4','  0.53253000','  0.45975384','-0.27986652',' 0',' 0 ','18 '),
( 660,' 661',' 661',' 46',' 4','  0.54491059','  0.53321907','-0.14147764',' 0',' 0 ','18 '),
( 661,' 662',' 662',' 45',' 3','  0.40066719','  0.38173764','-0.14153368',' 0',' 0 ','18 '),
( 662,' 663',' 663',' 44',' 2','  0.26126165','  0.22596295','-0.14201085',' 0',' 0 ','18 '),
( 663,' 664',' 664',' 43',' 1','  0.15688528','  0.07112662','-0.14242136',' 0',' 0 ','18 '),
( 664,' 665',' 665',' 42',' 1','  0.17156971',' -0.08236103','-0.14546066',' 0',' 0 ','18 '),
( 665,' 666',' 666',' 41',' 2','  0.28702366',' -0.23665252','-0.15048333',' 0',' 0 ','18 '),
( 666,' 667',' 667',' 40',' 3','  0.42581690',' -0.39068683','-0.15074452',' 0',' 0 ','18 '),
( 667,' 668',' 668',' 39',' 3','  0.47594954',' -0.46837800','-0.01723437',' 0',' 0 ','18 '),
( 668,' 669',' 669',' 38',' 2','  0.32285243',' -0.31514282','-0.01677748',' 0',' 0 ','18 '),
( 669,' 670',' 670',' 37',' 1','  0.17088930',' -0.16326615','-0.01076194',' 0',' 0 ','18 '),
( 670,' 671',' 671',' 36',' 0','  0.01730734',' -0.00663465','-0.00950162',' 0',' 0 ','18 '),
( 671,' 672',' 672',' 35',' 0','  0.13812295','  0.14512102','-0.00709511',' 0',' 0 ','18 '),
( 672,' 673',' 673',' 34',' 2','  0.29227709','  0.29941635','-0.00631704',' 0',' 0 ','18 '),
( 673,' 674',' 674',' 33',' 3','  0.44851594','  0.45567304','-0.00708069',' 0',' 0 ','18 '),
( 674,' 675',' 675',' 32',' 4','  0.60006410','  0.60726164','-0.00512650',' 0',' 0 ','18 '),
( 675,' 676',' 676',' 31',' 4','  0.54127803','  0.53364233',' 0.12683435',' 0',' 0 ','18 '),
( 676,' 677',' 677',' -3','-1',' 11.92028633','  4.95273390','10.84683700',' 1',' 1 ',Null),
( 677,' 678',' 678',' -2','-1',' 16.28562989','-12.18026600','10.81683700',' 1',' 1 ',Null),
( 678,' 679',' 679',' 30',' 4','  0.52306039','  0.45608463',' 0.26239559',' 0',' 0 ','18 '),
( 679,' 680',' 680',' 29',' 4','  0.54199245','  0.37655971',' 0.39618994',' 0',' 0 ','18 '),
( 680,' 681',' 681',' 28',' 4','  0.60409353','  0.30120474',' 0.53004543',' 0',' 0 ','18 '),
( 681,' 682',' 682',' 27',' 4','  0.54364057','  0.14497852',' 0.53037424',' 0',' 0 ','18 '),
( 682,' 683',' 683',' 26',' 3','  0.44778474','  0.22129710',' 0.39568014',' 0',' 0 ','18 '),
( 683,' 684',' 684',' 25',' 2','  0.39580618','  0.30158569',' 0.26269276',' 0',' 0 ','18 '),
( 684,' 685',' 685',' 24',' 2','  0.39858021','  0.37947584',' 0.12813323',' 0',' 0 ','18 '),
( 685,' 686',' 686',' 23',' 2','  0.25473117','  0.22458901',' 0.12650396',' 0',' 0 ','18 '),
( 686,' 687',' 687',' 22',' 2','  0.29338294','  0.14704823',' 0.26027031',' 0',' 0 ','18 '),
( 687,' 688',' 688',' 21',' 2','  0.39470900','  0.06732854',' 0.39535348',' 0',' 0 ','18 '),
( 688,' 689',' 689',' 20',' 4','  0.52279152',' -0.01270976',' 0.52908068',' 0',' 0 ','18 '),
( 689,' 690',' 690',' 19',' 4','  0.54680265',' -0.16375591',' 0.52817084',' 0',' 0 ','18 '),
( 690,' 691',' 691',' 18',' 2','  0.39679795',' -0.08562818',' 0.39390667',' 0',' 0 ','18 '),
( 691,' 692',' 692',' 17',' 2','  0.25449915',' -0.00985906',' 0.26075284',' 0',' 0 ','18 '),
( 692,' 693',' 693',' 16',' 0','  0.13902501','  0.06796157',' 0.12768231',' 0',' 0 ','18 '),
( 693,' 694',' 694',' 15',' 0','  0.14730327',' -0.08586150',' 0.12618582',' 0',' 0 ','18 '),
( 694,' 695',' 695',' 14',' 2','  0.29862627',' -0.16113648',' 0.25791007',' 0',' 0 ','18 '),
( 695,' 696',' 696',' 13',' 3','  0.45409113',' -0.23971008',' 0.39215277',' 0',' 0 ','18 '),
( 696,' 697',' 697',' 12',' 4','  0.60878948',' -0.32003633',' 0.52436870',' 0',' 0 ','18 '),
( 697,' 698',' 698',' 11',' 4','  0.55148718',' -0.39714358',' 0.38916002',' 0',' 0 ','18 '),
( 698,' 699',' 699',' 10',' 3','  0.40435772',' -0.31792406',' 0.25639378',' 0',' 0 ','18 '),
( 699,' 700',' 700','  9',' 2','  0.26723188',' -0.24036683',' 0.12336840',' 0',' 0 ','18 '),
( 700,' 701',' 701','  8',' 3','  0.41053414',' -0.39480329',' 0.11925310',' 0',' 0 ','18 '),
( 701,' 702',' 702','  7',' 4','  0.53094293',' -0.47116753',' 0.25133054',' 0',' 0 ','18 '),
( 702,' 703',' 703','  6',' 4','  0.55949380',' -0.54836654',' 0.11783142',' 0',' 0 ','18 '),
( 703,' 704',' 704','  5',' 4','  0.62314044',' -0.62270401','-0.01875280',' 0',' 0 ','18 '),
( 704,' 705',' 705','  4',' 4','  0.56872909',' -0.54577860','-0.15374017',' 0',' 0 ','18 '),
( 705,' 706',' 706','  3',' 4','  0.55159411',' -0.46759064','-0.28627540',' 0',' 0 ','18 '),
( 706,' 707',' 707','  2',' 4','  0.58126443',' -0.39344995','-0.42148683',' 0',' 0 ','18 '),
( 707,' 708',' 708','  1',' 4','  0.64052806',' -0.31091474','-0.55360602',' 0',' 0 ','18 '),
( 708,' 709',' 709',' -1','-1',' 11.96155334',' -5.12426610','10.81483700',' 1',' 1 ',Null),
( 709,' 710','   1',' -2','-1',' 11.96631094',' -5.11699770','10.82229100',' 1',' 1 ',Null),
( 710,' 711','   2',' 37',' 3','  0.46498356',' -0.22958639','-0.39581705',' 0',' 0 ','17 '),
( 711,' 712','   3',' 36',' 3','  0.41208759',' -0.30695604','-0.26464927',' 0',' 0 ','17 '),
( 712,' 713','   4',' 35',' 3','  0.40722957',' -0.38038001','-0.13013720',' 0',' 0 ','17 '),
( 713,' 714','   5',' 34',' 2','  0.30905704',' -0.30589037',' 0.00811213',' 0',' 0 ','17 '),
( 714,' 715','   6',' 33',' 2','  0.26618892',' -0.22828924','-0.12475543',' 0',' 0 ','17 '),
( 715,' 716','   7',' 32',' 2','  0.30741337',' -0.15084726','-0.25933060',' 0',' 0 ','17 '),
( 716,' 717','   8',' 31',' 3','  0.41035708',' -0.07549522','-0.39602636',' 0',' 0 ','17 '),
( 717,' 718','   9',' 30',' 3','  0.41072698','  0.08105349','-0.39655249',' 0',' 0 ','17 '),
( 718,' 719','  10',' 29',' 2','  0.26770573','  0.00265013','-0.26098386',' 0',' 0 ','17 '),
( 719,' 720','  11',' 28',' 1','  0.15588474',' -0.07523436','-0.12801474',' 0',' 0 ','17 '),
( 720,' 721','  12',' 27',' 1','  0.15492625',' -0.15175715',' 0.00803325',' 0',' 0 ','17 '),
( 721,' 722','  13',' 26',' 2','  0.27182859',' -0.23242761',' 0.14232223',' 0',' 0 ','17 '),
( 722,' 723','  14',' 25',' 2','  0.31042448',' -0.15088909',' 0.27622303',' 0',' 0 ','17 '),
( 723,' 724','  15',' 24',' 1','  0.15729014',' -0.07567385',' 0.14282741',' 0',' 0 ','17 '),
( 724,' 725','  16',' 23',' 0','  0.00212577','  0.00116120',' 0.00600755',' 0',' 0 ','17 '),
( 725,' 726','  17',' 22',' 1','  0.15335534','  0.07969716','-0.12617131',' 0',' 0 ','17 '),
( 726,' 727','  18',' 21',' 2','  0.30836895','  0.15763606','-0.26016746',' 0',' 0 ','17 '),
( 727,' 728','  19',' 20',' 3','  0.46230895','  0.23516100','-0.39316204',' 0',' 0 ','17 '),
( 728,' 729','  20',' 19',' 3','  0.40777613','  0.31302596','-0.25835721',' 0',' 0 ','17 '),
( 729,' 730','  21',' 18',' 2','  0.26629315','  0.23398146','-0.12607595',' 0',' 0 ','17 '),
( 730,' 731','  22',' 17',' 1','  0.15252261','  0.15566128',' 0.00396728',' 0',' 0 ','17 '),
( 731,' 732','  23',' 16',' 1','  0.15461558','  0.07772023',' 0.14217349',' 0',' 0 ','17 '),
( 732,' 733','  24',' 15',' 2','  0.26987432','  0.00130462',' 0.27658929',' 0',' 0 ','17 '),
( 733,' 734','  25',' 14',' 2','  0.31102874','  0.15657360',' 0.27728418',' 0',' 0 ','17 '),
( 734,' 735','  26',' 13',' 2','  0.26794991','  0.23418543',' 0.14246393',' 0',' 0 ','17 '),
( 735,' 736','  27',' 12',' 2','  0.30935097','  0.31251413',' 0.00721006',' 0',' 0 ','17 '),
( 736,' 737','  28',' 11',' 3','  0.40882542','  0.39015288','-0.12510229',' 0',' 0 ','17 '),
( 737,' 738','  29',' 10',' 3','  0.46077402','  0.46392938',' 0.00946751',' 0',' 0 ','17 '),
( 738,' 739','  30','  9',' 3','  0.41188095','  0.39119784',' 0.14483472',' 0',' 0 ','17 '),
( 739,' 740','  31','  8',' 3','  0.41063095','  0.31135958',' 0.27807535',' 0',' 0 ','17 '),
( 740,' 741','  32','  7',' 3','  0.46485739','  0.23365567',' 0.41041157',' 0',' 0 ','17 '),
( 741,' 742','  33','  6',' 3','  0.41074280','  0.07921819',' 0.41036148',' 0',' 0 ','17 '),
( 742,' 743','  34','  5',' 3','  0.41051509',' -0.07494653',' 0.40973683',' 0',' 0 ','17 '),
( 743,' 744','  35','  4',' 3','  0.46574291',' -0.22877608',' 0.41060313',' 0',' 0 ','17 '),
( 744,' 745','  36','  3',' 3','  0.41254117',' -0.30750307',' 0.27815548',' 0',' 0 ','17 '),
( 745,' 746','  37','  2',' 3','  0.41235232',' -0.38591329',' 0.14329975',' 0',' 0 ','17 '),
( 746,' 747','  38','  1',' 3','  0.46364555',' -0.46047806',' 0.00863458',' 0',' 0 ','17 '),
( 747,' 748','  39',' -1','-1',' 16.28648414','-12.18599800','10.80829100',' 1',' 1 ',Null),
( 748,' 749','  40',' -8','-1',' 13.15653092',' 12.08014600','-5.19197930',' 1',' 1 ',Null),
( 749,' 750','  41','127',' 6','  0.85713474',' -0.32599149','-0.79932724',' 0',' 0 ','28 '),
( 750,' 751','  42','126',' 5','  0.76148209',' -0.40121880','-0.65581445',' 0',' 0 ','28 '),
( 751,' 752','  43','125',' 5','  0.70351119',' -0.24759240','-0.66475185',' 0',' 0 ','28 '),
( 752,' 753','  44','124',' 5','  0.67028338',' -0.09623975','-0.66734703',' 0',' 0 ','28 '),
( 753,' 754','  45','123',' 6','  0.81518032',' -0.17185282','-0.80156673',' 0',' 0 ','28 '),
( 754,' 755','  46','122',' 5','  0.79951606',' -0.01785936','-0.80214322',' 0',' 0 ','28 '),
( 755,' 756','  47','121',' 6','  0.81558425','  0.13860453','-0.80464805',' 0',' 0 ','28 '),
( 756,' 757','  48','120',' 6','  0.85637822','  0.29073241','-0.80460331',' 0',' 0 ','28 '),
( 757,' 758','  49','119',' 6','  0.91958914','  0.44471958','-0.80211202',' 0',' 0 ','28 '),
( 758,' 759','  50','118',' 6','  0.85983055','  0.52773487','-0.67380822',' 0',' 0 ','28 '),
( 759,' 760','  51','117',' 5','  0.76612336','  0.36719983','-0.66964732',' 0',' 0 ','28 '),
( 760,' 761','  52','116',' 5','  0.70404847','  0.21097544','-0.67122554',' 0',' 0 ','28 '),
( 761,' 762','  53','115',' 5','  0.66998620','  0.06009777','-0.66900471',' 0',' 0 ','28 '),
( 762,' 763','  54','114',' 4','  0.53045861',' -0.01435121','-0.53310695',' 0',' 0 ','28 '),
( 763,' 764','  55','113',' 4','  0.55297161',' -0.16985215','-0.53195270',' 0',' 0 ','28 '),
( 764,' 765','  56','112',' 4','  0.60793079',' -0.32158457','-0.52452390',' 0',' 0 ','28 '),
( 765,' 766','  57','111',' 5','  0.69947865',' -0.47822965','-0.52207997',' 0',' 0 ','28 '),
( 766,' 767','  58','110',' 5','  0.66704701',' -0.55403840','-0.38827391',' 0',' 0 ','28 '),
( 767,' 768','  59','109',' 4','  0.54887438',' -0.39795400','-0.39068055',' 0',' 0 ','28 '),
( 768,' 769','  60','108',' 3','  0.45459531',' -0.24314153','-0.39277209',' 0',' 0 ','28 '),
( 769,' 770','  61','107',' 3','  0.40309330',' -0.08755658','-0.39817929',' 0',' 0 ','28 '),
( 770,' 771','  62','106',' 2','  0.30054428',' -0.16409331','-0.26054840',' 0',' 0 ','28 '),
( 771,' 772','  63','105',' 3','  0.40092566',' -0.31771196','-0.25936864',' 0',' 0 ','28 '),
( 772,' 773','  64','104',' 4','  0.52892428',' -0.47354454','-0.25692574',' 0',' 0 ','28 '),
( 773,' 774','  65','103',' 5','  0.67008617',' -0.63078643','-0.25429818',' 0',' 0 ','28 '),
( 774,' 775','  66','102',' 5','  0.70241172',' -0.70221892','-0.12037993',' 0',' 0 ','28 '),
( 775,' 776','  67','101',' 4','  0.55117004',' -0.54818335','-0.12043009',' 0',' 0 ','28 '),
( 776,' 777','  68','100',' 3','  0.40280301',' -0.39393413','-0.12368737',' 0',' 0 ','28 '),
( 777,' 778','  69',' 99',' 2','  0.26117849',' -0.23962553','-0.12663259',' 0',' 0 ','28 '),
( 778,' 779','  70',' 98',' 0','  0.14791848',' -0.15718712',' 0.00913849',' 0',' 0 ','28 '),
( 779,' 780','  71',' 97',' 0','  0.14632336',' -0.08630316','-0.12736281',' 0',' 0 ','28 '),
( 780,' 781','  72',' 96',' 2','  0.26009989',' -0.00909027','-0.26276746',' 0',' 0 ','28 '),
( 781,' 782','  73',' 95',' 3','  0.40381088','  0.06362847','-0.39975805',' 0',' 0 ','28 '),
( 782,' 783','  74',' 94',' 4','  0.55172623','  0.13990377','-0.53371301',' 0',' 0 ','28 '),
( 783,' 784','  75',' 93',' 4','  0.61349881','  0.29657887','-0.53422192',' 0',' 0 ','28 '),
( 784,' 785','  76',' 92',' 5','  0.70553533','  0.44818110','-0.53940650',' 0',' 0 ','28 '),
( 785,' 786','  77',' 91',' 6','  0.81947013','  0.60787186','-0.54126484',' 0',' 0 ','28 '),
( 786,' 787','  78',' -7','-1','  7.22866706','  5.00814570','-5.20597930',' 1',' 1 ',Null),
( 787,' 788','  79',' -6','-1','  7.28233818',' -5.06085430','-5.23697930',' 1',' 1 ',Null),
( 788,' 789','  80',' 90',' 5','  0.79321082','  0.68285810','-0.40725131',' 0',' 0 ','28 '),
( 789,' 790','  81',' 89',' 5','  0.66339933','  0.52803467','-0.40481167',' 0',' 0 ','28 '),
( 790,' 791','  82',' 88',' 4','  0.54940176','  0.37429369','-0.40494588',' 0',' 0 ','28 '),
( 791,' 792','  83',' 87',' 3','  0.45453809','  0.22083639','-0.39961645',' 0',' 0 ','28 '),
( 792,' 793','  84',' 86',' 2','  0.29861726','  0.14472447','-0.26353035',' 0',' 0 ','28 '),
( 793,' 794','  85',' 85',' 0','  0.14715766','  0.07030731','-0.13158769',' 0',' 0 ','28 '),
( 794,' 795','  86',' 84',' 0','  0.00783309',' -0.00242898',' 0.00527506',' 0',' 0 ','28 '),
( 795,' 796','  87',' 83',' 1','  0.16233434',' -0.07325785',' 0.14258460',' 0',' 0 ','28 '),
( 796,' 797','  88',' 82',' 1','  0.15934749','  0.08002769',' 0.13678635',' 0',' 0 ','28 '),
( 797,' 798','  89',' 81',' 0','  0.14946817','  0.15062054',' 0.00071451',' 0',' 0 ','28 '),
( 798,' 799','  90',' 80',' 2','  0.25916312','  0.22418988','-0.13369798',' 0',' 0 ','28 '),
( 799,' 800','  91',' 79',' 3','  0.40073389','  0.30083928','-0.26773958',' 0',' 0 ','28 '),
( 800,' 801','  92',' 78',' 4','  0.53061797','  0.45727916','-0.27282959',' 0',' 0 ','28 '),
( 801,' 802','  93',' 77',' 3','  0.40077005','  0.37856644','-0.13655087',' 0',' 0 ','28 '),
( 802,' 803','  94',' 76',' 2','  0.30471230','  0.30588352','-0.00214702',' 0',' 0 ','28 '),
( 803,' 804','  95',' 75',' 2','  0.26986588','  0.23392725',' 0.13489229',' 0',' 0 ','28 '),
( 804,' 805','  96',' 74',' 3','  0.40781557','  0.38620848',' 0.13270648',' 0',' 0 ','28 '),
( 805,' 806','  97',' 73',' 3','  0.45780607','  0.45897420','-0.00345855',' 0',' 0 ','28 '),
( 806,' 807','  98',' 72',' 4','  0.55107755','  0.53518033','-0.13777379',' 0',' 0 ','28 '),
( 807,' 808','  99',' 71',' 5','  0.66780289','  0.61147122','-0.27278116',' 0',' 0 ','28 '),
( 808,' 809',' 100',' 70',' 6','  0.81109925','  0.76298070','-0.28012151',' 0',' 0 ','28 '),
( 809,' 810',' 101',' 69',' 5','  0.70199438','  0.68925129','-0.14075849',' 0',' 0 ','28 '),
( 810,' 811',' 102',' 68',' 4','  0.61177705','  0.61294101','-0.00473568',' 0',' 0 ','28 '),
( 811,' 812',' 103',' 67',' 5','  0.70697803','  0.69602827',' 0.12867231',' 0',' 0 ','28 '),
( 812,' 813',' 104',' 66',' 5','  0.76558457','  0.76673689','-0.00711172',' 0',' 0 ','28 '),
( 813,' 814',' 105',' 65',' 6','  0.85410201','  0.84306709','-0.14556405',' 0',' 0 ','28 '),
( 814,' 815',' 106',' 64',' 6','  0.92537215','  0.92647689','-0.01280300',' 0',' 0 ','28 '),
( 815,' 816',' 107',' 63',' 6','  0.85845735','  0.85025896',' 0.12480825',' 0',' 0 ','28 '),
( 816,' 817',' 108',' 62',' 6','  0.81818232','  0.77688468',' 0.25849441',' 0',' 0 ','28 '),
( 817,' 818',' 109',' 61',' 6','  0.80496021','  0.70208719',' 0.39414457',' 0',' 0 ','28 '),
( 818,' 819',' 110',' -5','-1',' 13.21294337','-12.12385400','-5.25197930',' 1',' 1 ',Null),
( 819,' 820',' 111',' -4','-1',' 16.21141839',' 12.01914600','10.85702100',' 1',' 1 ',Null),
( 820,' 821',' 112',' 60',' 6','  0.83581811','  0.62728962',' 0.52979189',' 0',' 0 ','28 '),
( 821,' 822',' 113',' 59',' 5','  0.69456992','  0.54996142',' 0.39947953',' 0',' 0 ','28 '),
( 822,' 823',' 114',' 58',' 5','  0.69009795','  0.62296408',' 0.26111885',' 0',' 0 ','28 '),
( 823,' 824',' 115',' 57',' 4','  0.57180118','  0.54209334',' 0.13129427',' 0',' 0 ','28 '),
( 824,' 825',' 116',' 56',' 4','  0.55677208','  0.47121798',' 0.26768701',' 0',' 0 ','28 '),
( 825,' 826',' 117',' 55',' 3','  0.42861827','  0.31742586',' 0.26557270',' 0',' 0 ','28 '),
( 826,' 827',' 118',' 54',' 2','  0.32457866','  0.15851187',' 0.26781296',' 0',' 0 ','28 '),
( 827,' 828',' 119',' 53',' 2','  0.28498240','  0.00229358',' 0.27651510',' 0',' 0 ','28 '),
( 828,' 829',' 120',' 52',' 2','  0.31806562',' -0.14988347',' 0.27876095',' 0',' 0 ','28 '),
( 829,' 830',' 121',' 51',' 3','  0.42136201',' -0.07476155',' 0.40864564',' 0',' 0 ','28 '),
( 830,' 831',' 122',' 50',' 3','  0.42782932','  0.07901280',' 0.40985095',' 0',' 0 ','28 '),
( 831,' 832',' 123',' 49',' 3','  0.47939054','  0.23645766',' 0.40161604',' 0',' 0 ','28 '),
( 832,' 833',' 124',' 48',' 4','  0.57808304','  0.39494158',' 0.40204113',' 0',' 0 ','28 '),
( 833,' 834',' 125',' 47',' 5','  0.72741980','  0.47330243',' 0.53337027',' 0',' 0 ','28 '),
( 834,' 835',' 126',' 46',' 4','  0.63607020','  0.31932575',' 0.53459635',' 0',' 0 ','28 '),
( 835,' 836',' 127',' 45',' 4','  0.57503663','  0.16183088',' 0.53990178',' 0',' 0 ','28 '),
( 836,' 837',' 128',' 44',' 4','  0.55367175','  0.00790552',' 0.54521194',' 0',' 0 ','28 '),
( 837,' 838',' 129',' 43',' 4','  0.57152048',' -0.14541563',' 0.54775532',' 0',' 0 ','28 '),
( 838,' 839',' 130',' 42',' 5','  0.71989476',' -0.21917920',' 0.68149146',' 0',' 0 ','28 '),
( 839,' 840',' 131',' 41',' 5','  0.68937902',' -0.06503680',' 0.67928608',' 0',' 0 ','28 '),
( 840,' 841',' 132',' 40',' 5','  0.69176291','  0.08848603',' 0.67627277',' 0',' 0 ','28 '),
( 841,' 842',' 133',' 39',' 5','  0.72926958','  0.24547033',' 0.67404816',' 0',' 0 ','28 '),
( 842,' 843',' 134',' 38',' 5','  0.79038670','  0.39907533',' 0.66671069',' 0',' 0 ','28 '),
( 843,' 844',' 135',' 37',' 6','  0.87909632','  0.55249213',' 0.66544206',' 0',' 0 ','28 '),
( 844,' 845',' 136',' 36',' 6','  0.94605392','  0.47769741',' 0.80108930',' 0',' 0 ','28 '),
( 845,' 846',' 137',' 35',' 6','  0.87870436','  0.32282413',' 0.80413749',' 0',' 0 ','28 '),
( 846,' 847',' 138',' 34',' 6','  0.83815358','  0.16644926',' 0.81075108',' 0',' 0 ','28 '),
( 847,' 848',' 139',' 33',' 6','  0.82044697','  0.01484847',' 0.81190723',' 0',' 0 ','28 '),
( 848,' 849',' 140',' 32',' 6','  0.83360579',' -0.13875746',' 0.81589771',' 0',' 0 ','28 '),
( 849,' 850',' 141',' 31',' 6','  0.87046656',' -0.29488261',' 0.81529195',' 0',' 0 ','28 '),
( 850,' 851',' 142',' -3','-1',' 11.92997751','  4.94514570','10.84302100',' 1',' 1 ',Null),
( 851,' 852',' 143',' -2','-1',' 11.96395923',' -5.11685430','10.82602100',' 1',' 1 ',Null),
( 852,' 853',' 144',' 30',' 6','  0.92037472',' -0.44695198',' 0.81673960',' 0',' 0 ','28 '),
( 853,' 854',' 145',' 29',' 6','  0.85582063',' -0.53002196',' 0.68620777',' 0',' 0 ','28 '),
( 854,' 855',' 146',' 28',' 5','  0.76604967',' -0.37269830',' 0.68144844',' 0',' 0 ','28 '),
( 855,' 856',' 147',' 27',' 4','  0.61365035',' -0.30215180',' 0.54634142',' 0',' 0 ','28 '),
( 856,' 857',' 148',' 26',' 5','  0.70277194',' -0.45369611',' 0.55145038',' 0',' 0 ','28 '),
( 857,' 858',' 149',' 25',' 6','  0.81538077',' -0.61211954',' 0.55601574',' 0',' 0 ','28 '),
( 858,' 859',' 150',' 24',' 6','  0.80375411',' -0.69569397',' 0.42504935',' 0',' 0 ','28 '),
( 859,' 860',' 151',' 23',' 5','  0.67237409',' -0.53856108',' 0.42162349',' 0',' 0 ','28 '),
( 860,' 861',' 152',' 22',' 4','  0.55444368',' -0.38272055',' 0.41683975',' 0',' 0 ','28 '),
( 861,' 862',' 153',' 21',' 3','  0.45747864',' -0.22855460',' 0.40859260',' 0',' 0 ','28 '),
( 862,' 863',' 154',' 20',' 3','  0.40430184',' -0.30697290',' 0.28055779',' 0',' 0 ','28 '),
( 863,' 864',' 155',' 19',' 4','  0.53227108',' -0.46297593',' 0.28523169',' 0',' 0 ','28 '),
( 864,' 865',' 156',' 18',' 5','  0.67278634',' -0.61937401',' 0.29039475',' 0',' 0 ','28 '),
( 865,' 866',' 157',' 17',' 6','  0.81398432',' -0.77130271',' 0.29287366',' 0',' 0 ','28 '),
( 866,' 867',' 158',' 16',' 6','  0.85488476',' -0.85069944',' 0.15731196',' 0',' 0 ','28 '),
( 867,' 868',' 159',' 15',' 5','  0.70337504',' -0.69672256',' 0.15521063',' 0',' 0 ','28 '),
( 868,' 869',' 160',' 14',' 4','  0.55182231',' -0.54121123',' 0.15335477',' 0',' 0 ','28 '),
( 869,' 870',' 161',' 13',' 2','  0.39723901',' -0.38126398',' 0.14608441',' 0',' 0 ','28 '),
( 870,' 871',' 162',' 12',' 2','  0.25867445',' -0.22763628',' 0.14558579',' 0',' 0 ','28 '),
( 871,' 872',' 163',' 11',' 2','  0.30072548',' -0.30982035',' 0.00884452',' 0',' 0 ','28 '),
( 872,' 873',' 164',' 10',' 3','  0.45591187',' -0.46496646',' 0.01358181',' 0',' 0 ','28 '),
( 873,' 874',' 165','  9',' 4','  0.60866265',' -0.61769114',' 0.01648041',' 0',' 0 ','28 '),
( 874,' 875',' 166','  8',' 5','  0.76301522',' -0.77201595',' 0.01946026',' 0',' 0 ','28 '),
( 875,' 876',' 167','  7',' 6','  0.91743668',' -0.92638624',' 0.02376983',' 0',' 0 ','28 '),
( 876,' 877',' 168','  6',' 6','  0.85851071',' -0.85884178','-0.11519966',' 0',' 0 ','28 '),
( 877,' 878',' 169','  5',' 6','  0.81979260',' -0.78724691','-0.25077851',' 0',' 0 ','28 '),
( 878,' 879',' 170','  4',' 6','  0.80452822',' -0.71082441','-0.38631955',' 0',' 0 ','28 '),
( 879,' 880',' 171','  3',' 6','  0.81646496',' -0.63223655','-0.52036845',' 0',' 0 ','28 '),
( 880,' 881',' 172','  2',' 6','  0.85870981',' -0.55589700','-0.65492662',' 0',' 0 ','28 '),
( 881,' 882',' 173','  1',' 6','  0.92367483',' -0.47929860','-0.78785219',' 0',' 0 ','28 '),
( 882,' 883',' 174',' -1','-1',' 16.28019912','-12.18785400','10.81102100',' 1',' 1 ',Null),
( 883,' 884',' 175',' -3','-1',' 37.65743471',' 30.43782500','22.17887400',' 1',' 1 ',Null),
( 884,' 885',' 176',' -2','-1',' 30.09459797',' 20.37382500','22.15587400',' 1',' 1 ',Null),
( 885,' 886',' 177',' 21',' 0','  0.13885510','  0.13830772',' 0.01120454',' 0',' 2 ','11 '),
( 886,' 887',' 178',' 20',' 0','  0.14852737','  0.05858615',' 0.14328162',' 0',' 2 ','11 '),
( 887,' 888',' 179',' 19',' 1','  0.16368377',' -0.09597580',' 0.13994811',' 0',' 2 ','11 '),
( 888,' 889',' 180',' 18',' 0','  0.01564795',' -0.01613182',' 0.00698711',' 0',' 2 ','11 '),
( 889,' 890',' 181',' 17',' 0','  0.14664686','  0.06367317','-0.12486226',' 0',' 2 ','11 '),
( 890,' 891',' 182',' 16',' 1','  0.16208550',' -0.09096045','-0.12747733',' 0',' 2 ','11 '),
( 891,' 892',' 183',' 15',' 1','  0.17050025',' -0.17097158',' 0.00493704',' 0',' 2 ','11 '),
( 892,' 893',' 184',' 14',' 0','  0.14472487','  0.14338577','-0.00870449',' 0',' 2 ','10 '),
( 893,' 894',' 185',' 13',' 0','  0.13612275','  0.06562360',' 0.12599821',' 0',' 2 ','10 '),
( 894,' 895',' 186',' 12',' 0','  0.14820064',' -0.08831441',' 0.12637589',' 0',' 2 ','10 '),
( 895,' 896',' 187',' 11',' 0','  0.01676846',' -0.00880959','-0.00754979',' 0',' 2 ','10 '),
( 896,' 897',' 188',' 10',' 1','  0.16377496','  0.06686064','-0.14228240',' 0',' 2 ','10 '),
( 897,' 898',' 189','  9',' 1','  0.17314371',' -0.08751621','-0.14267430',' 0',' 2 ','10 '),
( 898,' 899',' 190','  8',' 1','  0.16507281',' -0.16490174','-0.00768442',' 0',' 2 ','10 '),
( 899,' 900',' 191','  7',' 1','  0.16374891','  0.16320542',' 0.00258781',' 0',' 2 ','9 '),
( 900,' 901',' 192','  6',' 1','  0.15358114','  0.08308900',' 0.13585737',' 0',' 2 ','9 '),
( 901,' 902',' 193','  5',' 0','  0.14533054',' -0.07219576',' 0.13341134',' 0',' 2 ','9 '),
( 902,' 903',' 194','  4',' 0','  0.00984219','  0.00729049',' 0.00097021',' 0',' 2 ','9 '),
( 903,' 904',' 195','  3',' 1','  0.16592742','  0.08859327','-0.13298413',' 0',' 2 ','9 '),
( 904,' 905',' 196','  2',' 1','  0.15789457',' -0.06720654','-0.13609834',' 0',' 2 ','9 '),
( 905,' 906',' 197','  1',' 0','  0.14661251',' -0.14685950','-0.00132539',' 0',' 2 ','9 '),
( 906,' 907',' 198',' -1','-1',' 25.83494372',' 13.30882500','22.14987400',' 1',' 1 ',Null),
( 907,' 908',' 199',' -2','-1',' 11.97700990',' -5.12309060','10.82696600',' 1',' 1 ',Null),
( 908,' 909',' 200',' 19',' 2','  0.31293645',' -0.14601445','-0.26825876',' 0',' 0 ','13 '),
( 909,' 910',' 201',' 18',' 2','  0.26835801',' -0.22343564','-0.13232347',' 0',' 0 ','13 '),
( 910,' 911',' 202',' 17',' 1','  0.15156450',' -0.14411085',' 0.00244019',' 0',' 0 ','13 '),
( 911,' 912',' 203',' 16',' 1','  0.15789639',' -0.06883652','-0.13378041',' 0',' 0 ','13 '),
( 912,' 913',' 204',' 15',' 2','  0.27478858','  0.00730340','-0.27031862',' 0',' 0 ','13 '),
( 913,' 914',' 205',' 14',' 2','  0.31152277','  0.16083311','-0.26667028',' 0',' 0 ','13 '),
( 914,' 915',' 206',' 13',' 1','  0.15740270','  0.08481776','-0.13260050',' 0',' 0 ','13 '),
( 915,' 916',' 207',' 12',' 0','  0.00361501','  0.00722742',' 0.00086117',' 0',' 0 ','13 '),
( 916,' 917',' 208',' 11',' 1','  0.15278610',' -0.06900881',' 0.13675432',' 0',' 0 ','13 '),
( 917,' 918',' 209',' 10',' 1','  0.15358741','  0.08666939',' 0.13604425',' 0',' 0 ','13 '),
( 918,' 919',' 210','  9',' 1','  0.15650500','  0.16388893',' 0.00027855',' 0',' 0 ','13 '),
( 919,' 920',' 211','  8',' 2','  0.27455021','  0.24380245','-0.13521056',' 0',' 0 ','13 '),
( 920,' 921',' 212','  7',' 2','  0.31071801','  0.31811033','-0.00097657',' 0',' 0 ','13 '),
( 921,' 922',' 213','  6',' 2','  0.26930242','  0.24188909',' 0.13697443',' 0',' 0 ','13 '),
( 922,' 923',' 214','  5',' 2','  0.30952930','  0.16445778',' 0.27121666',' 0',' 0 ','13 '),
( 923,' 924',' 215','  4',' 2','  0.26797577','  0.01059746',' 0.27242709',' 0',' 0 ','13 '),
( 924,' 925',' 216','  3',' 2','  0.30494511',' -0.14434816',' 0.26895405',' 0',' 0 ','13 '),
( 925,' 926',' 217','  2',' 2','  0.26656082',' -0.22405481',' 0.13662435',' 0',' 0 ','13 '),
( 926,' 927',' 218','  1',' 2','  0.30564652',' -0.29819584',' 0.00192218',' 0',' 0 ','13 '),
( 927,' 928',' 219',' -1','-1',' 16.29168084','-12.17809100','10.81796600',' 1',' 1 ',Null),
( 928,' 929',' 220',' -8','-1',' 13.14551674',' 12.07778800','-5.20238240',' 1',' 1 ',Null),
( 929,' 930',' 221','127',' 6','  0.86896287',' -0.28909844','-0.81232956',' 0',' 0 ','27 '),
( 930,' 931',' 222','126',' 5','  0.77899456',' -0.37253472','-0.67556809',' 0',' 0 ','27 '),
( 931,' 932',' 223','125',' 5','  0.71487828',' -0.21914440','-0.67354957',' 0',' 0 ','27 '),
( 932,' 933',' 224','124',' 5','  0.68142881',' -0.06323183','-0.67328424',' 0',' 0 ','27 '),
( 933,' 934',' 225','123',' 6','  0.83069377',' -0.13730721','-0.81351563',' 0',' 0 ','27 '),
( 934,' 935',' 226','122',' 6','  0.81400180','  0.01932419','-0.80944512',' 0',' 0 ','27 '),
( 935,' 936',' 227','121',' 6','  0.82346466','  0.17093278','-0.80260144',' 0',' 0 ','27 '),
( 936,' 937',' 228','120',' 6','  0.86463941','  0.32531349','-0.79961825',' 0',' 0 ','27 '),
( 937,' 938',' 229','119',' 6','  0.92844973','  0.48217709','-0.79342961',' 0',' 0 ','27 '),
( 938,' 939',' 230','118',' 6','  0.86087990','  0.55714394','-0.65805544',' 0',' 0 ','27 '),
( 939,' 940',' 231','117',' 5','  0.77390925','  0.39921491','-0.66294899',' 0',' 0 ','27 '),
( 940,' 941',' 232','116',' 5','  0.71292879','  0.24779377','-0.66672123',' 0',' 0 ','27 '),
( 941,' 942',' 233','115',' 5','  0.68345496','  0.08816411','-0.67420107',' 0',' 0 ','27 '),
( 942,' 943',' 234','114',' 4','  0.54038846','  0.01532384','-0.53586103',' 0',' 0 ','27 '),
( 943,' 944',' 235','113',' 4','  0.56531775',' -0.14118488','-0.54096090',' 0',' 0 ','27 '),
( 944,' 945',' 236','112',' 4','  0.62243965',' -0.29635859','-0.53879566',' 0',' 0 ','27 '),
( 945,' 946',' 237','111',' 5','  0.71734834',' -0.45627937','-0.54284436',' 0',' 0 ','27 '),
( 946,' 947',' 238','110',' 5','  0.68855576',' -0.53998325','-0.41318634',' 0',' 0 ','27 '),
( 947,' 948',' 239','109',' 4','  0.56796374',' -0.38359267','-0.40744780',' 0',' 0 ','27 '),
( 948,' 949',' 240','108',' 3','  0.46971195',' -0.23166269','-0.39983095',' 0',' 0 ','27 '),
( 949,' 950',' 241','107',' 3','  0.41373258',' -0.06451350','-0.40295778',' 0',' 0 ','27 '),
( 950,' 951',' 242','106',' 2','  0.31338299',' -0.14956000','-0.26674932',' 0',' 0 ','27 '),
( 951,' 952',' 243','105',' 3','  0.41452430',' -0.30628018','-0.26647288',' 0',' 0 ','27 '),
( 952,' 953',' 244','104',' 4','  0.54721685',' -0.46362455','-0.27399805',' 0',' 0 ','27 '),
( 953,' 954',' 245','103',' 5','  0.68826006',' -0.62003175','-0.27833926',' 0',' 0 ','27 '),
( 954,' 955',' 246','102',' 5','  0.71743444',' -0.69435023','-0.14453532',' 0',' 0 ','27 '),
( 955,' 956',' 247','101',' 4','  0.56647341',' -0.54008524','-0.14081659',' 0',' 0 ','27 '),
( 956,' 957',' 248','100',' 3','  0.41749007',' -0.38602737','-0.13510565',' 0',' 0 ','27 '),
( 957,' 958',' 249',' 99',' 2','  0.27350714',' -0.22859021','-0.13371787',' 0',' 0 ','27 '),
( 958,' 959',' 250',' 98',' 1','  0.16411906',' -0.15665208',' 0.00149773',' 0',' 0 ','27 '),
( 959,' 960',' 251',' 97',' 1','  0.15680200',' -0.07446039','-0.12924314',' 0',' 0 ','27 '),
( 960,' 961',' 252',' 96',' 2','  0.26688427','  0.00744288','-0.26241435',' 0',' 0 ','27 '),
( 961,' 962',' 253',' 95',' 3','  0.40939071','  0.08913636','-0.39668649',' 0',' 0 ','27 '),
( 962,' 963',' 254',' 94',' 4','  0.55774201','  0.17148132','-0.52860288',' 0',' 0 ','27 '),
( 963,' 964',' 255',' 93',' 4','  0.61940185','  0.32554246','-0.52700869',' 0',' 0 ','27 '),
( 964,' 965',' 256',' 92',' 5','  0.70803424','  0.47734941','-0.52515041',' 0',' 0 ','27 '),
( 965,' 966',' 257',' 91',' 6','  0.81904179','  0.63371894','-0.52335981',' 0',' 0 ','27 '),
( 966,' 967',' 258',' -7','-1','  7.22143156','  5.00078820','-5.21238240',' 1',' 1 ',Null),
( 967,' 968',' 259',' -6','-1','  7.29513255',' -5.06621180','-5.23738240',' 1',' 1 ',Null),
( 968,' 969',' 260',' 90',' 6','  0.80521094','  0.70882384','-0.39103666',' 0',' 0 ','27 '),
( 969,' 970',' 261',' 89',' 5','  0.67355986','  0.55404619','-0.38910933',' 0',' 0 ','27 '),
( 970,' 971',' 262',' 88',' 4','  0.55828546','  0.40138304','-0.39112025',' 0',' 0 ','27 '),
( 971,' 972',' 263',' 87',' 3','  0.46210901','  0.24382666','-0.39260200',' 0',' 0 ','27 '),
( 972,' 973',' 264',' 86',' 2','  0.30686638','  0.16141765','-0.26096911',' 0',' 0 ','27 '),
( 973,' 974',' 265',' 85',' 1','  0.15028352','  0.08255430','-0.12569531',' 0',' 0 ','27 '),
( 974,' 975',' 266',' 84',' 0','  0.00776766',' -0.00007311',' 0.00644190',' 0',' 0 ','27 '),
( 975,' 976',' 267',' 83',' 1','  0.16003667',' -0.08168369',' 0.13739356',' 0',' 0 ','27 '),
( 976,' 977',' 268',' 82',' 1','  0.15321855','  0.07536961',' 0.14180710',' 0',' 0 ','27 '),
( 977,' 978',' 269',' 81',' 0','  0.14425806','  0.15160998',' 0.00951203',' 0',' 0 ','27 '),
( 978,' 979',' 270',' 80',' 2','  0.26142026','  0.23601947','-0.12238435',' 0',' 0 ','27 '),
( 979,' 980',' 271',' 79',' 3','  0.40884783','  0.32168226','-0.25708005',' 0',' 0 ','27 '),
( 980,' 981',' 272',' 78',' 4','  0.53590948','  0.47614254','-0.25537033',' 0',' 0 ','27 '),
( 981,' 982',' 273',' 77',' 3','  0.40483116','  0.39276215','-0.11968788',' 0',' 0 ','27 '),
( 982,' 983',' 274',' 76',' 2','  0.30136084','  0.30855612',' 0.01661375',' 0',' 0 ','27 '),
( 983,' 984',' 275',' 75',' 2','  0.26260919','  0.22831643',' 0.14651646',' 0',' 0 ','27 '),
( 984,' 985',' 276',' 74',' 3','  0.40476830','  0.38577588',' 0.14834272',' 0',' 0 ','27 '),
( 985,' 986',' 277',' 73',' 3','  0.45738892','  0.46467342',' 0.01639790',' 0',' 0 ','27 '),
( 986,' 987',' 278',' 72',' 4','  0.55269794','  0.54665093','-0.11688349',' 0',' 0 ','27 '),
( 987,' 988',' 279',' 71',' 5','  0.67339899','  0.62912076','-0.25433368',' 0',' 0 ','27 '),
( 988,' 989',' 280',' 70',' 6','  0.81853683','  0.78383220','-0.25479401',' 0',' 0 ','27 '),
( 989,' 990',' 281',' 69',' 5','  0.70495285','  0.70162429','-0.11827686',' 0',' 0 ','27 '),
( 990,' 991',' 282',' 68',' 4','  0.61182839','  0.61914411',' 0.01680451',' 0',' 0 ','27 '),
( 991,' 992',' 283',' 67',' 5','  0.70450787','  0.69612550',' 0.15294115',' 0',' 0 ','27 '),
( 992,' 993',' 284',' 66',' 5','  0.76633082','  0.77363207',' 0.01905521',' 0',' 0 ','27 '),
( 993,' 994',' 285',' 65',' 6','  0.85871619','  0.85746354','-0.11740532',' 0',' 0 ','27 '),
( 994,' 995',' 286',' 64',' 6','  0.92016125','  0.92749112',' 0.01871012',' 0',' 0 ','27 '),
( 995,' 996',' 287',' 63',' 6','  0.85617194','  0.85032771',' 0.15470575',' 0',' 0 ','27 '),
( 996,' 997',' 288',' 62',' 6','  0.81423912','  0.77099750',' 0.28725144',' 0',' 0 ','27 '),
( 997,' 998',' 289',' 61',' 6','  0.80434353','  0.69499485',' 0.42188689',' 0',' 0 ','27 '),
( 998,' 999',' 290',' -5','-1',' 13.22557474','-12.12721200','-5.25538240',' 1',' 1 ',Null),
( 999,'1000',' 291',' -4','-1',' 16.18978176',' 12.01978800','10.85861800',' 1',' 1 ',Null),
(1000,'1001',' 292',' 60',' 6','  0.82160936','  0.61601125',' 0.55645095',' 0',' 0 ','27 '),
(1001,'1002',' 293',' 59',' 5','  0.67717593','  0.54133409',' 0.42103253',' 0',' 0 ','27 '),
(1002,'1003',' 294',' 58',' 5','  0.67161579','  0.61746384',' 0.28543034',' 0',' 0 ','27 '),
(1003,'1004',' 295',' 57',' 4','  0.55431381','  0.54176908',' 0.15197008',' 0',' 0 ','27 '),
(1004,'1005',' 296',' 56',' 4','  0.53068046','  0.45822754',' 0.28449206',' 0',' 0 ','27 '),
(1005,'1006',' 297',' 55',' 3','  0.40473323','  0.30313986',' 0.28082226',' 0',' 0 ','27 '),
(1006,'1007',' 298',' 54',' 2','  0.30798954','  0.14718689',' 0.27893008',' 0',' 0 ','27 '),
(1007,'1008',' 299',' 53',' 2','  0.27061946',' -0.00673114',' 0.27471808',' 0',' 0 ','27 '),
(1008,'1009',' 300',' 52',' 2','  0.31376702',' -0.16101270',' 0.26918375',' 0',' 0 ','27 '),
(1009,'1010',' 301',' 51',' 3','  0.41442911',' -0.08720182',' 0.40794780',' 0',' 0 ','27 '),
(1010,'1011',' 302',' 50',' 3','  0.41168228','  0.06912888',' 0.41150406',' 0',' 0 ','27 '),
(1011,'1012',' 303',' 49',' 3','  0.46398276','  0.22324251',' 0.41521230',' 0',' 0 ','27 '),
(1012,'1013',' 304',' 48',' 4','  0.55330765','  0.37842993',' 0.41497677',' 0',' 0 ','27 '),
(1013,'1014',' 305',' 47',' 5','  0.70590357','  0.45557742',' 0.54988056',' 0',' 0 ','27 '),
(1014,'1015',' 306',' 46',' 4','  0.61665695','  0.29902759',' 0.54783215',' 0',' 0 ','27 '),
(1015,'1016',' 307',' 45',' 4','  0.55766605','  0.14207506',' 0.54563980',' 0',' 0 ','27 '),
(1016,'1017',' 308',' 44',' 4','  0.54215477',' -0.01463013',' 0.54617529',' 0',' 0 ','27 '),
(1017,'1018',' 309',' 43',' 4','  0.56349156',' -0.16881566',' 0.53968638',' 0',' 0 ','27 '),
(1018,'1019',' 310',' 42',' 5','  0.71712904',' -0.24813201',' 0.67451244',' 0',' 0 ','27 '),
(1019,'1020',' 311',' 41',' 5','  0.68193218',' -0.09215798',' 0.67908961',' 0',' 0 ','27 '),
(1020,'1021',' 312',' 40',' 5','  0.67992538','  0.06208325',' 0.68219600',' 0',' 0 ','27 '),
(1021,'1022',' 313',' 39',' 5','  0.70927026','  0.21663770',' 0.68218716',' 0',' 0 ','27 '),
(1022,'1023',' 314',' 38',' 5','  0.77285689','  0.37607191',' 0.68374772',' 0',' 0 ','27 '),
(1023,'1024',' 315',' 37',' 6','  0.86055674','  0.53272925',' 0.68610705',' 0',' 0 ','27 '),
(1024,'1025',' 316',' 36',' 6','  0.92575361','  0.45039380',' 0.81737320',' 0',' 0 ','27 '),
(1025,'1026',' 317',' 35',' 6','  0.86077557','  0.29452402',' 0.81596076',' 0',' 0 ','27 '),
(1026,'1027',' 318',' 34',' 6','  0.82243951','  0.13999418',' 0.81615715',' 0',' 0 ','27 '),
(1027,'1028',' 319',' 33',' 6','  0.80989357',' -0.01460963',' 0.81406328',' 0',' 0 ','27 '),
(1028,'1029',' 320',' 32',' 6','  0.82712977',' -0.17108313',' 0.81210420',' 0',' 0 ','27 '),
(1029,'1030',' 321',' 31',' 6','  0.86729716',' -0.32301257',' 0.80634611',' 0',' 0 ','27 '),
(1030,'1031',' 322',' -3','-1',' 11.91565482','  4.94578820','10.84861800',' 1',' 1 ',Null),
(1031,'1032',' 323',' -2','-1',' 11.96870867',' -5.11921180','10.81961800',' 1',' 1 ',Null),
(1032,'1033',' 324',' 30',' 6','  0.93207358',' -0.48170021',' 0.79788214',' 0',' 0 ','27 '),
(1033,'1034',' 325',' 29',' 6','  0.87041137',' -0.55922408',' 0.66515722',' 0',' 0 ','27 '),
(1034,'1035',' 326',' 28',' 5','  0.77834942',' -0.40198057',' 0.66643862',' 0',' 0 ','27 '),
(1035,'1036',' 327',' 27',' 4','  0.62274512',' -0.32086906',' 0.53364343',' 0',' 0 ','27 '),
(1036,'1037',' 328',' 26',' 5','  0.71593901',' -0.47730845',' 0.53133570',' 0',' 0 ','27 '),
(1037,'1038',' 329',' 25',' 6','  0.82792718',' -0.63172169',' 0.53071670',' 0',' 0 ','27 '),
(1038,'1039',' 330',' 24',' 6','  0.81708160',' -0.70869225',' 0.39788681',' 0',' 0 ','27 '),
(1039,'1040',' 331',' 23',' 5','  0.68535875',' -0.55551149',' 0.39536909',' 0',' 0 ','27 '),
(1040,'1041',' 332',' 22',' 4','  0.56580097',' -0.39744853',' 0.39968628',' 0',' 0 ','27 '),
(1041,'1042',' 333',' 21',' 3','  0.47152115',' -0.24154149',' 0.40489517',' 0',' 0 ','27 '),
(1042,'1043',' 334',' 20',' 3','  0.41681697',' -0.31657623',' 0.26667181',' 0',' 0 ','27 '),
(1043,'1044',' 335',' 19',' 4','  0.54872736',' -0.47526125',' 0.26543191',' 0',' 0 ','27 '),
(1044,'1045',' 336',' 18',' 5','  0.68650899',' -0.62879498',' 0.26234501',' 0',' 0 ','27 '),
(1045,'1046',' 337',' 17',' 6','  0.82910994',' -0.78138674',' 0.25976499',' 0',' 0 ','27 '),
(1046,'1047',' 338',' 16',' 6','  0.87340058',' -0.85752804',' 0.12554324',' 0',' 0 ','27 '),
(1047,'1048',' 339',' 15',' 5','  0.72231001',' -0.70423630',' 0.12795478',' 0',' 0 ','27 '),
(1048,'1049',' 340',' 14',' 4','  0.56920677',' -0.54763043',' 0.13053776',' 0',' 0 ','27 '),
(1049,'1050',' 341',' 13',' 3','  0.42081213',' -0.39253396',' 0.13525076',' 0',' 0 ','27 '),
(1050,'1051',' 342',' 12',' 2','  0.27911105',' -0.23899327',' 0.13551797',' 0',' 0 ','27 '),
(1051,'1052',' 343',' 11',' 2','  0.31845787',' -0.31096430','-0.00136743',' 0',' 0 ','27 '),
(1052,'1053',' 344',' 10',' 3','  0.47195519',' -0.46448070','-0.00123041',' 0',' 0 ','27 '),
(1053,'1054',' 345','  9',' 4','  0.62871467',' -0.62117417','-0.00676757',' 0',' 0 ','27 '),
(1054,'1055',' 346','  8',' 5','  0.78282753',' -0.77527139','-0.00901087',' 0',' 0 ','27 '),
(1055,'1056',' 347','  7',' 6','  0.93686271',' -0.92932200','-0.00926256',' 0',' 0 ','27 '),
(1056,'1057',' 348','  6',' 6','  0.87390330',' -0.85376315','-0.14397536',' 0',' 0 ','27 '),
(1057,'1058',' 349','  5',' 6','  0.83143587',' -0.77138724','-0.28658616',' 0',' 0 ','27 '),
(1058,'1059',' 350','  4',' 6','  0.81849605',' -0.69240577','-0.41997276',' 0',' 0 ','27 '),
(1059,'1060',' 351','  3',' 6','  0.83316972',' -0.61218607','-0.55251773',' 0',' 0 ','27 '),
(1060,'1061',' 352','  2',' 6','  0.87013980',' -0.52993398','-0.67990747',' 0',' 0 ','27 '),
(1061,'1062',' 353','  1',' 6','  0.93488070',' -0.44541887','-0.81340581',' 0',' 0 ','27 '),
(1062,'1063',' 354',' -1','-1',' 16.29004447','-12.17621200','10.81761800',' 1',' 1 ',Null),
(1063,'1064',' 355',' -6','-1','  7.30461881',' -5.07329980','-5.24372390',' 1',' 1 ',Null),
(1064,'1065',' 356',' 91',' 5','  0.67449191','  0.62248759','-0.27239806',' 0',' 0 ','23 '),
(1065,'1066',' 357',' 90',' 5','  0.70828774','  0.70111610','-0.13865615',' 0',' 0 ','23 '),
(1066,'1067',' 358',' 89',' 4','  0.55345793','  0.54250504','-0.13702620',' 0',' 0 ','23 '),
(1067,'1068',' 359',' 88',' 4','  0.53603713','  0.46579418','-0.27346407',' 0',' 0 ','23 '),
(1068,'1069',' 360',' 87',' 3','  0.40606120','  0.38812066','-0.13683817',' 0',' 0 ','23 '),
(1069,'1070',' 361',' 86',' 2','  0.26510445','  0.23340280','-0.13417066',' 0',' 0 ','23 '),
(1070,'1071',' 362',' 85',' 3','  0.41223804','  0.31417222','-0.27094901',' 0',' 0 ','23 '),
(1071,'1072',' 363',' 84',' 4','  0.55926303','  0.38632120','-0.40689880',' 0',' 0 ','23 '),
(1072,'1073',' 364',' 83',' 4','  0.61729221','  0.30507996','-0.53632594',' 0',' 0 ','23 '),
(1073,'1074',' 365',' 82',' 3','  0.46617868','  0.22953506','-0.40540369',' 0',' 0 ','23 '),
(1074,'1075',' 366',' 81',' 2','  0.30937446','  0.15788860','-0.26585913',' 0',' 0 ','23 '),
(1075,'1076',' 367',' 80',' 1','  0.15230317','  0.08085206','-0.12897270',' 0',' 0 ','23 '),
(1076,'1077',' 368',' 79',' 0','  0.00483298','  0.00316354',' 0.00672137',' 0',' 0 ','23 '),
(1077,'1078',' 369',' 78',' 0','  0.14959203','  0.15702886',' 0.00545286',' 0',' 0 ','23 '),
(1078,'1079',' 370',' 77',' 2','  0.30568750','  0.31306627','-0.00165112',' 0',' 0 ','23 '),
(1079,'1080',' 371',' 76',' 3','  0.45956247','  0.46694840','-0.00258356',' 0',' 0 ','23 '),
(1080,'1081',' 372',' 75',' 4','  0.61368229','  0.62105490','-0.00462840',' 0',' 0 ','23 '),
(1081,'1082',' 373',' 74',' 5','  0.77014632','  0.77751504','-0.00601280',' 0',' 0 ','23 '),
(1082,'1083',' 374',' 73',' 5','  0.70634092','  0.70223293',' 0.13167200',' 0',' 0 ','23 '),
(1083,'1084',' 375',' 72',' 4','  0.55297763','  0.54588952',' 0.13039227',' 0',' 0 ','23 '),
(1084,'1085',' 376',' 71',' 3','  0.40513744','  0.39158426',' 0.13319281',' 0',' 0 ','23 '),
(1085,'1086',' 377',' 70',' 2','  0.26670395','  0.23783468',' 0.13881767',' 0',' 0 ','23 '),
(1086,'1087',' 378',' 69',' 2','  0.30716759','  0.15977365',' 0.27120276',' 0',' 0 ','23 '),
(1087,'1088',' 379',' 68',' 3','  0.40594998','  0.31495702',' 0.26947692',' 0',' 0 ','23 '),
(1088,'1089',' 380',' 67',' 4','  0.53275981','  0.47092636',' 0.26717406',' 0',' 0 ','23 '),
(1089,'1090',' 381',' 66',' 5','  0.67123318','  0.62552674',' 0.26623861',' 0',' 0 ','23 '),
(1090,'1091',' 382',' 65',' 5','  0.67497650','  0.55257038',' 0.40249783',' 0',' 0 ','23 '),
(1091,'1092',' 383',' 64',' 5','  0.70968746','  0.47505771',' 0.53831449',' 0',' 0 ','23 '),
(1092,'1093',' 384',' 63',' 5','  0.76985480','  0.39716473',' 0.66839092',' 0',' 0 ','23 '),
(1093,'1094',' 385',' 62',' 5','  0.70731980','  0.24114337',' 0.67206565',' 0',' 0 ','23 '),
(1094,'1095',' 386',' 61',' 5','  0.67684968','  0.08501461',' 0.67685945',' 0',' 0 ','23 '),
(1095,'1096',' 387',' -5','-1',' 13.23248950','-12.13330000','-5.25872390',' 1',' 1 ',Null),
(1096,'1097',' 388',' -4','-1',' 16.18517846',' 12.01570000','10.85627600',' 1',' 1 ',Null),
(1097,'1098',' 389',' 60',' 5','  0.67806669',' -0.06743270',' 0.67839017',' 0',' 0 ','23 '),
(1098,'1099',' 390',' 59',' 5','  0.71272787',' -0.22377983',' 0.67864970',' 0',' 0 ','23 '),
(1099,'1100',' 391',' 58',' 5','  0.77829727',' -0.37969311',' 0.67965475',' 0',' 0 ','23 '),
(1100,'1101',' 392',' 57',' 5','  0.71487973',' -0.45909047',' 0.54613614',' 0',' 0 ','23 '),
(1101,'1102',' 393',' 56',' 4','  0.62289951',' -0.30026913',' 0.54605911',' 0',' 0 ','23 '),
(1102,'1103',' 394',' 55',' 4','  0.56141250',' -0.14605519',' 0.54449140',' 0',' 0 ','23 '),
(1103,'1104',' 395',' 54',' 4','  0.53719846','  0.00788771',' 0.54166819',' 0',' 0 ','23 '),
(1104,'1105',' 396',' 53',' 4','  0.55642232','  0.16146213',' 0.53915020',' 0',' 0 ','23 '),
(1105,'1106',' 397',' 52',' 4','  0.61690118','  0.31850609',' 0.53720343',' 0',' 0 ','23 '),
(1106,'1107',' 398',' 51',' 4','  0.55718571','  0.39601207',' 0.40380400',' 0',' 0 ','23 '),
(1107,'1108',' 399',' 50',' 3','  0.46127112','  0.23698802',' 0.40456838',' 0',' 0 ','23 '),
(1108,'1109',' 400',' 49',' 3','  0.40851931','  0.08314389',' 0.40591351',' 0',' 0 ','23 '),
(1109,'1110',' 401',' 48',' 3','  0.41306156',' -0.07024021',' 0.41016146',' 0',' 0 ','23 '),
(1110,'1111',' 402',' 47',' 3','  0.46697546',' -0.22228772',' 0.41103016',' 0',' 0 ','23 '),
(1111,'1112',' 403',' 46',' 4','  0.56044191',' -0.37865570',' 0.41070285',' 0',' 0 ','23 '),
(1112,'1113',' 404',' 45',' 5','  0.67934373',' -0.53608333',' 0.41201161',' 0',' 0 ','23 '),
(1113,'1114',' 405',' 44',' 5','  0.68098199',' -0.61505667',' 0.28057550',' 0',' 0 ','23 '),
(1114,'1115',' 406',' 43',' 4','  0.54344050',' -0.45990257',' 0.28181167',' 0',' 0 ','23 '),
(1115,'1116',' 407',' 42',' 3','  0.41467820',' -0.30433277',' 0.27788483',' 0',' 0 ','23 '),
(1116,'1117',' 408',' 41',' 2','  0.31496079',' -0.14962707',' 0.27747215',' 0',' 0 ','23 '),
(1117,'1118',' 409',' 40',' 2','  0.26856969','  0.00344003',' 0.27300982',' 0',' 0 ','23 '),
(1118,'1119',' 410',' 39',' 1','  0.15182041','  0.08184437',' 0.13680826',' 0',' 0 ','23 '),
(1119,'1120',' 411',' 38',' 1','  0.15927426',' -0.07237764',' 0.14230106',' 0',' 0 ','23 '),
(1120,'1121',' 412',' 37',' 2','  0.27255925',' -0.22631409',' 0.14463961',' 0',' 0 ','23 '),
(1121,'1122',' 413',' 36',' 3','  0.41710384',' -0.38540752',' 0.14463551',' 0',' 0 ','23 '),
(1122,'1123',' 414',' 35',' 4','  0.56323823',' -0.53702488',' 0.14867545',' 0',' 0 ','23 '),
(1123,'1124',' 415',' 34',' 5','  0.71249991',' -0.69110106',' 0.14481384',' 0',' 0 ','23 '),
(1124,'1125',' 416',' 33',' 5','  0.77485170',' -0.76735082',' 0.01417759',' 0',' 0 ','23 '),
(1125,'1126',' 417',' 32',' 5','  0.71049092',' -0.69197462','-0.12049588',' 0',' 0 ','23 '),
(1126,'1127',' 418',' 31',' 5','  0.68254598',' -0.62118283','-0.26142906',' 0',' 0 ','23 '),
(1127,'1128',' 419',' -3','-1',' 11.91663164','  4.94670020','10.84927600',' 1',' 1 ',Null),
(1128,'1129',' 420',' -2','-1',' 11.97733298',' -5.12529980','10.82627600',' 1',' 1 ',Null),
(1129,'1130',' 421',' 30',' 5','  0.68213679',' -0.54499891','-0.39568228',' 0',' 0 ','23 '),
(1130,'1131',' 422',' 29',' 5','  0.71462010',' -0.46732529','-0.52964593',' 0',' 0 ','23 '),
(1131,'1132',' 423',' 28',' 5','  0.77736639',' -0.38806549','-0.66476387',' 0',' 0 ','23 '),
(1132,'1133',' 424',' 27',' 5','  0.71809369',' -0.23171823','-0.67262819',' 0',' 0 ','23 '),
(1133,'1134',' 425',' 26',' 4','  0.62400083',' -0.31334071','-0.53076529',' 0',' 0 ','23 '),
(1134,'1135',' 426',' 25',' 4','  0.56339535',' -0.38923082','-0.39561324',' 0',' 0 ','23 '),
(1135,'1136',' 427',' 24',' 4','  0.54229948',' -0.46629914','-0.25946169',' 0',' 0 ','23 '),
(1136,'1137',' 428',' 23',' 4','  0.56027682',' -0.53817012','-0.12288645',' 0',' 0 ','23 '),
(1137,'1138',' 429',' 22',' 4','  0.62108282',' -0.61358625',' 0.01284799',' 0',' 0 ','23 '),
(1138,'1139',' 430',' 21',' 3','  0.46478655',' -0.45730905',' 0.01036915',' 0',' 0 ','23 '),
(1139,'1140',' 431',' 20',' 3','  0.41301055',' -0.38492836','-0.12446702',' 0',' 0 ','23 '),
(1140,'1141',' 432',' 19',' 3','  0.41473017',' -0.31188322','-0.26016900',' 0',' 0 ','23 '),
(1141,'1142',' 433',' 18',' 3','  0.46587269',' -0.23451246','-0.39364607',' 0',' 0 ','23 '),
(1142,'1143',' 434',' 17',' 4','  0.56582046',' -0.15755621','-0.53675926',' 0',' 0 ','23 '),
(1143,'1144',' 435',' 16',' 5','  0.67786937',' -0.07914080','-0.66784744',' 0',' 0 ','23 '),
(1144,'1145',' 436',' 15',' 5','  0.67895010','  0.07421513','-0.67118852',' 0',' 0 ','23 '),
(1145,'1146',' 437',' 14',' 4','  0.53927115',' -0.00267969','-0.53470627',' 0',' 0 ','23 '),
(1146,'1147',' 438',' 13',' 3','  0.40586439',' -0.07805093','-0.39228843',' 0',' 0 ','23 '),
(1147,'1148',' 439',' 12',' 2','  0.31164722',' -0.15660907','-0.26050531',' 0',' 0 ','23 '),
(1148,'1149',' 440',' 11',' 2','  0.26839299',' -0.22802483','-0.12433645',' 0',' 0 ','23 '),
(1149,'1150',' 441',' 10',' 2','  0.31299246',' -0.30545861',' 0.01213167',' 0',' 0 ','23 '),
(1150,'1151',' 442','  9',' 1','  0.15675312',' -0.14928967',' 0.00717760',' 0',' 0 ','23 '),
(1151,'1152',' 443','  8',' 1','  0.15695206',' -0.07596177','-0.12848903',' 0',' 0 ','23 '),
(1152,'1153',' 444','  7',' 2','  0.26725677','  0.00439180','-0.26276946',' 0',' 0 ','23 '),
(1153,'1154',' 445','  6',' 3','  0.41164803','  0.07590117','-0.40144531',' 0',' 0 ','23 '),
(1154,'1155',' 446','  5',' 4','  0.55992879','  0.15248511','-0.53634630',' 0',' 0 ','23 '),
(1155,'1156',' 447','  4',' 5','  0.71480099','  0.23074597','-0.67455506',' 0',' 0 ','23 '),
(1156,'1157',' 448','  3',' 5','  0.77344706','  0.38197817','-0.67224387',' 0',' 0 ','23 '),
(1157,'1158',' 449','  2',' 5','  0.71064675','  0.46471776','-0.53951162',' 0',' 0 ','23 '),
(1158,'1159',' 450','  1',' 5','  0.67864854','  0.54600510','-0.40846051',' 0',' 0 ','23 '),
(1159,'1160',' 451',' -1','-1',' 16.29894533','-12.18930000','10.81627600',' 1',' 1 ',Null),
(1160,'1161',' 452',' -2','-1',' 11.97240041',' -5.09749950','10.83396500',' 1',' 1 ',Null),
(1161,'1162',' 453',' 37',' 3','  0.47755829',' -0.24153944','-0.40304833',' 0',' 0 ','16 '),
(1162,'1163',' 454',' 36',' 3','  0.42418369',' -0.31734053','-0.26838419',' 0',' 0 ','16 '),
(1163,'1164',' 455',' 35',' 3','  0.42468278',' -0.39343659','-0.13572057',' 0',' 0 ','16 '),
(1164,'1165',' 456',' 34',' 2','  0.32410199',' -0.31662140','-0.00065527',' 0',' 0 ','16 '),
(1165,'1166',' 457',' 33',' 2','  0.28664406',' -0.24384860','-0.13343883',' 0',' 0 ','16 '),
(1166,'1167',' 458',' 32',' 2','  0.32330942',' -0.16614877','-0.26828619',' 0',' 0 ','16 '),
(1167,'1168',' 459',' 31',' 3','  0.41862336',' -0.08863769','-0.40297895',' 0',' 0 ','16 '),
(1168,'1169',' 460',' 30',' 3','  0.40995912','  0.06822741','-0.40095748',' 0',' 0 ','16 '),
(1169,'1170',' 461',' 29',' 2','  0.27334117',' -0.00891525','-0.26838150',' 0',' 0 ','16 '),
(1170,'1171',' 462',' 28',' 1','  0.16872343',' -0.08695609','-0.13537629',' 0',' 0 ','16 '),
(1171,'1172',' 463',' 27',' 1','  0.17030554',' -0.16278481','-0.00077111',' 0',' 0 ','16 '),
(1172,'1173',' 464',' 26',' 2','  0.27966071',' -0.24173008',' 0.13145162',' 0',' 0 ','16 '),
(1173,'1174',' 465',' 25',' 2','  0.31161976',' -0.16341171',' 0.26507796',' 0',' 0 ','16 '),
(1174,'1175',' 466',' 24',' 1','  0.15895810',' -0.08667740',' 0.13256981',' 0',' 0 ','16 '),
(1175,'1176',' 467',' 23',' 0','  0.01860458',' -0.00974060','-0.00266838',' 0',' 0 ','16 '),
(1176,'1177',' 468',' 22',' 1','  0.15092715','  0.06706984','-0.13417817',' 0',' 0 ','16 '),
(1177,'1178',' 469',' 21',' 2','  0.30281369','  0.14148474','-0.26705937',' 0',' 0 ','16 '),
(1178,'1179',' 470',' 20',' 3','  0.46056685','  0.22219184','-0.40296535',' 0',' 0 ','16 '),
(1179,'1180',' 471',' 19',' 2','  0.39716381','  0.29629959','-0.26810900',' 0',' 0 ','16 '),
(1180,'1181',' 472',' 18',' 2','  0.25404303','  0.22059400','-0.13374462',' 0',' 0 ','16 '),
(1181,'1182',' 473',' 17',' 0','  0.13477413','  0.14215763',' 0.00056558',' 0',' 0 ','16 '),
(1182,'1183',' 474',' 16',' 0','  0.14274420','  0.06686301',' 0.13425752',' 0',' 0 ','16 '),
(1183,'1184',' 475',' 15',' 2','  0.26382865',' -0.00914744',' 0.26777661',' 0',' 0 ','16 '),
(1184,'1185',' 476',' 14',' 2','  0.29641397','  0.14483077',' 0.26711994',' 0',' 0 ','16 '),
(1185,'1186',' 477',' 13',' 1','  0.24840605','  0.21828579',' 0.13580782',' 0',' 0 ','16 '),
(1186,'1187',' 478',' 12',' 2','  0.29118226','  0.29859005',' 0.00013488',' 0',' 0 ','16 '),
(1187,'1188',' 479',' 11',' 2','  0.39094010','  0.37409251','-0.13117722',' 0',' 0 ','16 '),
(1188,'1189',' 480',' 10',' 3','  0.44348806','  0.45091200',' 0.00068845',' 0',' 0 ','16 '),
(1189,'1190',' 481','  9',' 2','  0.38984077','  0.37511318',' 0.13405504',' 0',' 0 ','16 '),
(1190,'1191',' 482','  8',' 2','  0.39328646','  0.29922793',' 0.26816308',' 0',' 0 ','16 '),
(1191,'1192',' 483','  7',' 3','  0.45024033','  0.22101331',' 0.40083191',' 0',' 0 ','16 '),
(1192,'1193',' 484','  6',' 3','  0.40209347','  0.06762648',' 0.40203344',' 0',' 0 ','16 '),
(1193,'1194',' 485','  5',' 3','  0.40931786',' -0.08933719',' 0.40218247',' 0',' 0 ','16 '),
(1194,'1195',' 486','  4',' 3','  0.46733378',' -0.24056534',' 0.40056863',' 0',' 0 ','16 '),
(1195,'1196',' 487','  3',' 3','  0.41978882',' -0.31998214',' 0.26718146',' 0',' 0 ','16 '),
(1196,'1197',' 488','  2',' 3','  0.42146337',' -0.39313997',' 0.13549286',' 0',' 0 ','16 '),
(1197,'1198',' 489','  1',' 3','  0.47916451',' -0.47172091',' 0.00262811',' 0',' 0 ','16 '),
(1198,'1199',' 490',' -1','-1',' 16.29090164','-12.17349900','10.82196500',' 1',' 1 ',Null),
(1199,'1200',' 491',' -3','-1',' 37.64265713',' 30.42488500','22.17987700',' 1',' 1 ',Null),
(1200,'1201',' 492',' -2','-1',' 30.07504291',' 20.35988500','22.14687700',' 1',' 1 ',Null),
(1201,'1202',' 493',' 21',' 0','  0.14919201','  0.15656843',' 0.00011264',' 0',' 2 ','8 '),
(1202,'1203',' 494',' 20',' 0','  0.14856615','  0.07868590',' 0.13483837',' 0',' 2 ','8 '),
(1203,'1204',' 495',' 19',' 1','  0.15450643',' -0.07827165',' 0.13302240',' 0',' 2 ','8 '),
(1204,'1205',' 496',' 18',' 0','  0.00878257','  0.00117158','-0.00168147',' 0',' 2 ','8 '),
(1205,'1206',' 497',' 17',' 1','  0.15692729','  0.07968559','-0.13483822',' 0',' 2 ','8 '),
(1206,'1207',' 498',' 16',' 1','  0.16477449',' -0.07670098','-0.13720194',' 0',' 2 ','8 '),
(1207,'1208',' 499',' 15',' 1','  0.16244230',' -0.15482865','-0.00303791',' 0',' 2 ','8 '),
(1208,'1209',' 500',' 14',' 0','  0.13689472','  0.14372398','-0.00844742',' 0',' 2 ','7 '),
(1209,'1210',' 501',' 13',' 0','  0.13456039','  0.06600594',' 0.12561668',' 0',' 2 ','7 '),
(1210,'1211',' 502',' 12',' 1','  0.15612099',' -0.08878432',' 0.12741150',' 0',' 2 ','7 '),
(1211,'1212',' 503',' 11',' 0','  0.02353002',' -0.01246051','-0.00808512',' 0',' 2 ','7 '),
(1212,'1213',' 504',' 10',' 1','  0.15882182','  0.06405571','-0.14391822',' 0',' 2 ','7 '),
(1213,'1214',' 505','  9',' 1','  0.17690280',' -0.09301207','-0.14114591',' 0',' 2 ','7 '),
(1214,'1215',' 506','  8',' 1','  0.17640593',' -0.16863870','-0.00626885',' 0',' 2 ','7 '),
(1215,'1216',' 507','  7',' 0','  0.13786417','  0.14524620',' 0.00846977',' 0',' 2 ','6 '),
(1216,'1217',' 508','  6',' 0','  0.14954519','  0.06555493',' 0.14226116',' 0',' 2 ','6 '),
(1217,'1218',' 509','  5',' 1','  0.16627970',' -0.08821591',' 0.14048049',' 0',' 2 ','6 '),
(1218,'1219',' 510','  4',' 0','  0.01676125',' -0.00909943',' 0.00718739',' 0',' 2 ','6 '),
(1219,'1220',' 511','  3',' 0','  0.14279316','  0.06828619','-0.12471071',' 0',' 2 ','6 '),
(1220,'1221',' 512','  2',' 1','  0.16170101',' -0.08621945','-0.12734476',' 0',' 2 ','6 '),
(1221,'1222',' 513','  1',' 1','  0.17068209',' -0.16323706',' 0.00577252',' 0',' 2 ','6 '),
(1222,'1223',' 514',' -1','-1',' 25.82209121',' 13.29588500','22.14487700',' 1',' 1 ',Null),
(1223,'1224',' 515',' -8','-1',' 13.14623900',' 12.08225600','-5.19383940',' 1',' 1 ',Null),
(1224,'1225',' 516','127',' 6','  0.86902980',' -0.31986557','-0.80056661',' 0',' 0 ','26 '),
(1225,'1226',' 517','126',' 5','  0.77415709',' -0.39476920','-0.65700336',' 0',' 0 ','26 '),
(1226,'1227',' 518','125',' 5','  0.71347515',' -0.24113870','-0.66430163',' 0',' 0 ','26 '),
(1227,'1228',' 519','124',' 5','  0.68125448',' -0.08698304','-0.67020922',' 0',' 0 ','26 '),
(1228,'1229',' 520','123',' 6','  0.82557154',' -0.16361632','-0.80318600',' 0',' 0 ','26 '),
(1229,'1230',' 521','122',' 6','  0.80968757',' -0.00968572','-0.80503651',' 0',' 0 ','26 '),
(1230,'1231',' 522','121',' 6','  0.82232027','  0.14485645','-0.80628736',' 0',' 0 ','26 '),
(1231,'1232',' 523','120',' 6','  0.86169755','  0.30090304','-0.80571657',' 0',' 0 ','26 '),
(1232,'1233',' 524','119',' 6','  0.92382788','  0.45305360','-0.80478067',' 0',' 0 ','26 '),
(1233,'1234',' 525','118',' 6','  0.86203123','  0.53756939','-0.67528057',' 0',' 0 ','26 '),
(1234,'1235',' 526','117',' 5','  0.76905040','  0.37548267','-0.67079533',' 0',' 0 ','26 '),
(1235,'1236',' 527','116',' 5','  0.71056666','  0.22018960','-0.67349956',' 0',' 0 ','26 '),
(1236,'1237',' 528','115',' 5','  0.67600754','  0.06404917','-0.66916322',' 0',' 0 ','26 '),
(1237,'1238',' 529','114',' 4','  0.53889869',' -0.01023479','-0.53413884',' 0',' 0 ','26 '),
(1238,'1239',' 530','113',' 4','  0.56420184',' -0.16373226','-0.53313937',' 0',' 0 ','26 '),
(1239,'1240',' 531','112',' 4','  0.61911911',' -0.31825939','-0.52205486',' 0',' 0 ','26 '),
(1240,'1241',' 532','111',' 5','  0.71207290',' -0.47204036','-0.52197698',' 0',' 0 ','26 '),
(1241,'1242',' 533','110',' 5','  0.68339769',' -0.55179744','-0.38832242',' 0',' 0 ','26 '),
(1242,'1243',' 534','109',' 4','  0.56074066',' -0.39277269','-0.38828935',' 0',' 0 ','26 '),
(1243,'1244',' 535','108',' 3','  0.46435740',' -0.23888837','-0.38916709',' 0',' 0 ','26 '),
(1244,'1245',' 536','107',' 3','  0.41289730',' -0.08510859','-0.39792159',' 0',' 0 ','26 '),
(1245,'1246',' 537','106',' 2','  0.31127959',' -0.15945435','-0.25828707',' 0',' 0 ','26 '),
(1246,'1247',' 538','105',' 3','  0.41439172',' -0.31599601','-0.25458529',' 0',' 0 ','26 '),
(1247,'1248',' 539','104',' 4','  0.54286815',' -0.47064171','-0.25271415',' 0',' 0 ','26 '),
(1248,'1249',' 540','103',' 5','  0.68144610',' -0.62443801','-0.25067493',' 0',' 0 ','26 '),
(1249,'1250',' 541','102',' 5','  0.71680781',' -0.69955856','-0.11370955',' 0',' 0 ','26 '),
(1250,'1251',' 542','101',' 4','  0.56793861',' -0.54711327','-0.11810604',' 0',' 0 ','26 '),
(1251,'1252',' 543','100',' 3','  0.41485598',' -0.38865236','-0.11888436',' 0',' 0 ','26 '),
(1252,'1253',' 544',' 99',' 2','  0.27278182',' -0.23350990','-0.12340907',' 0',' 0 ','26 '),
(1253,'1254',' 545',' 98',' 1','  0.15917923',' -0.15166369',' 0.00937133',' 0',' 0 ','26 '),
(1254,'1255',' 546',' 97',' 1','  0.15701746',' -0.07997318','-0.12596553',' 0',' 0 ','26 '),
(1255,'1256',' 547',' 96',' 2','  0.26990460',' -0.00672430','-0.26506276',' 0',' 0 ','26 '),
(1256,'1257',' 548',' 95',' 3','  0.41228561','  0.06868359','-0.40324156',' 0',' 0 ','26 '),
(1257,'1258',' 549',' 94',' 4','  0.55744425','  0.14370311','-0.53606359',' 0',' 0 ','26 '),
(1258,'1259',' 550',' 93',' 4','  0.61319518','  0.29810273','-0.53545928',' 0',' 0 ','26 '),
(1259,'1260',' 551',' 92',' 5','  0.70895159','  0.45897888','-0.54208753',' 0',' 0 ','26 '),
(1260,'1261',' 552',' 91',' 6','  0.82088142','  0.61870427','-0.54343734',' 0',' 0 ','26 '),
(1261,'1262',' 553',' -7','-1','  7.22770961','  5.00625560','-5.21583940',' 1',' 1 ',Null),
(1262,'1263',' 554',' -6','-1','  7.29461048',' -5.05774440','-5.24483940',' 1',' 1 ',Null),
(1263,'1264',' 555',' 90',' 6','  0.80388344','  0.69652196','-0.40952853',' 0',' 0 ','26 '),
(1264,'1265',' 556',' 89',' 5','  0.67438011','  0.53911651','-0.41038992',' 0',' 0 ','26 '),
(1265,'1266',' 557',' 88',' 4','  0.55927096','  0.38420586','-0.40884780',' 0',' 0 ','26 '),
(1266,'1267',' 558',' 87',' 3','  0.46033384','  0.22264148','-0.40246446',' 0',' 0 ','26 '),
(1267,'1268',' 559',' 86',' 2','  0.30541575','  0.14729532','-0.26704304',' 0',' 0 ','26 '),
(1268,'1269',' 560',' 85',' 0','  0.14975848','  0.07108015','-0.13109387',' 0',' 0 ','26 '),
(1269,'1270',' 561',' 84',' 0','  0.00831982',' -0.00048388',' 0.00700575',' 0',' 0 ','26 '),
(1270,'1271',' 562',' 83',' 1','  0.16092727',' -0.06823916',' 0.14649190',' 0',' 0 ','26 '),
(1271,'1272',' 563',' 82',' 1','  0.15450711','  0.08387026',' 0.13874900',' 0',' 0 ','26 '),
(1272,'1273',' 564',' 81',' 0','  0.14787592','  0.15528641',' 0.00151270',' 0',' 0 ','26 '),
(1273,'1274',' 565',' 80',' 2','  0.26244586','  0.22984981','-0.13485608',' 0',' 0 ','26 '),
(1274,'1275',' 566',' 79',' 3','  0.40870552','  0.30693499','-0.27363615',' 0',' 0 ','26 '),
(1275,'1276',' 567',' 78',' 4','  0.53343801','  0.46182920','-0.27496992',' 0',' 0 ','26 '),
(1276,'1277',' 568',' 77',' 3','  0.40563846','  0.38660945','-0.13965895',' 0',' 0 ','26 '),
(1277,'1278',' 569',' 76',' 2','  0.30242984','  0.30978268','-0.00279302',' 0',' 0 ','26 '),
(1278,'1279',' 570',' 75',' 2','  0.26514597','  0.23789841',' 0.13558566',' 0',' 0 ','26 '),
(1279,'1280',' 571',' 74',' 3','  0.40675729','  0.39381689',' 0.13160937',' 0',' 0 ','26 '),
(1280,'1281',' 572',' 73',' 3','  0.45787250','  0.46521744','-0.00486276',' 0',' 0 ','26 '),
(1281,'1282',' 573',' 72',' 4','  0.55308926','  0.54112287','-0.14075540',' 0',' 0 ','26 '),
(1282,'1283',' 574',' 71',' 5','  0.67301072','  0.61827573','-0.27806010',' 0',' 0 ','26 '),
(1283,'1284',' 575',' 70',' 6','  0.81551933','  0.77248403','-0.27798273',' 0',' 0 ','26 '),
(1284,'1285',' 576',' 69',' 5','  0.70517451','  0.69681793','-0.14395260',' 0',' 0 ','26 '),
(1285,'1286',' 577',' 68',' 4','  0.61228974','  0.61958589','-0.00880432',' 0',' 0 ','26 '),
(1286,'1287',' 578',' 67',' 5','  0.70568698','  0.70226372',' 0.12781579',' 0',' 0 ','26 '),
(1287,'1288',' 579',' 66',' 5','  0.76660210','  0.77394632','-0.00765206',' 0',' 0 ','26 '),
(1288,'1289',' 580',' 65',' 6','  0.85606657','  0.85068211','-0.14315397',' 0',' 0 ','26 '),
(1289,'1290',' 581',' 64',' 6','  0.92106678','  0.92839547','-0.00985339',' 0',' 0 ','26 '),
(1290,'1291',' 582',' 63',' 6','  0.85684897','  0.85552810',' 0.12668628',' 0',' 0 ','26 '),
(1291,'1292',' 583',' 62',' 6','  0.81669096','  0.78186919',' 0.26378374',' 0',' 0 ','26 '),
(1292,'1293',' 584',' 61',' 6','  0.80196993','  0.70628128',' 0.39788665',' 0',' 0 ','26 '),
(1293,'1294',' 585',' -5','-1',' 13.22018885','-12.12374400','-5.24983940',' 1',' 1 ',Null),
(1294,'1295',' 586',' -4','-1',' 16.19226462',' 12.02625600','10.85516100',' 1',' 1 ',Null),
(1295,'1296',' 587',' 60',' 6','  0.81872284','  0.63182270',' 0.53404844',' 0',' 0 ','26 '),
(1296,'1297',' 588',' 59',' 5','  0.67363449','  0.55266445',' 0.40008817',' 0',' 0 ','26 '),
(1297,'1298',' 589',' 58',' 5','  0.67146413','  0.62655241',' 0.26440064',' 0',' 0 ','26 '),
(1298,'1299',' 590',' 57',' 4','  0.55439745','  0.54767982',' 0.12895903',' 0',' 0 ','26 '),
(1299,'1300',' 591',' 56',' 4','  0.53172818','  0.47117773',' 0.26462801',' 0',' 0 ','26 '),
(1300,'1301',' 592',' 55',' 3','  0.40694432','  0.31698156',' 0.26864362',' 0',' 0 ','26 '),
(1301,'1302',' 593',' 54',' 2','  0.31055645','  0.16357654',' 0.27292235',' 0',' 0 ','26 '),
(1302,'1303',' 594',' 53',' 2','  0.27305201','  0.01179824',' 0.27748715',' 0',' 0 ','26 '),
(1303,'1304',' 595',' 52',' 2','  0.31781533',' -0.14416350',' 0.28379580',' 0',' 0 ','26 '),
(1304,'1305',' 596',' 51',' 3','  0.41734479',' -0.06540406',' 0.41540837',' 0',' 0 ','26 '),
(1305,'1306',' 597',' 50',' 3','  0.41488403','  0.09055178',' 0.41094403',' 0',' 0 ','26 '),
(1306,'1307',' 598',' 49',' 3','  0.46288300','  0.24226211',' 0.40336750',' 0',' 0 ','26 '),
(1307,'1308',' 599',' 48',' 4','  0.55770632','  0.39869062',' 0.40191091',' 0',' 0 ','26 '),
(1308,'1309',' 600',' 47',' 5','  0.70854748','  0.47760142',' 0.53455277',' 0',' 0 ','26 '),
(1309,'1310',' 601',' 46',' 4','  0.61766796','  0.32338569',' 0.53521663',' 0',' 0 ','26 '),
(1310,'1311',' 602',' 45',' 4','  0.56201219','  0.16953712',' 0.54259839',' 0',' 0 ','26 '),
(1311,'1312',' 603',' 44',' 4','  0.54494510','  0.01602561',' 0.54934738',' 0',' 0 ','26 '),
(1312,'1313',' 604',' 43',' 4','  0.56752046',' -0.14059353',' 0.55234356',' 0',' 0 ','26 '),
(1313,'1314',' 605',' 42',' 5','  0.71821504',' -0.21294502',' 0.68803649',' 0',' 0 ','26 '),
(1314,'1315',' 606',' 41',' 5','  0.68259114',' -0.06074475',' 0.68364699',' 0',' 0 ','26 '),
(1315,'1316',' 607',' 40',' 5','  0.68323227','  0.09276200',' 0.68235376',' 0',' 0 ','26 '),
(1316,'1317',' 608',' 39',' 5','  0.71369188','  0.24662017',' 0.67689016',' 0',' 0 ','26 '),
(1317,'1318',' 609',' 38',' 5','  0.77295636','  0.40107038',' 0.66968921',' 0',' 0 ','26 '),
(1318,'1319',' 610',' 37',' 6','  0.85949038','  0.55459850',' 0.66729822',' 0',' 0 ','26 '),
(1319,'1320',' 611',' 36',' 6','  0.92477813','  0.47753902',' 0.80085021',' 0',' 0 ','26 '),
(1320,'1321',' 612',' 35',' 6','  0.86755531','  0.32595371',' 0.81144035',' 0',' 0 ','26 '),
(1321,'1322',' 613',' 34',' 6','  0.82606091','  0.17286115',' 0.81379834',' 0',' 0 ','26 '),
(1322,'1323',' 614',' 33',' 6','  0.81260428','  0.01767656',' 0.81700972',' 0',' 0 ','26 '),
(1323,'1324',' 615',' 32',' 6','  0.82956641',' -0.13642952',' 0.82146562',' 0',' 0 ','26 '),
(1324,'1325',' 616',' 31',' 6','  0.86901825',' -0.29055085',' 0.82079959',' 0',' 0 ','26 '),
(1325,'1326',' 617',' -3','-1',' 11.91552690','  4.95525560','10.84416100',' 1',' 1 ',Null),
(1326,'1327',' 618',' -2','-1',' 11.96647893',' -5.11074440','10.82116100',' 1',' 1 ',Null),
(1327,'1328',' 619',' 30',' 6','  0.93358003',' -0.44641947',' 0.82030275',' 0',' 0 ','26 '),
(1328,'1329',' 620',' 29',' 6','  0.86760724',' -0.52587819',' 0.68880461',' 0',' 0 ','26 '),
(1329,'1330',' 621',' 28',' 5','  0.77864452',' -0.36658990',' 0.68739662',' 0',' 0 ','26 '),
(1330,'1331',' 622',' 27',' 4','  0.62620425',' -0.29484847',' 0.55287981',' 0',' 0 ','26 '),
(1331,'1332',' 623',' 26',' 5','  0.71651527',' -0.45071514',' 0.55536732',' 0',' 0 ','26 '),
(1332,'1333',' 624',' 25',' 6','  0.82938020',' -0.60996701',' 0.55825690',' 0',' 0 ','26 '),
(1333,'1334',' 625',' 24',' 6','  0.81459111',' -0.69035162',' 0.42476201',' 0',' 0 ','26 '),
(1334,'1335',' 626',' 23',' 5','  0.68244576',' -0.53277380',' 0.42148454',' 0',' 0 ','26 '),
(1335,'1336',' 627',' 22',' 4','  0.56575350',' -0.37668786',' 0.41982851',' 0',' 0 ','26 '),
(1336,'1337',' 628',' 21',' 3','  0.47144531',' -0.21925655',' 0.41783335',' 0',' 0 ','26 '),
(1337,'1338',' 629',' 20',' 3','  0.41583907',' -0.29939972',' 0.28513249',' 0',' 0 ','26 '),
(1338,'1339',' 630',' 19',' 4','  0.54715461',' -0.46148659',' 0.28640248',' 0',' 0 ','26 '),
(1339,'1340',' 631',' 18',' 5','  0.68533696',' -0.61593360',' 0.28923655',' 0',' 0 ','26 '),
(1340,'1341',' 632',' 17',' 6','  0.83003063',' -0.77102673',' 0.29246349',' 0',' 0 ','26 '),
(1341,'1342',' 633',' 16',' 6','  0.86959240',' -0.84848548',' 0.15803555',' 0',' 0 ','26 '),
(1342,'1343',' 634',' 15',' 5','  0.71770907',' -0.69390816',' 0.15684111',' 0',' 0 ','26 '),
(1343,'1344',' 635',' 14',' 4','  0.56570067',' -0.53825401',' 0.15358474',' 0',' 0 ','26 '),
(1344,'1345',' 636',' 13',' 3','  0.41730716',' -0.38288955',' 0.15207771',' 0',' 0 ','26 '),
(1345,'1346',' 637',' 12',' 2','  0.27208949',' -0.22420126',' 0.14721091',' 0',' 0 ','26 '),
(1346,'1347',' 638',' 11',' 2','  0.31458822',' -0.30697955',' 0.01476835',' 0',' 0 ','26 '),
(1347,'1348',' 639',' 10',' 3','  0.47102075',' -0.46340879',' 0.01719409',' 0',' 0 ','26 '),
(1348,'1349',' 640','  9',' 4','  0.62661411',' -0.61902533',' 0.01812105',' 0',' 0 ','26 '),
(1349,'1350',' 641','  8',' 5','  0.77825666',' -0.77058633',' 0.02340048',' 0',' 0 ','26 '),
(1350,'1351',' 642','  7',' 6','  0.93284877',' -0.92520858',' 0.02379186',' 0',' 0 ','26 '),
(1351,'1352',' 643','  6',' 6','  0.87105937',' -0.85568348','-0.11284241',' 0',' 0 ','26 '),
(1352,'1353',' 644','  5',' 6','  0.82746669',' -0.78044222','-0.24839888',' 0',' 0 ','26 '),
(1353,'1354',' 645','  4',' 6','  0.81553370',' -0.70890134','-0.38533800',' 0',' 0 ','26 '),
(1354,'1355',' 646','  3',' 6','  0.82651748',' -0.63153208','-0.51978740',' 0',' 0 ','26 '),
(1355,'1356',' 647','  2',' 6','  0.86527597',' -0.55091811','-0.65654343',' 0',' 0 ','26 '),
(1356,'1357',' 648','  1',' 6','  0.92874854',' -0.47575881','-0.78868372',' 0',' 0 ','26 '),
(1357,'1358',' 649',' -1','-1',' 16.29952587','-12.18574400','10.82116100',' 1',' 1 ',Null),
(1358,'1359',' 650',' -4','-1',' 16.19427770',' 12.02470600','10.85988100',' 1',' 1 ',Null),
(1359,'1360',' 651',' 61',' 4','  0.55875100',' -0.14261055','-0.53375633',' 0',' 0 ','21 '),
(1360,'1361',' 652',' 60',' 4','  0.53941779','  0.01395865','-0.53490848',' 0',' 0 ','21 '),
(1361,'1362',' 653',' 59',' 4','  0.56271949','  0.16982368','-0.53431091',' 0',' 0 ','21 '),
(1362,'1363',' 654',' 58',' 4','  0.62500271','  0.32414846','-0.53434747',' 0',' 0 ','21 '),
(1363,'1364',' 655',' 57',' 4','  0.56496031','  0.40170594','-0.40017137',' 0',' 0 ','21 '),
(1364,'1365',' 656',' 56',' 3','  0.46891061','  0.24463540','-0.40002425',' 0',' 0 ','21 '),
(1365,'1366',' 657',' 55',' 3','  0.41341498','  0.08987311','-0.40064332',' 0',' 0 ','21 '),
(1366,'1367',' 658',' 54',' 3','  0.41118240',' -0.06596696','-0.40010686',' 0',' 0 ','21 '),
(1367,'1368',' 659',' 53',' 3','  0.46516972',' -0.22245521','-0.39991978',' 0',' 0 ','21 '),
(1368,'1369',' 660',' 52',' 3','  0.40906914',' -0.29819831','-0.26741746',' 0',' 0 ','21 '),
(1369,'1370',' 661',' 51',' 2','  0.30914471',' -0.14339130','-0.26538254',' 0',' 0 ','21 '),
(1370,'1371',' 662',' 50',' 2','  0.27068707','  0.01039373','-0.26620103',' 0',' 0 ','21 '),
(1371,'1372',' 663',' 49',' 2','  0.31185479','  0.16520206','-0.26453670',' 0',' 0 ','21 '),
(1372,'1373',' 664',' 48',' 3','  0.41573774','  0.32352646','-0.26558055',' 0',' 0 ','21 '),
(1373,'1374',' 665',' 47',' 4','  0.54677861','  0.48095637','-0.26893299',' 0',' 0 ','21 '),
(1374,'1375',' 666',' 46',' 4','  0.56448423','  0.55533290','-0.13138253',' 0',' 0 ','21 '),
(1375,'1376',' 667',' 45',' 3','  0.41698364','  0.40008694','-0.13590026',' 0',' 0 ','21 '),
(1376,'1377',' 668',' 44',' 2','  0.27166313','  0.24205685','-0.13248196',' 0',' 0 ','21 '),
(1377,'1378',' 669',' 43',' 1','  0.16000867','  0.08729618','-0.13418712',' 0',' 0 ','21 '),
(1378,'1379',' 670',' 42',' 1','  0.15495469',' -0.06900054','-0.13031802',' 0',' 0 ','21 '),
(1379,'1380',' 671',' 41',' 2','  0.26560409',' -0.22320892','-0.12723649',' 0',' 0 ','21 '),
(1380,'1381',' 672',' 40',' 3','  0.41049281',' -0.38222668','-0.12461988',' 0',' 0 ','21 '),
(1381,'1382',' 673',' 39',' 3','  0.46187035',' -0.45441118',' 0.00867105',' 0',' 0 ','21 '),
(1382,'1383',' 674',' 38',' 2','  0.30832015',' -0.30085988',' 0.00799965',' 0',' 0 ','21 '),
(1383,'1384',' 675',' 37',' 1','  0.15361045',' -0.14616262',' 0.00601460',' 0',' 0 ','21 '),
(1384,'1385',' 676',' 36',' 0','  0.00000000','  0.00744006',' 0.00446992',' 0',' 0 ','21 '),
(1385,'1386',' 677',' 35',' 1','  0.15347397','  0.16091371',' 0.00478632',' 0',' 0 ','21 '),
(1386,'1387',' 678',' 34',' 2','  0.31064910','  0.31807813',' 0.00185238',' 0',' 0 ','21 '),
(1387,'1388',' 679',' 33',' 3','  0.46716866','  0.47460636',' 0.00298432',' 0',' 0 ','21 '),
(1388,'1389',' 680',' 32',' 4','  0.62408814','  0.63152454',' 0.00233277',' 0',' 0 ','21 '),
(1389,'1390',' 681',' 31',' 4','  0.56263353','  0.55431814',' 0.13668512',' 0',' 0 ','21 '),
(1390,'1391',' 682',' -3','-1',' 11.92099740','  4.95370590','10.85088100',' 1',' 1 ',Null),
(1391,'1392',' 683',' -2','-1',' 11.97140826',' -5.11229410','10.82588100',' 1',' 1 ',Null),
(1392,'1393',' 684',' 30',' 4','  0.53633115','  0.47169997',' 0.27301012',' 0',' 0 ','21 '),
(1393,'1394',' 685',' 29',' 4','  0.56032393','  0.39487553',' 0.40926204',' 0',' 0 ','21 '),
(1394,'1395',' 686',' 28',' 4','  0.61825257','  0.31678453',' 0.53976632',' 0',' 0 ','21 '),
(1395,'1396',' 687',' 27',' 4','  0.56139287','  0.16099124',' 0.54445510',' 0',' 0 ','21 '),
(1396,'1397',' 688',' 26',' 3','  0.46608107','  0.24032124',' 0.40819993',' 0',' 0 ','21 '),
(1397,'1398',' 689',' 25',' 3','  0.41147531','  0.31793058',' 0.27448394',' 0',' 0 ','21 '),
(1398,'1399',' 690',' 24',' 3','  0.40965709','  0.39532618',' 0.13623975',' 0',' 0 ','21 '),
(1399,'1400',' 691',' 23',' 2','  0.26948817','  0.24141213',' 0.13818957',' 0',' 0 ','21 '),
(1400,'1401',' 692',' 22',' 2','  0.31140045','  0.16146233',' 0.27511246',' 0',' 0 ','21 '),
(1401,'1402',' 693',' 21',' 3','  0.41206196','  0.08643393',' 0.40888929',' 0',' 0 ','21 '),
(1402,'1403',' 694',' 20',' 4','  0.54020693','  0.00423553',' 0.54466735',' 0',' 0 ','21 '),
(1403,'1404',' 695',' 19',' 4','  0.56221887',' -0.14753404',' 0.54490779',' 0',' 0 ','21 '),
(1404,'1405',' 696',' 18',' 3','  0.41394524',' -0.07071292',' 0.41097057',' 0',' 0 ','21 '),
(1405,'1406',' 697',' 17',' 2','  0.27233371','  0.00442158',' 0.27678690',' 0',' 0 ','21 '),
(1406,'1407',' 698',' 16',' 1','  0.15077573','  0.08099395',' 0.13608735',' 0',' 0 ','21 '),
(1407,'1408',' 699',' 15',' 1','  0.15716427',' -0.07217246',' 0.13997806',' 0',' 0 ','21 '),
(1408,'1409',' 700',' 14',' 2','  0.31466000',' -0.14912201',' 0.27741540',' 0',' 0 ','21 '),
(1409,'1410',' 701',' 13',' 3','  0.46914595',' -0.22516294',' 0.41189325',' 0',' 0 ','21 '),
(1410,'1411',' 702',' 12',' 4','  0.62274029',' -0.30350442',' 0.54402437',' 0',' 0 ','21 '),
(1411,'1412',' 703',' 11',' 4','  0.56560418',' -0.38390564',' 0.41282838',' 0',' 0 ','21 '),
(1412,'1413',' 704',' 10',' 3','  0.41438806',' -0.30305662',' 0.27889528',' 0',' 0 ','21 '),
(1413,'1414',' 705','  9',' 2','  0.27153279',' -0.22631213',' 0.14263637',' 0',' 0 ','21 '),
(1414,'1415',' 706','  8',' 3','  0.41511765',' -0.38270448',' 0.14627929',' 0',' 0 ','21 '),
(1415,'1416',' 707','  7',' 4','  0.53997567',' -0.45864826',' 0.27711511',' 0',' 0 ','21 '),
(1416,'1417',' 708','  6',' 4','  0.56204990',' -0.53788863',' 0.14054602',' 0',' 0 ','21 '),
(1417,'1418',' 709','  5',' 4','  0.61863106',' -0.61117819',' 0.00845024',' 0',' 0 ','21 '),
(1418,'1419',' 710','  4',' 4','  0.56178259',' -0.53677050','-0.13493791',' 0',' 0 ','21 '),
(1419,'1420',' 711','  3',' 4','  0.53650981',' -0.45664457','-0.26472932',' 0',' 0 ','21 '),
(1420,'1421',' 712','  2',' 4','  0.55864074',' -0.37691447','-0.40093244',' 0',' 0 ','21 '),
(1421,'1422',' 713','  1',' 4','  0.61791102',' -0.29629801','-0.53363528',' 0',' 0 ','21 '),
(1422,'1423',' 714',' -1','-1',' 16.28986970','-12.17929400','10.81388100',' 1',' 1 ',Null);

ALTER TABLE ONLY mangadatadb.cube
    ADD CONSTRAINT wavelength_fk
    FOREIGN KEY (wavelength_pk) REFERENCES mangadatadb.wavelength(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.cube
    ADD CONSTRAINT pipeline_info_fk
    FOREIGN KEY (pipeline_info_pk) REFERENCES mangadatadb.pipeline_info(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.cube
    ADD CONSTRAINT ifudesign_fk
    FOREIGN KEY (ifudesign_pk) REFERENCES mangadatadb.ifudesign(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.cube
    ADD CONSTRAINT manga_target_fk
    FOREIGN KEY (manga_target_pk) REFERENCES mangasampledb.manga_target(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.cube
    ADD CONSTRAINT cube_shape_fk
    FOREIGN KEY (cube_shape_pk) REFERENCES mangadatadb.cube_shape(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.wcs
    ADD CONSTRAINT cube_fk
    FOREIGN KEY (cube_pk) REFERENCES mangadatadb.cube(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.rssfiber
    ADD CONSTRAINT cube_fk
    FOREIGN KEY (cube_pk) REFERENCES mangadatadb.cube(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.rssfiber
    ADD CONSTRAINT fibers_fk
    FOREIGN KEY (fibers_pk) REFERENCES mangadatadb.fibers(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;
ALTER TABLE ONLY mangadatadb.spaxel
    ADD CONSTRAINT cube_fk
    FOREIGN KEY (cube_pk) REFERENCES mangadatadb.cube(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.fits_header_value
    ADD CONSTRAINT cube_fk
    FOREIGN KEY (cube_pk) REFERENCES mangadatadb.cube(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.fits_header_value
    ADD CONSTRAINT fits_header_keyword_fk
    FOREIGN KEY (fits_header_keyword_pk) REFERENCES mangadatadb.fits_header_keyword(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.pipeline_info
    ADD CONSTRAINT pipeline_name_fk
    FOREIGN KEY (pipeline_name_pk) REFERENCES mangadatadb.pipeline_name(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.pipeline_info
    ADD CONSTRAINT pipeline_stage_fk
    FOREIGN KEY (pipeline_stage_pk) REFERENCES mangadatadb.pipeline_stage(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.pipeline_info
    ADD CONSTRAINT pipeline_version_fk
    FOREIGN KEY (pipeline_version_pk) REFERENCES mangadatadb.pipeline_version(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.pipeline_info
    ADD CONSTRAINT pipeline_completion_status_fk
    FOREIGN KEY (pipeline_completion_status_pk) REFERENCES mangadatadb.pipeline_completion_status(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.ifu_to_block
    ADD CONSTRAINT ifudesign_fk
    FOREIGN KEY (ifudesign_pk) REFERENCES mangadatadb.ifudesign(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.ifu_to_block
    ADD CONSTRAINT slitblock_fk
    FOREIGN KEY (slitblock_pk) REFERENCES mangadatadb.slitblock(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.fibers
    ADD CONSTRAINT ifudesign_fk
    FOREIGN KEY (ifudesign_pk) REFERENCES mangadatadb.ifudesign(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.fibers
    ADD CONSTRAINT fiber_type_fk
    FOREIGN KEY (fiber_type_pk) REFERENCES mangadatadb.fiber_type(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.fibers
    ADD CONSTRAINT target_type_fk
    FOREIGN KEY (target_type_pk) REFERENCES mangadatadb.target_type(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.sample
    ADD CONSTRAINT cube_fk
    FOREIGN KEY (cube_pk) REFERENCES mangadatadb.cube(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.cart_to_cube
    ADD CONSTRAINT cart_fk
    FOREIGN KEY (cart_pk) REFERENCES mangadatadb.cart(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE ONLY mangadatadb.cart_to_cube
    ADD CONSTRAINT cube_fk
    FOREIGN KEY (cube_pk) REFERENCES mangadatadb.cube(pk)
    ON UPDATE CASCADE ON DELETE CASCADE;

/* Indexes */
CREATE INDEX CONCURRENTLY wavelength_pk_idx ON mangadatadb.cube using BTREE(wavelength_pk);
CREATE INDEX CONCURRENTLY pipelineinfo_pk_idx ON mangadatadb.cube using BTREE(pipeline_info_pk);
CREATE INDEX CONCURRENTLY ifudesign_pk_idx ON mangadatadb.cube using BTREE(ifudesign_pk);
CREATE INDEX CONCURRENTLY manga_target_pk_idx ON mangadatadb.cube using BTREE(manga_target_pk);
CREATE INDEX CONCURRENTLY cube_shape_pk_idx ON mangadatadb.cube using BTREE(cube_shape_pk);
CREATE INDEX CONCURRENTLY rssfib_cube_pk_idx ON mangadatadb.rssfiber using BTREE(cube_pk);
CREATE INDEX CONCURRENTLY fibers_pk_idx ON mangadatadb.rssfiber using BTREE(fibers_pk);
CREATE INDEX CONCURRENTLY spaxel_cube_pk_idx ON mangadatadb.spaxel using BTREE(cube_pk);
CREATE INDEX CONCURRENTLY spaxel_x_idx ON mangadatadb.spaxel using BTREE(x);
CREATE INDEX CONCURRENTLY spaxel_y_idx ON mangadatadb.spaxel using BTREE(y);
CREATE INDEX CONCURRENTLY wcs_cube_pk_idx ON mangadatadb.wcs using BTREE(cube_pk);
CREATE INDEX CONCURRENTLY fitshead_cube_pk_idx ON mangadatadb.fits_header_value using BTREE(cube_pk);
CREATE INDEX CONCURRENTLY fits_header_keyword_pk_idx ON mangadatadb.fits_header_value using BTREE(fits_header_keyword_pk);
CREATE INDEX CONCURRENTLY pipeline_version_pk_idx ON mangadatadb.pipeline_info using BTREE(pipeline_version_pk);


/* Functions related to MaNGA Data DB */
SET search_path TO functions;

/* return the sum of cube.specres for a given index range */
CREATE OR REPLACE FUNCTION specres_sum_segment(cube mangadatadb.cube, index1 integer, index2 integer) RETURNS numeric
    LANGUAGE plpgsql STABLE
    AS $$

DECLARE result numeric;
BEGIN
	select sum(f) from unnest(cube.specres[index1:index2]) as f into result;
	return result;
END; $$;

/* return the sum of cube.specres for a given index range */
CREATE OR REPLACE FUNCTION new_specres_sum_segment(specres real[], index1 integer, index2 integer) RETURNS numeric
    LANGUAGE plpgsql STABLE
    AS $$

DECLARE result numeric;
BEGIN
	select sum(f) from unnest(specres[index1:index2]) as f into result;
	return result;
END; $$;

/* return a wavelength subset based on input range */
CREATE OR REPLACE FUNCTION filter_wavelength(wavelength real[], minwave integer, maxwave integer) RETURNS numeric[]
    LANGUAGE plpgsql STABLE
    AS $$

DECLARE result numeric[];
BEGIN
	select array_agg(f) from (select unnest(wavelength) as f) as g where (f between minwave and maxwave) into result;
	return result;
END; $$;

CREATE OR REPLACE FUNCTION rest_wavelength(nsaz double precision) returns numeric[]
    LANGUAGE plpgsql STABLE
    AS $$

DECLARE result numeric[];
BEGIN
	select array_agg(f) from (select unnest(w.wavelength)/(1+nsaz) as f from mangadatadb.wavelength as w) as g into result;
	return result;
END; $$;



